// Copyright (c) 2017 Valve Corporation
// Copyright (c) 2017 LunarG Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "pass_fixture.h"
#include "pass_utils.h"

namespace {

using namespace spvtools;

using InsertExtractElimTest = PassTest<::testing::Test>;

TEST_F(InsertExtractElimTest, Simple) {
  // Note: The SPIR-V assembly has had store/load elimination
  // performed to allow the inserts and extracts to directly
  // reference each other.
  //
  // #version 140
  //
  // in vec4 BaseColor;
  //
  // struct S_t {
  //     vec4 v0;
  //     vec4 v1;
  // };
  //
  // void main()
  // {
  //     S_t s0;
  //     s0.v1 = BaseColor;
  //     gl_FragColor = s0.v1;
  // }

  const std::string predefs =
      R"(OpCapability Shader
%1 = OpExtInstImport "GLSL.std.450"
OpMemoryModel Logical GLSL450
OpEntryPoint Fragment %main "main" %BaseColor %gl_FragColor
OpExecutionMode %main OriginUpperLeft
OpSource GLSL 140
OpName %main "main"
OpName %S_t "S_t"
OpMemberName %S_t 0 "v0"
OpMemberName %S_t 1 "v1"
OpName %s0 "s0"
OpName %BaseColor "BaseColor"
OpName %gl_FragColor "gl_FragColor"
%void = OpTypeVoid
%8 = OpTypeFunction %void
%float = OpTypeFloat 32
%v4float = OpTypeVector %float 4
%S_t = OpTypeStruct %v4float %v4float
%_ptr_Function_S_t = OpTypePointer Function %S_t
%int = OpTypeInt 32 1
%int_1 = OpConstant %int 1
%_ptr_Input_v4float = OpTypePointer Input %v4float
%BaseColor = OpVariable %_ptr_Input_v4float Input
%_ptr_Function_v4float = OpTypePointer Function %v4float
%_ptr_Output_v4float = OpTypePointer Output %v4float
%gl_FragColor = OpVariable %_ptr_Output_v4float Output
)";

  const std::string before =
      R"(%main = OpFunction %void None %8
%17 = OpLabel
%s0 = OpVariable %_ptr_Function_S_t Function
%18 = OpLoad %v4float %BaseColor
%19 = OpLoad %S_t %s0
%20 = OpCompositeInsert %S_t %18 %19 1
OpStore %s0 %20
%21 = OpCompositeExtract %v4float %20 1
OpStore %gl_FragColor %21
OpReturn
OpFunctionEnd
)";

  const std::string after =
      R"(%main = OpFunction %void None %8
%17 = OpLabel
%s0 = OpVariable %_ptr_Function_S_t Function
%18 = OpLoad %v4float %BaseColor
%19 = OpLoad %S_t %s0
%20 = OpCompositeInsert %S_t %18 %19 1
OpStore %s0 %20
OpStore %gl_FragColor %18
OpReturn
OpFunctionEnd
)";

  SinglePassRunAndCheck<opt::InsertExtractElimPass>(
      predefs + before, predefs + after, true, true);
}

TEST_F(InsertExtractElimTest, OptimizeAcrossNonConflictingInsert) {
  // Note: The SPIR-V assembly has had store/load elimination
  // performed to allow the inserts and extracts to directly
  // reference each other.
  //
  // #version 140
  //
  // in vec4 BaseColor;
  //
  // struct S_t {
  //     vec4 v0;
  //     vec4 v1;
  // };
  //
  // void main()
  // {
  //     S_t s0;
  //     s0.v1 = BaseColor;
  //     s0.v0[2] = 0.0;
  //     gl_FragColor = s0.v1;
  // }

  const std::string predefs =
      R"(OpCapability Shader
%1 = OpExtInstImport "GLSL.std.450"
OpMemoryModel Logical GLSL450
OpEntryPoint Fragment %main "main" %BaseColor %gl_FragColor
OpExecutionMode %main OriginUpperLeft
OpSource GLSL 140
OpName %main "main"
OpName %S_t "S_t"
OpMemberName %S_t 0 "v0"
OpMemberName %S_t 1 "v1"
OpName %s0 "s0"
OpName %BaseColor "BaseColor"
OpName %gl_FragColor "gl_FragColor"
%void = OpTypeVoid
%8 = OpTypeFunction %void
%float = OpTypeFloat 32
%v4float = OpTypeVector %float 4
%S_t = OpTypeStruct %v4float %v4float
%_ptr_Function_S_t = OpTypePointer Function %S_t
%int = OpTypeInt 32 1
%int_1 = OpConstant %int 1
%float_0 = OpConstant %float 0
%_ptr_Input_v4float = OpTypePointer Input %v4float
%BaseColor = OpVariable %_ptr_Input_v4float Input
%_ptr_Function_v4float = OpTypePointer Function %v4float
%_ptr_Output_v4float = OpTypePointer Output %v4float
%gl_FragColor = OpVariable %_ptr_Output_v4float Output
)";

  const std::string before =
      R"(%main = OpFunction %void None %8
%18 = OpLabel
%s0 = OpVariable %_ptr_Function_S_t Function
%19 = OpLoad %v4float %BaseColor
%20 = OpLoad %S_t %s0
%21 = OpCompositeInsert %S_t %19 %20 1
%22 = OpCompositeInsert %S_t %float_0 %21 0 2
OpStore %s0 %22
%23 = OpCompositeExtract %v4float %22 1
OpStore %gl_FragColor %23
OpReturn
OpFunctionEnd
)";

  const std::string after =
      R"(%main = OpFunction %void None %8
%18 = OpLabel
%s0 = OpVariable %_ptr_Function_S_t Function
%19 = OpLoad %v4float %BaseColor
%20 = OpLoad %S_t %s0
%21 = OpCompositeInsert %S_t %19 %20 1
%22 = OpCompositeInsert %S_t %float_0 %21 0 2
OpStore %s0 %22
OpStore %gl_FragColor %19
OpReturn
OpFunctionEnd
)";

  SinglePassRunAndCheck<opt::InsertExtractElimPass>(
      predefs + before, predefs + after, true, true);
}

TEST_F(InsertExtractElimTest, OptimizeOpaque) {
  // SPIR-V not representable in GLSL; not generatable from HLSL
  // for the moment.

  const std::string predefs =
      R"(OpCapability Shader
%1 = OpExtInstImport "GLSL.std.450"
OpMemoryModel Logical GLSL450
OpEntryPoint Fragment %main "main" %outColor %texCoords
OpExecutionMode %main OriginUpperLeft
OpSource GLSL 140
OpName %main "main"
OpName %S_t "S_t"
OpMemberName %S_t 0 "v0"
OpMemberName %S_t 1 "v1"
OpMemberName %S_t 2 "smp"
OpName %outColor "outColor"
OpName %sampler15 "sampler15"
OpName %s0 "s0"
OpName %texCoords "texCoords"
OpDecorate %sampler15 DescriptorSet 0
%void = OpTypeVoid
%9 = OpTypeFunction %void
%float = OpTypeFloat 32
%v2float = OpTypeVector %float 2
%v4float = OpTypeVector %float 4
%_ptr_Output_v4float = OpTypePointer Output %v4float
%outColor = OpVariable %_ptr_Output_v4float Output
%14 = OpTypeImage %float 2D 0 0 0 1 Unknown
%15 = OpTypeSampledImage %14
%S_t = OpTypeStruct %v2float %v2float %15
%_ptr_Function_S_t = OpTypePointer Function %S_t
%17 = OpTypeFunction %void %_ptr_Function_S_t
%_ptr_UniformConstant_15 = OpTypePointer UniformConstant %15
%_ptr_Function_15 = OpTypePointer Function %15
%sampler15 = OpVariable %_ptr_UniformConstant_15 UniformConstant
%int = OpTypeInt 32 1
%int_0 = OpConstant %int 0
%int_2 = OpConstant %int 2
%_ptr_Function_v2float = OpTypePointer Function %v2float
%_ptr_Input_v2float = OpTypePointer Input %v2float
%texCoords = OpVariable %_ptr_Input_v2float Input
)";

  const std::string before =
      R"(%main = OpFunction %void None %9
%25 = OpLabel
%s0 = OpVariable %_ptr_Function_S_t Function 
%26 = OpLoad %v2float %texCoords
%27 = OpLoad %S_t %s0 
%28 = OpCompositeInsert %S_t %26 %27 0
%29 = OpLoad %15 %sampler15
%30 = OpCompositeInsert %S_t %29 %28 2
OpStore %s0 %30
%31 = OpCompositeExtract %15 %30 2
%32 = OpCompositeExtract %v2float %30 0
%33 = OpImageSampleImplicitLod %v4float %31 %32
OpStore %outColor %33
OpReturn
OpFunctionEnd
)";

  const std::string after =
      R"(%main = OpFunction %void None %9
%25 = OpLabel
%s0 = OpVariable %_ptr_Function_S_t Function
%26 = OpLoad %v2float %texCoords
%27 = OpLoad %S_t %s0
%28 = OpCompositeInsert %S_t %26 %27 0
%29 = OpLoad %15 %sampler15
%30 = OpCompositeInsert %S_t %29 %28 2
OpStore %s0 %30
%33 = OpImageSampleImplicitLod %v4float %29 %26
OpStore %outColor %33
OpReturn
OpFunctionEnd
)";

  SinglePassRunAndCheck<opt::InsertExtractElimPass>(
      predefs + before, predefs + after, true, true);
}

TEST_F(InsertExtractElimTest, OptimizeNestedStruct) {
  // The following HLSL has been pre-optimized to get the SPIR-V:
  // struct S0
  // {
  //     int x;
  //     SamplerState ss;
  // };
  //
  // struct S1
  // {
  //     float b;
  //     S0 s0;
  // };
  //
  // struct S2
  // {
  //     int a1;
  //     S1 resources;
  // };
  //
  // SamplerState samp;
  // Texture2D tex;
  //
  // float4 main(float4 vpos : VPOS) : COLOR0
  // {
  //     S1 s1;
  //     S2 s2;
  //     s1.s0.ss = samp;
  //     s2.resources = s1;
  //     return tex.Sample(s2.resources.s0.ss, float2(0.5));
  // }

  const std::string predefs =
      R"(OpCapability Shader
%1 = OpExtInstImport "GLSL.std.450"
OpMemoryModel Logical GLSL450
OpEntryPoint Fragment %main "main" %_entryPointOutput
OpExecutionMode %main OriginUpperLeft
OpSource HLSL 500
OpName %main "main"
OpName %S0 "S0"
OpMemberName %S0 0 "x"
OpMemberName %S0 1 "ss"
OpName %S1 "S1"
OpMemberName %S1 0 "b"
OpMemberName %S1 1 "s0"
OpName %samp "samp"
OpName %S2 "S2"
OpMemberName %S2 0 "a1"
OpMemberName %S2 1 "resources"
OpName %tex "tex"
OpName %_entryPointOutput "@entryPointOutput"
OpDecorate %samp DescriptorSet 0
OpDecorate %tex DescriptorSet 0
OpDecorate %_entryPointOutput Location 0
%void = OpTypeVoid
%10 = OpTypeFunction %void
%float = OpTypeFloat 32
%v4float = OpTypeVector %float 4
%_ptr_Function_v4float = OpTypePointer Function %v4float
%14 = OpTypeFunction %v4float %_ptr_Function_v4float
%int = OpTypeInt 32 1
%16 = OpTypeSampler
%S0 = OpTypeStruct %int %16
%S1 = OpTypeStruct %float %S0
%_ptr_Function_S1 = OpTypePointer Function %S1
%int_1 = OpConstant %int 1
%_ptr_UniformConstant_16 = OpTypePointer UniformConstant %16
%samp = OpVariable %_ptr_UniformConstant_16 UniformConstant
%_ptr_Function_16 = OpTypePointer Function %16
%S2 = OpTypeStruct %int %S1
%_ptr_Function_S2 = OpTypePointer Function %S2
%22 = OpTypeImage %float 2D 0 0 0 1 Unknown
%_ptr_UniformConstant_22 = OpTypePointer UniformConstant %22
%tex = OpVariable %_ptr_UniformConstant_22 UniformConstant
%24 = OpTypeSampledImage %22
%v2float = OpTypeVector %float 2
%float_0_5 = OpConstant %float 0.5
%27 = OpConstantComposite %v2float %float_0_5 %float_0_5
%_ptr_Input_v4float = OpTypePointer Input %v4float
%_ptr_Output_v4float = OpTypePointer Output %v4float
%_entryPointOutput = OpVariable %_ptr_Output_v4float Output
)";

  const std::string before =
      R"(%main = OpFunction %void None %10
%30 = OpLabel
%31 = OpVariable %_ptr_Function_S1 Function
%32 = OpVariable %_ptr_Function_S2 Function
%33 = OpLoad %16 %samp
%34 = OpLoad %S1 %31
%35 = OpCompositeInsert %S1 %33 %34 1 1
OpStore %31 %35
%36 = OpLoad %S2 %32
%37 = OpCompositeInsert %S2 %35 %36 1
OpStore %32 %37
%38 = OpLoad %22 %tex
%39 = OpCompositeExtract %16 %37 1 1 1
%40 = OpSampledImage %24 %38 %39
%41 = OpImageSampleImplicitLod %v4float %40 %27
OpStore %_entryPointOutput %41
OpReturn
OpFunctionEnd
)";

  const std::string after =
      R"(%main = OpFunction %void None %10
%30 = OpLabel
%31 = OpVariable %_ptr_Function_S1 Function
%32 = OpVariable %_ptr_Function_S2 Function
%33 = OpLoad %16 %samp
%34 = OpLoad %S1 %31
%35 = OpCompositeInsert %S1 %33 %34 1 1
OpStore %31 %35
%36 = OpLoad %S2 %32
%37 = OpCompositeInsert %S2 %35 %36 1
OpStore %32 %37
%38 = OpLoad %22 %tex
%40 = OpSampledImage %24 %38 %33
%41 = OpImageSampleImplicitLod %v4float %40 %27
OpStore %_entryPointOutput %41
OpReturn
OpFunctionEnd
)";

  SinglePassRunAndCheck<opt::InsertExtractElimPass>(
      predefs + before, predefs + after, true, true);
}

TEST_F(InsertExtractElimTest, ConflictingInsertPreventsOptimization) {
  // Note: The SPIR-V assembly has had store/load elimination
  // performed to allow the inserts and extracts to directly
  // reference each other.
  //
  // #version 140
  //
  // in vec4 BaseColor;
  //
  // struct S_t {
  //     vec4 v0;
  //     vec4 v1;
  // };
  //
  // void main()
  // {
  //     S_t s0;
  //     s0.v1 = BaseColor;
  //     s0.v1[2] = 0.0;
  //     gl_FragColor = s0.v1;
  // }

  const std::string assembly =
      R"(OpCapability Shader
%1 = OpExtInstImport "GLSL.std.450"
OpMemoryModel Logical GLSL450
OpEntryPoint Fragment %main "main" %BaseColor %gl_FragColor
OpExecutionMode %main OriginUpperLeft
OpSource GLSL 140
OpName %main "main"
OpName %S_t "S_t"
OpMemberName %S_t 0 "v0"
OpMemberName %S_t 1 "v1"
OpName %s0 "s0"
OpName %BaseColor "BaseColor"
OpName %gl_FragColor "gl_FragColor"
%void = OpTypeVoid
%8 = OpTypeFunction %void
%float = OpTypeFloat 32
%v4float = OpTypeVector %float 4
%S_t = OpTypeStruct %v4float %v4float
%_ptr_Function_S_t = OpTypePointer Function %S_t
%int = OpTypeInt 32 1
%int_1 = OpConstant %int 1
%float_0 = OpConstant %float 0
%_ptr_Input_v4float = OpTypePointer Input %v4float
%BaseColor = OpVariable %_ptr_Input_v4float Input
%_ptr_Function_v4float = OpTypePointer Function %v4float
%_ptr_Output_v4float = OpTypePointer Output %v4float
%gl_FragColor = OpVariable %_ptr_Output_v4float Output
%main = OpFunction %void None %8
%18 = OpLabel
%s0 = OpVariable %_ptr_Function_S_t Function
%19 = OpLoad %v4float %BaseColor
%20 = OpLoad %S_t %s0
%21 = OpCompositeInsert %S_t %19 %20 1
%22 = OpCompositeInsert %S_t %float_0 %21 1 2
OpStore %s0 %22
%23 = OpCompositeExtract %v4float %22 1
OpStore %gl_FragColor %23
OpReturn
OpFunctionEnd
)";

  SinglePassRunAndCheck<opt::InsertExtractElimPass>(assembly, assembly, true,
                                                    true);
}

TEST_F(InsertExtractElimTest, ConflictingInsertPreventsOptimization2) {
  // Note: The SPIR-V assembly has had store/load elimination
  // performed to allow the inserts and extracts to directly
  // reference each other.
  //
  // #version 140
  //
  // in vec4 BaseColor;
  //
  // struct S_t {
  //     vec4 v0;
  //     vec4 v1;
  // };
  //
  // void main()
  // {
  //     S_t s0;
  //     s0.v1[1] = 1.0;
  //     s0.v1 = Baseline;
  //     gl_FragColor = vec4(s0.v1[1], 0.0, 0.0, 0.0);
  // }

  const std::string assembly =
      R"(OpCapability Shader
%1 = OpExtInstImport "GLSL.std.450"
OpMemoryModel Logical GLSL450
OpEntryPoint Fragment %main "main" %BaseColor %gl_FragColor
OpExecutionMode %main OriginUpperLeft
OpSource GLSL 140
OpName %main "main"
OpName %S_t "S_t"
OpMemberName %S_t 0 "v0"
OpMemberName %S_t 1 "v1"
OpName %s0 "s0"
OpName %BaseColor "BaseColor"
OpName %gl_FragColor "gl_FragColor"
%void = OpTypeVoid
%8 = OpTypeFunction %void
%float = OpTypeFloat 32
%v4float = OpTypeVector %float 4
%S_t = OpTypeStruct %v4float %v4float
%_ptr_Function_S_t = OpTypePointer Function %S_t
%int = OpTypeInt 32 1
%int_1 = OpConstant %int 1
%float_1 = OpConstant %float 1
%uint = OpTypeInt 32 0
%uint_1 = OpConstant %uint 1
%_ptr_Function_float = OpTypePointer Function %float
%_ptr_Input_v4float = OpTypePointer Input %v4float
%BaseColor = OpVariable %_ptr_Input_v4float Input
%_ptr_Function_v4float = OpTypePointer Function %v4float
%_ptr_Output_v4float = OpTypePointer Output %v4float
%gl_FragColor = OpVariable %_ptr_Output_v4float Output
%float_0 = OpConstant %float 0
%main = OpFunction %void None %8
%22 = OpLabel
%s0 = OpVariable %_ptr_Function_S_t Function
%23 = OpLoad %S_t %s0
%24 = OpCompositeInsert %S_t %float_1 %23 1 1
%25 = OpLoad %v4float %BaseColor
%26 = OpCompositeInsert %S_t %25 %24 1
%27 = OpCompositeExtract %float %26 1 1
%28 = OpCompositeConstruct %v4float %27 %float_0 %float_0 %float_0
OpStore %gl_FragColor %28
OpReturn
OpFunctionEnd
)";

  SinglePassRunAndCheck<opt::InsertExtractElimPass>(assembly, assembly, true,
                                                    true);
}

// TODO(greg-lunarg): Add tests to verify handling of these cases:
//

}  // anonymous namespace
