/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: bitmemcmp.c 804 2009-03-25 08:44:36Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: bitmemcmp.c 804 2009-03-25 08:44:36Z takahiko $");

#include <inttypes.h>   // as a substitute for stdint.h (Solaris 9 doesn't have)
#include <string.h>
#include "bitmemcmp.h"

/**
 * 2つのメモリ領域の中身を指定したビット数分比較する.
 * @return an integer less than, equal to, or greater than 0, according as
 *         s1 is lexicographically less than, equal to, or greater than s2.
 *         0 if bits is 0.
 */
int
bitmemcmp(const void *s1, const void *s2, size_t bits)
{
    /*
     * this is the substitute function for bitncmp() of libbind.
     * libbind の bitncmp は第3引数に8の倍数を指定した場合に,
     * 指定したビット数を超える領域にアクセスし, アクセス違反をおこす.
     * ISC に報告済み (2369, RT #18054)
     * BIND 9.3.6, 9.4.3, 9.6.0 に修正が取り込まれるはずなので,
     * それ以降の libbind の bitncmp() で置き換えてよい.
     */

    static const uint8_t bitmask[] = {
        0,
        0x80, 0xc0, 0xe0, 0xf0,
        0xf8, 0xfc, 0xfe, 0xff,
    };

    size_t bytes = bits / 8;
    if (bytes > 0) {
        int cmpstat = memcmp(s1, s2, bytes);
        if (0 != cmpstat) {
            return cmpstat;
        }   // end if
    }   // end if

    size_t oddbits = bits % 8;
    if (oddbits != 0) {
        uint8_t odd1 = ((const uint8_t *) s1)[bytes];
        uint8_t odd2 = ((const uint8_t *) s2)[bytes];
        if ((odd1 & bitmask[oddbits]) != (odd2 & bitmask[oddbits])) {
            return (odd1 & bitmask[oddbits]) > (odd2 & bitmask[oddbits]) ? 1 : -1;
        }   // end if
    }   // end if

    return 0;
}   // end function : bitmemcmp
