/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: dkimdigest.c 702 2009-03-08 06:11:35Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: dkimdigest.c 702 2009-03-08 06:11:35Z takahiko $");

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
#include <assert.h>
#include <limits.h>

#include <openssl/err.h>
#include <openssl/evp.h>

#include "ptrop.h"
#include "dkimlogger.h"
#include "xbuffer.h"
#include "strtokarray.h"
#include "strpairlist.h"
#include "dkimsignature.h"
#include "dkimcanon.h"
#include "dkimheaders.h"
#include "dkimpolicy.h"
#include "dkimdigest.h"

struct DkimDigest {
    const DkimPolicy *policy;
    const EVP_MD *digest_type;
    int pubkey_type;
    EVP_MD_CTX header_digest;
    EVP_MD_CTX body_digest;
    DkimCanon *canon;
    /// ダイジェストに含めるボディの最大長. sig-l-tag itself.
    long long body_length_limit;
    /// これまでにダイジェスト計算に含めたバイト数. sig-l-tag (body length limit) のため.
    long long current_body_length;

    FILE *fp_canon_header;
    FILE *fp_canon_body;
};

#ifndef DUMPBINDATA
#define DUMPBINDATA(__fp, __subject, __buf, __len) \
    do { \
        int __i; \
        fprintf(__fp, "%s:", __subject); \
        for (__i = 0; __i < (__len); ++__i) { \
            fprintf(__fp, " %02x", (__buf)[__i]); \
        } \
        fprintf(__fp, "\n"); \
    } while(0)
#endif

static void
DkimDigest_logOpenSSLErrors(const DkimDigest *self)
{
    unsigned long errinfo;
    const char *errfilename, *errstr;
    int errline, errflags;

    while (0 != (errinfo = ERR_get_error_line_data(&errfilename, &errline, &errstr, &errflags))) {
        DkimLogSysError(self->policy, "[OpenSSL] module=%s, function=%s, reason=%s",
                        ERR_lib_error_string(errinfo), ERR_func_error_string(errinfo),
                        ERR_reason_error_string(errinfo));
        DkimLogSysError(self->policy, "[OpenSSL] file=%s, line=%d, err=%s", errfilename, errline,
                        (errflags & ERR_TXT_STRING) ? errstr : "(none)");
    }   // end while
}   // end function : DkimDigest_logOpenSSLErrors

/*
 * canonicalization 後のデータをダンプするフラグのセット
 */
dkim_stat_t
DkimDigest_openCanonDump(DkimDigest *self, const char *header_dump_filename,
                         const char *body_dump_filename)
{
    assert(NULL != self);
    assert(NULL == self->fp_canon_header);
    assert(NULL == self->fp_canon_body);

    self->fp_canon_header = fopen(header_dump_filename, "wb");
    if (NULL == self->fp_canon_header) {
        DkimLogNotice(self->policy, "failed to open header-canon-dump file: %s",
                      header_dump_filename);
        return DSTAT_WARN_CANONDUMP_OPEN_FAILURE;
    }   // end if
    self->fp_canon_body = fopen(body_dump_filename, "wb");
    if (NULL == self->fp_canon_body) {
        fclose(self->fp_canon_header);
        DkimLogNotice(self->policy, "failed to open body-canon-dump file: %s", body_dump_filename);
        return DSTAT_WARN_CANONDUMP_OPEN_FAILURE;
    }   // end if
    return DSTAT_OK;
}   // end function : DkimDigest_setCanonDump

static dkim_stat_t
DkimDigest_dumpCanonHeader(DkimDigest *self, const void *data, size_t len)
{
    if (NULL != self->fp_canon_header) {
        if (0 == fwrite(data, 1, len, self->fp_canon_header)) {
            // ファイルへのダンプはデバッグが目的なので, エラーが起きても続行する
            DkimLogNotice(self->policy, "canonicalized data dump (for header) failed");
            return DSTAT_WARN_CANONDUMP_UPDATE_FAILURE;
        }   // end if
    }   // end if
    return DSTAT_OK;
}   // end function : DkimDigest_dumpCanonHeader

static dkim_stat_t
DkimDigest_dumpCanonBody(DkimDigest *self, const void *data, size_t len)
{
    if (NULL != self->fp_canon_body) {
        if (0 == fwrite(data, 1, len, self->fp_canon_body)) {
            // ファイルへのダンプはデバッグが目的なので, エラーが起きても続行する
            DkimLogNotice(self->policy, "canonicalized data dump (for body) failed");
            return DSTAT_WARN_CANONDUMP_UPDATE_FAILURE;
        }   // end if
    }   // end if
    return DSTAT_OK;
}   // end function : DkimDigest_dumpCanonBody

static dkim_stat_t
DkimDigest_closeCanonDump(DkimDigest *self)
{
    if (NULL != self->fp_canon_header) {
        (void) fclose(self->fp_canon_header);   // エラーを検出したところでどうにもできない
        self->fp_canon_header = NULL;
    }   // end if
    if (NULL != self->fp_canon_body) {
        (void) fclose(self->fp_canon_body);
        self->fp_canon_body = NULL;
    }   // end if
    return DSTAT_OK;
}   // end function : DkimDigest_closeCanonDump

DkimDigest *
DkimDigest_newWithSignature(const DkimPolicy *policy, const DkimSignature *signature,
                            dkim_stat_t *dstat)
{
    return DkimDigest_new(policy, DkimSignature_getDigestAlg(signature),
                          DkimSignature_getPubKeyAlg(signature),
                          DkimSignature_getHeaderCanonAlg(signature),
                          DkimSignature_getBodyCanonAlg(signature),
                          DkimSignature_getBodyLengthLimit(signature), dstat);
}   // end function : DkimDigest_newWithSignature

/**
 * DkimDigest オブジェクトの構築
 * @return 空の DkimDigest オブジェクト
 */
DkimDigest *
DkimDigest_new(const DkimPolicy *policy, dkim_digestalg_t digest_alg, dkim_pubkeyalg_t pubkey_alg,
               dkim_canonalg_t header_canon_alg, dkim_canonalg_t body_canon_alg,
               long long body_length_limit, dkim_stat_t *dstat)
{
    DkimDigest *self = (DkimDigest *) malloc(sizeof(DkimDigest));
    if (NULL == self) {
        DkimLogNoResource(policy);
        SETDEREF(dstat, DSTAT_SYSERR_NORESOURCE);
        return NULL;
    }   // end if
    memset(self, 0, sizeof(DkimDigest));

    switch (digest_alg) {
    case DKIM_DIGESTALG_SHA1:
        self->digest_type = EVP_sha1();
        break;
    case DKIM_DIGESTALG_SHA256:
        self->digest_type = EVP_sha256();
        break;
    default:
        DkimLogPermFail(policy, "unsupported digest algorithm specified: digestalg=0x%x",
                        digest_alg);
        SETDEREF(dstat, DSTAT_PERMFAIL_UNSUPPORTED_DIGESTALG);
        goto cleanup;
    }   // end switch

    switch (pubkey_alg) {
    case DKIM_PUBKEYALG_RSA:
        self->pubkey_type = EVP_PKEY_RSA;
        break;
    default:
        DkimLogPermFail(policy, "unsupported public key algorithm specified: pubkeyalg=0x%x",
                        pubkey_alg);
        SETDEREF(dstat, DSTAT_PERMFAIL_UNSUPPORTED_PUBKEYALG);
        goto cleanup;
    }   // end switch

    self->canon = DkimCanon_new(policy, header_canon_alg, body_canon_alg, dstat);
    if (NULL == self->canon) {
        goto cleanup;
    }   // end if
    if (0 == EVP_DigestInit(&(self->header_digest), self->digest_type)) {
        DkimLogSysError(policy, "Digest Initilization (of header) failed");
        DkimDigest_logOpenSSLErrors(self);
        SETDEREF(dstat, DSTAT_SYSERR_NORESOURCE);
        goto cleanup;
    }   // end if
    if (0 == EVP_DigestInit(&(self->body_digest), self->digest_type)) {
        DkimLogSysError(policy, "Digest Initilization (of body) failed");
        DkimDigest_logOpenSSLErrors(self);
        SETDEREF(dstat, DSTAT_SYSERR_NORESOURCE);
        goto cleanup;
    }   // end if

    self->policy = policy;
    self->body_length_limit = body_length_limit;

    SETDEREF(dstat, DSTAT_OK);
    return self;

  cleanup:
    DkimDigest_free(self);
    return NULL;
}   // end function : DkimDigest_new

/**
 * DkimDigest オブジェクトの解放
 * @param self 解放する DkimDigest オブジェクト
 */
void
DkimDigest_free(DkimDigest *self)
{
    assert(NULL != self);
    (void) DkimDigest_closeCanonDump(self);
    if (NULL != self->canon) {
        DkimCanon_free(self->canon);
    }   // end if
    (void) EVP_MD_CTX_cleanup(&(self->header_digest));
    (void) EVP_MD_CTX_cleanup(&(self->body_digest));

    // self->digest_type は clean-up 不要
    free(self);
}   // end function : DkimDigest_free

/**
 * メッセージ本文のダイジェスト値を更新する.
 * @param self DkimDigest オブジェクト
 * @param buf canonicalization 済みのメッセージ本文 (の一部)
 * @param len buf が保持するデータの長さ
 */
static dkim_stat_t
DkimDigest_updateBodyChunk(DkimDigest *self, const unsigned char *buf, size_t len)
{
    /*
     * [RFC4871] 3.4.5.
     * The body length count allows the signer of a message to permit data
     * to be appended to the end of the body of a signed message.  The body
     * length count MUST be calculated following the canonicalization
     * algorithm; for example, any whitespace ignored by a canonicalization
     * algorithm is not included as part of the body length count.  Signers
     * of MIME messages that include a body length count SHOULD be sure that
     * the length extends to the closing MIME boundary string.
     *
     * - body length limit は canonicalization 後のデータに対して適用する.
     */
    long long srclen = len;
    if (0 <= self->body_length_limit) {
        if (self->body_length_limit < self->current_body_length) {
            // 既に BodyLengthLimit を超過したサイズをダイジェストに含めてしまっている場合
            DkimLogImplError(self->policy, "body length limit over detected");
            return DSTAT_SYSERR_IMPLERROR;
        }   // end if
        if (self->body_length_limit < self->current_body_length + srclen) {
            // BodyLengthLimit ちょうどに収めるために，サイズを丸める必要がある場合
            srclen = self->body_length_limit - self->current_body_length;
        }   // end if
    }   // end if

    if (0 < srclen) {
        if (0 == EVP_DigestUpdate(&self->body_digest, buf, srclen)) {
            DkimLogSysError(self->policy, "Digest update (of body) failed");
            DkimDigest_logOpenSSLErrors(self);
            return DSTAT_SYSERR_DIGEST_UPDATE_FAILURE;
        }   // end if
        (void) DkimDigest_dumpCanonBody(self, buf, srclen);
        self->current_body_length += srclen;
    }   // end if
    return DSTAT_OK;
}   // end function : DkimDigest_updateBodyChunk

/**
 * メッセージ本文のダイジェスト値を更新する.
 * @param self DkimDigest オブジェクト
 * @param buf メッセージ本文 (の一部)
 * @param len buf が保持するデータの長さ
 */
dkim_stat_t
DkimDigest_updateBody(DkimDigest *self, const unsigned char *buf, size_t len)
{
    assert(NULL != self);
    assert(NULL != buf);

    if (0 <= self->body_length_limit && self->body_length_limit <= self->current_body_length) {
        // body length limit を越えたバイトを既に書き込んでいる場合は
        // canonicalization の前に処理をスキップ
        return DSTAT_OK;
    }   // end if

    const unsigned char *canonbuf;
    size_t canonsize;
    // canonicalize してから
    dkim_stat_t canon_stat = DkimCanon_body(self->canon, buf, len, &canonbuf, &canonsize);
    if (DSTAT_OK != canon_stat) {
        return canon_stat;
    }   // end if
    // ダイジェスト値を更新する.
    return DkimDigest_updateBodyChunk(self, canonbuf, canonsize);
}   // end function : DkimDigest_updateBody

/**
 * メッセージヘッダのダイジェスト値を更新する.
 * @param self DkimDigest オブジェクト
 * @param headerf メッセージヘッダフィールド名
 * @param headerv メッセージヘッダフィールド値
 * @param crlf canonicalization 後に CRLF を付加する必要がある場合
 *             (=headerv の末尾に CRLF が含まれない場合) は true.
 */
static dkim_stat_t
DkimDigest_updateHeader(DkimDigest *self, const char *headerf, const char *headerv, bool crlf)
{
    const unsigned char *canonbuf;
    size_t canonsize;
    dkim_stat_t canon_stat =
        DkimCanon_header(self->canon, headerf, headerv, crlf, &canonbuf, &canonsize);
    if (DSTAT_OK != canon_stat) {
        return canon_stat;
    }   // end if
    (void) DkimDigest_dumpCanonHeader(self, canonbuf, canonsize);
    if (0 == EVP_DigestUpdate(&self->header_digest, canonbuf, canonsize)) {
        DkimLogSysError(self->policy, "Digest update (of header) failed");
        DkimDigest_logOpenSSLErrors(self);
        return DSTAT_SYSERR_DIGEST_UPDATE_FAILURE;
    }   // end if

    return DSTAT_OK;
}   // end function : DkimDigest_updateHeader

/**
 * メッセージヘッダ部のダイジェスト値を計算する.
 * @param self DkimDigest オブジェクト
 * @param headers メールの全ヘッダを保持する DkimHeaders オブジェクト
 * @param selecthdr sig-h-tag で指定されている, ダイジェスト値計算対象ヘッダのリスト
 */
static dkim_stat_t
DkimDigest_digestHeaders(DkimDigest *self, const MailHeaders *headers, const StrArray *selecthdr)
{
    size_t n;
    dkim_stat_t final_stat;

    StrPairList *kvl = StrPairList_new();
    if (NULL == kvl) {
        DkimLogNoResource(self->policy);
        return DSTAT_SYSERR_NORESOURCE;
    }   // end if

    // MailHeaders から StrPairList へ shallow copy
    StrPairListItem *cur = StrPairList_tail(kvl);
    size_t headernum = MailHeaders_getCount(headers);
    for (n = 0; n < headernum; ++n) {
        const char *key, *val;
        MailHeaders_get(headers, n, &key, &val);
        cur = StrPairList_insertShallowly(kvl, cur, key, val);
        if (NULL == cur) {
            DkimLogNoResource(self->policy);
            final_stat = DSTAT_SYSERR_NORESOURCE;
            goto finally;
        }   // end if
    }   // end if

    // selecthdr の示す通りにヘッダを選択し, ダイジェスト値を更新する.
    size_t selecthdrnum = StrArray_getCount(selecthdr);
    for (n = 0; n < selecthdrnum; ++n) {
        const char *headerf = StrArray_get(selecthdr, n);
        // ヘッダは元のメールにおいて下にあるものを優先して選択する.
        cur = StrPairList_rfindIgnoreCaseByKey(kvl, headerf, NULL);
        if (NULL != cur) {
            dkim_stat_t update_stat =
                DkimDigest_updateHeader(self, StrPairListItem_key(cur), StrPairListItem_value(cur),
                                        true);
            if (DSTAT_OK != update_stat) {
                final_stat = update_stat;
                goto finally;
            }   // end if
            StrPairList_deleteShallowly(kvl, cur);
        } else {
            // "h"= タグで指定されているヘッダが見つからない場合は，
            // そのヘッダを null string として扱う
            // -> つまり何もしなければよい
        }   // end if
    }   // end if
    final_stat = DSTAT_OK;

  finally:
    StrPairList_freeShallowly(kvl);
    return final_stat;
}   // end function : DkimDigest_digestHeaders

/**
 * DKIM-Signature ヘッダ自身をダイジェスト値に含めて更新する.
 * @param self DkimDigest オブジェクト
 * @param signature DkimSignature オブジェクト
 */
static dkim_stat_t
DkimDigest_updateSignHeader(DkimDigest *self, const DkimSignature *signature)
{
    const unsigned char *canonbuf;
    size_t canonsize;
    const char *rawheaderf = DkimSignature_getRawHeaderName(signature);
    const char *rawheaderv = DkimSignature_getRawHeaderValue(signature);
    const char *b_tag_value_head;
    const char *b_tag_value_tail;
    DkimSignature_getReferenceToBodyHashOfRawHeaderValue(signature, &b_tag_value_head,
                                                         &b_tag_value_tail);
    dkim_stat_t canon_stat =
        DkimCanon_signheader(self->canon, rawheaderf, rawheaderv, b_tag_value_head,
                             b_tag_value_tail, &canonbuf, &canonsize);
    if (DSTAT_OK != canon_stat) {
        return canon_stat;
    }   // end if
    (void) DkimDigest_dumpCanonHeader(self, canonbuf, canonsize);
    if (0 == EVP_DigestUpdate(&self->header_digest, canonbuf, canonsize)) {
        DkimLogSysError(self->policy, "Digest update (of signature header) failed");
        DkimDigest_logOpenSSLErrors(self);
        return DSTAT_SYSERR_DIGEST_UPDATE_FAILURE;
    }   // end if

    return DSTAT_OK;
}   // end function : DkimDigest_updateSignHeaders

/**
 * メッセージのダイジェスト値が署名に付いているダイジェスト値と一致するか検証する.
 * @param headers メールの全ヘッダを保持する DkimHeaders オブジェクト
 * @param signature 検証する DKIM-Signature ヘッダから構築した DkimSignature オブジェクト
 * @param pkey signature 公開鍵
 * @return DSTAT_INFO_DIGEST_MATCH: 署名が一致した.
 *         DSTAT_PERMFAIL_BODYDIGEST_MISMATCH: 本文のダイジェスト値が一致しなかった.
 *         DSTAT_PERMFAIL_HEADERDIGEST_MISMATCH: ヘッダ部のダイジェスト値が一致しなかった.
 *         その他: 検証エラー
 */
dkim_stat_t
DkimDigest_verifyMessage(DkimDigest *self, const MailHeaders *headers,
                         const DkimSignature *signature, EVP_PKEY *pkey)
{
    assert(NULL != self);
    assert(NULL != headers);
    assert(NULL != signature);
    assert(NULL != pkey);

    const unsigned char *canonbuf;
    const unsigned char *signbuf;
    size_t canonsize, signlen;
    unsigned char md[EVP_MD_size(self->digest_type)];   // 安全にいくなら EVP_MAX_MD_SIZE
    unsigned int mdlen;

    // DKIM-Signature ヘッダで指定されたアルゴリズムに相応しい公開鍵のタイプか確認
    if (pkey->type != self->pubkey_type) {
        DkimLogPermFail(self->policy, "Public key type mismatch: signature=0x%x, pubkey=0x%x",
                        pkey->type, self->pubkey_type);
        return DSTAT_PERMFAIL_PUBKEY_TYPE_MISMATCH;
    }   // end if

    // body-hash の計算 & 検証
    // canonicalization の残りをはき出す
    dkim_stat_t ret = DkimCanon_bodyfin(self->canon, &canonbuf, &canonsize);
    if (DSTAT_OK != ret) {
        return ret;
    }   // end if
    // 最後の body-chunk をダイジェストに含める
    ret = DkimDigest_updateBodyChunk(self, canonbuf, canonsize);
    if (DSTAT_OK != ret) {
        return ret;
    }   // end if
    if (0 == EVP_DigestFinal(&self->body_digest, md, &mdlen)) {
        DkimLogSysError(self->policy, "Digest finish (of body) failed");
        DkimDigest_logOpenSSLErrors(self);
        return DSTAT_SYSERR_DIGEST_UPDATE_FAILURE;
    }   // end if

    const XBuffer *bodyhash = DkimSignature_getBodyHash(signature);
    if (!XBuffer_compareToBytes(bodyhash, md, mdlen)) {
        DkimLogPermFail(self->policy, "Digest of message body mismatch");
        return DSTAT_PERMFAIL_BODYDIGEST_MISMATCH;
    }   // end if

    // body hash が一致していることを確認してからヘッダの検証
    // sig-h-tag によって指定されたヘッダをダイジェストに含める.
    ret = DkimDigest_digestHeaders(self, headers, DkimSignature_getSelectHeader(signature));
    if (DSTAT_OK != ret) {
        return ret;
    }   // end if
    // DKIM-Signature ヘッダをダイジェストに追加
    ret = DkimDigest_updateSignHeader(self, signature);
    if (DSTAT_OK != ret) {
        return ret;
    }   // end if
    DkimDigest_closeCanonDump(self);

    const XBuffer *headerhash = DkimSignature_getHeaderHash(signature);
    signbuf = (const unsigned char *) XBuffer_getBytes(headerhash);
    signlen = XBuffer_getSize(headerhash);
    int vret = EVP_VerifyFinal(&self->header_digest, signbuf, signlen, pkey);
    switch (vret) {
    case 1:    // 署名一致
        return DSTAT_INFO_DIGEST_MATCH;
    case 0:    // 署名不一致
        DkimLogPermFail(self->policy, "Digest of message header mismatch");
        return DSTAT_PERMFAIL_HEADERDIGEST_MISMATCH;
    case -1:   // エラー
        DkimLogSysError(self->policy, "Digest verification error");
        DkimDigest_logOpenSSLErrors(self);
        return DSTAT_SYSERR_DIGEST_VERIFICATION_FAILURE;
    default:
        DkimLogImplError(self->policy, "EVP_VerifyFinal returns unexpected value: ret=0x%x", vret);
        DkimDigest_logOpenSSLErrors(self);
        return DSTAT_SYSERR_IMPLERROR;
    }   // end switch
}   // end function : DkimDigest_verifyMessage

/**
 * メッセージに DKIM 署名を生成する.
 * @param headers メールの全ヘッダを保持する DkimHeaders オブジェクト
 * @param signature 生成したダイジェストを格納する DkimSignature オブジェクト
 *                sig-b-tag, sig-bh-tag, rawheader を書き換える.
 *                sig-h-tag を参照してヘッダを選択する.
 * @param pkey signature 公開鍵
 * @return DSTAT_INFO_DIGEST_MATCH: 署名が一致した.
 *         DSTAT_PERMFAIL_BODYDIGEST_MISMATCH: 本文のダイジェスト値が一致しなかった.
 *         DSTAT_PERMFAIL_HEADERDIGEST_MISMATCH: ヘッダ部のダイジェスト値が一致しなかった.
 *         その他: 検証エラー
 */
dkim_stat_t
DkimDigest_signMessage(DkimDigest *self, const MailHeaders *headers, DkimSignature *signature,
                       EVP_PKEY *pkey)
{
    assert(NULL != self);
    assert(NULL != headers);
    assert(NULL != signature);
    assert(NULL != pkey);

    // XXX signature と self の署名/ダイジェストアルゴリズムが一致しているか確認した方がいい

    // body-hash の計算
    const unsigned char *canonbuf;
    size_t canonsize;
    dkim_stat_t ret = DkimCanon_bodyfin(self->canon, &canonbuf, &canonsize);
    if (DSTAT_OK != ret) {
        return ret;
    }   // end if
    ret = DkimDigest_updateBodyChunk(self, canonbuf, canonsize);
    if (DSTAT_OK != ret) {
        return ret;
    }   // end if
    unsigned char bodyhashbuf[EVP_MD_size(self->digest_type)];  // 安全にいくなら EVP_MAX_MD_SIZE
    unsigned int bodyhashlen;
    bodyhashlen = EVP_MD_size(self->digest_type);
    if (0 == EVP_DigestFinal(&self->body_digest, bodyhashbuf, &bodyhashlen)) {
        DkimLogSysError(self->policy, "DigestFinal (of body) failed");
        DkimDigest_logOpenSSLErrors(self);
        return DSTAT_SYSERR_DIGEST_UPDATE_FAILURE;
    }   // end if
    ret = DkimSignature_setBodyHash(signature, bodyhashbuf, bodyhashlen);
    if (DSTAT_OK != ret) {
        return ret;
    }   // end if

#ifdef DEBUG
    DUMPBINDATA(stdout, "bodyhash", bodyhashbuf, bodyhashlen);
#endif

    // header hash の計算と署名
    ret = DkimDigest_digestHeaders(self, headers, DkimSignature_getSelectHeader(signature));
    if (DSTAT_OK != ret) {
        return ret;
    }   // end if
    // sig-b-tag 抜きの DKIM-Signature ヘッダの生成
    const char *rawheaderf, *rawheaderv;
    ret = DkimSignature_buildRawHeader(signature, true, &rawheaderf, &rawheaderv);
    if (DSTAT_OK != ret) {
        return ret;
    }   // end if

    // sig-b-tag 抜きなので通常のヘッダと同様にダイジェスト値を計算すればよい.
    // DKIM-Signature ヘッダの末尾には CRLF を付加しない.
    ret = DkimDigest_updateHeader(self, rawheaderf, rawheaderv, false);
    if (DSTAT_OK != ret) {
        return ret;
    }   // end if
    DkimDigest_closeCanonDump(self);

    unsigned char signbuf[EVP_PKEY_size(pkey)];
    unsigned int signlen;
    if (0 == EVP_SignFinal(&self->header_digest, signbuf, &signlen, pkey)) {
        DkimLogSysError(self->policy, "SignFinal (of body) failed");
        DkimDigest_logOpenSSLErrors(self);
        return DSTAT_SYSERR_DIGEST_UPDATE_FAILURE;
    }   // end if
    ret = DkimSignature_setHeaderHash(signature, signbuf, signlen);
    if (DSTAT_OK != ret) {
        return ret;
    }   // end if

    return DSTAT_OK;
}   // end function : DkimDigest_signMessage
