/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: dkimenum.c 873 2009-04-01 06:30:54Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: dkimenum.c 873 2009-04-01 06:30:54Z takahiko $");

#include <stdio.h>
#include <string.h>

#include "xskip.h"
#include "keywordmap.h"
#include "dkim.h"
#include "dkimtvobj.h"
#include "dkimenum.h"

static const KeywordMap dkim_canonalg_tbl[] = {
    {"simple", DKIM_CANONALG_SIMPLE},
    {"relaxed", DKIM_CANONALG_RELAXED},
    {"nowsp", DKIM_CANONALG_NOWSP}, // obsolete
    {NULL, DKIM_CANONALG_NULL},
};

static const KeywordMap dkim_pubkeyalg_tbl[] = {
    {"rsa", DKIM_PUBKEYALG_RSA},
    {NULL, DKIM_PUBKEYALG_NULL},
};

static const KeywordMap dkim_digestalg_tbl[] = {
    {"sha1", DKIM_DIGESTALG_SHA1},
    {"sha256", DKIM_DIGESTALG_SHA256},
    {NULL, DKIM_DIGESTALG_NULL},
};

static const KeywordMap dkim_srvtype_tbl[] = {
    {"*", DKIM_SRVTYPE_ANY},
    {"email", DKIM_SRVTYPE_EMAIL},
    {NULL, DKIM_DIGESTALG_NULL},
};

static const KeywordMap dkim_keyflag_tbl[] = {
    {"y", DKIM_KEYFLAG_TESTING},
    {"s", DKIM_KEYFLAG_PROHIBIT_SUBDOMAIN},
    {NULL, DKIM_KEYFLAG_NULL},
};

static const KeywordMap dkim_keyretr_tbl[] = {
    {"dns/txt", DKIM_KEYRETR_DNS_TXT},
    // {"dns/dkk", DKIM_KEYRETR_DNS_DKK},
    {"dns", DKIM_KEYRETR_DNS_TXT},  // for backward compatibility
    {NULL, DKIM_KEYRETR_NULL},
};

static const KeywordMap dkim_signpractice_tbl[] = {
    {"unknown", DKIM_ADSP_SIGNPRACTICE_UNKNOWN},
    {"all", DKIM_ADSP_SIGNPRACTICE_ALL},
    {"discardable", DKIM_ADSP_SIGNPRACTICE_DISCARDABLE},
    {NULL, DKIM_ADSP_SIGNPRACTICE_NULL},
};

static const KeywordMap dkim_score_tbl[] = {
    {"none", DKIM_SCORE_NONE},
    {"pass", DKIM_SCORE_PASS},
    {"fail", DKIM_SCORE_FAIL},
    {"policy", DKIM_SCORE_POLICY},
    {"neutral", DKIM_SCORE_NEUTRAL},
    {"temperror", DKIM_SCORE_TEMPERROR},
    {"permerror", DKIM_SCORE_PERMERROR},
    {NULL, DKIM_SCORE_NULL},
};

static const KeywordMap dkim_adsp_score_tbl[] = {
    {"none", DKIM_ADSP_SCORE_NONE},
    {"pass", DKIM_ADSP_SCORE_PASS},
    {"unknown", DKIM_ADSP_SCORE_UNKNOWN},
    {"fail", DKIM_ADSP_SCORE_FAIL},
    {"discard", DKIM_ADSP_SCORE_DISCARD},
    {"nxdomain", DKIM_ADSP_SCORE_NXDOMAIN},
    {"temperror", DKIM_ADSP_SCORE_TEMPERROR},
    {"permerror", DKIM_ADSP_SCORE_PERMERROR},
    {NULL, DKIM_ADSP_SCORE_NULL},
};

/*
 * [RFC4871] 3.2. tag 値は指定されていない限り case-sensitive に扱う
 * Tags MUST be interpreted in a case-sensitive manner.  Values MUST be
 * processed as case sensitive unless the specific tag description of
 * semantics specifies case insensitivity.
 */

////////////////////////////////////////////////////////////

dkim_canonalg_t
DkimEnum_lookupCanonAlgByName(const char *keyword)
{
    return (dkim_canonalg_t) KeywordMap_lookupByCaseString(dkim_canonalg_tbl, keyword);
}   // end function : DkimEnum_lookupCanonAlgByName

dkim_canonalg_t
DkimEnum_lookupCanonAlgByNameSlice(const char *head, const char *tail)
{
    return (dkim_canonalg_t) KeywordMap_lookupByCaseStringSlice(dkim_canonalg_tbl, head, tail);
}   // end function : DkimEnum_lookupCanonAlgByNameSlice

const char *
DkimEnum_lookupCanonAlgByValue(dkim_canonalg_t val)
{
    return KeywordMap_lookupByValue(dkim_canonalg_tbl, val);
}   // end function : DkimEnum_lookupCanonAlgByValue

////////////////////////////////////////////////////////////

dkim_pubkeyalg_t
DkimEnum_lookupPubkeyAlgByName(const char *keyword)
{
    return (dkim_pubkeyalg_t) KeywordMap_lookupByCaseString(dkim_pubkeyalg_tbl, keyword);
}   // end function : DkimEnum_lookupPubkeyAlgByName

dkim_pubkeyalg_t
DkimEnum_lookupPubkeyAlgByNameSlice(const char *head, const char *tail)
{
    return (dkim_pubkeyalg_t) KeywordMap_lookupByCaseStringSlice(dkim_pubkeyalg_tbl, head, tail);
}   // end function : DkimEnum_lookupPubkeyAlgByNameSlice

const char *
DkimEnum_lookupPubkeyAlgByValue(dkim_pubkeyalg_t val)
{
    return KeywordMap_lookupByValue(dkim_pubkeyalg_tbl, val);
}   // end function : DkimEnum_lookupPubkeyAlgByValue

////////////////////////////////////////////////////////////

dkim_digestalg_t
DkimEnum_lookupDigestAlgByName(const char *keyword)
{
    return (dkim_digestalg_t) KeywordMap_lookupByCaseString(dkim_digestalg_tbl, keyword);
}   // end function : DkimEnum_lookupDigestAlgByName

dkim_digestalg_t
DkimEnum_lookupDigestAlgByNameSlice(const char *head, const char *tail)
{
    return (dkim_digestalg_t) KeywordMap_lookupByCaseStringSlice(dkim_digestalg_tbl, head, tail);
}   // end function : DkimEnum_lookupDigestAlgByNameSlice

const char *
DkimEnum_lookupDigestAlgByValue(dkim_digestalg_t val)
{
    return KeywordMap_lookupByValue(dkim_digestalg_tbl, val);
}   // end function : DkimEnum_lookupDigestAlgByValue

////////////////////////////////////////////////////////////

dkim_srvtype_t
DkimEnum_lookupServiceTypeByName(const char *keyword)
{
    return (dkim_srvtype_t) KeywordMap_lookupByCaseString(dkim_srvtype_tbl, keyword);
}   // end function : DkimEnum_lookupServiceTypeByName

dkim_srvtype_t
DkimEnum_lookupServiceTypeByNameSlice(const char *head, const char *tail)
{
    return (dkim_srvtype_t) KeywordMap_lookupByCaseStringSlice(dkim_srvtype_tbl, head, tail);
}   // end function : DkimEnum_lookupServiceTypeByNameSlice

const char *
DkimEnum_lookupServiceTypeByValue(dkim_srvtype_t val)
{
    return KeywordMap_lookupByValue(dkim_srvtype_tbl, val);
}   // end function : DkimEnum_lookupServiceTypeByValue

////////////////////////////////////////////////////////////

dkim_keyflag_t
DkimEnum_lookupKeyFlagByName(const char *keyword)
{
    return (dkim_keyflag_t) KeywordMap_lookupByCaseString(dkim_keyflag_tbl, keyword);
}   // end function : DkimEnum_lookupKeyFlagByName

dkim_keyflag_t
DkimEnum_lookupKeyFlagByNameSlice(const char *head, const char *tail)
{
    return (dkim_keyflag_t) KeywordMap_lookupByCaseStringSlice(dkim_keyflag_tbl, head, tail);
}   // end function : DkimEnum_lookupKeyFlagByNameSlice

const char *
DkimEnum_lookupKeyFlagByValue(dkim_keyflag_t val)
{
    return KeywordMap_lookupByValue(dkim_keyflag_tbl, val);
}   // end function : DkimEnum_lookupKeyFlagByValue

////////////////////////////////////////////////////////////

dkim_keyretr_t
DkimEnum_lookupKeyRetrByName(const char *keyword)
{
    return (dkim_keyretr_t) KeywordMap_lookupByCaseString(dkim_keyretr_tbl, keyword);
}   // end function : DkimEnum_lookupKeyRetrByName

dkim_keyretr_t
DkimEnum_lookupKeyRetrByNameSlice(const char *head, const char *tail)
{
    return (dkim_keyretr_t) KeywordMap_lookupByCaseStringSlice(dkim_keyretr_tbl, head, tail);
}   // end function : DkimEnum_lookupKeyRetrByNameSlice

const char *
DkimEnum_lookupKeyRetrByValue(dkim_keyretr_t val)
{
    return KeywordMap_lookupByValue(dkim_keyretr_tbl, val);
}   // end function : DkimEnum_lookupKeyRetrByValue

////////////////////////////////////////////////////////////

dkim_adsp_signpractice_t
DkimEnum_lookupSignPracticeByName(const char *keyword)
{
    return (dkim_adsp_signpractice_t) KeywordMap_lookupByCaseString(dkim_signpractice_tbl, keyword);
}   // end function : DkimEnum_lookupSignPracticeByName

dkim_adsp_signpractice_t
DkimEnum_lookupSignPracticeByNameSlice(const char *head, const char *tail)
{
    return (dkim_adsp_signpractice_t) KeywordMap_lookupByCaseStringSlice(dkim_signpractice_tbl,
                                                                         head, tail);
}   // end function : DkimEnum_lookupSignPracticeByNameSlice

const char *
DkimEnum_lookupSignPracticeByValue(dkim_adsp_signpractice_t val)
{
    return KeywordMap_lookupByValue(dkim_signpractice_tbl, val);
}   // end function : DkimEnum_lookupSignPracticeByValue

////////////////////////////////////////////////////////////

dkim_score_t
DkimEnum_lookupScoreByName(const char *keyword)
{
    return (dkim_score_t) KeywordMap_lookupByCaseString(dkim_score_tbl, keyword);
}   // end function : DkimEnum_lookupScoreByName

dkim_score_t
DkimEnum_lookupScoreByNameSlice(const char *head, const char *tail)
{
    return (dkim_score_t) KeywordMap_lookupByCaseStringSlice(dkim_score_tbl, head, tail);
}   // end function : DkimEnum_lookupScoreByNameSlice

const char *
DkimEnum_lookupScoreByValue(dkim_score_t val)
{
    return KeywordMap_lookupByValue(dkim_score_tbl, val);
}   // end function : DkimEnum_lookupScoreByValue

////////////////////////////////////////////////////////////

dkim_adsp_score_t
DkimEnum_lookupAdspScoreByName(const char *keyword)
{
    return (dkim_adsp_score_t) KeywordMap_lookupByCaseString(dkim_adsp_score_tbl, keyword);
}   // end function : DkimEnum_lookupAdspScoreByName

dkim_adsp_score_t
DkimEnum_lookupAdspScoreByNameSlice(const char *head, const char *tail)
{
    return (dkim_adsp_score_t) KeywordMap_lookupByCaseStringSlice(dkim_adsp_score_tbl, head, tail);
}   // end function : DkimEnum_lookupAdspScoreByNameSlice

const char *
DkimEnum_lookupAdspScoreByValue(dkim_adsp_score_t val)
{
    return KeywordMap_lookupByValue(dkim_adsp_score_tbl, val);
}   // end function : DkimEnum_lookupAdspScoreByValue

////////////////////////////////////////////////////////////

typedef struct dstatmap {
    dkim_stat_t val;
    const char *string;
} DkimStatMap;

#define CODE2STRMAP(s) {s, #s}

static const DkimStatMap dstatcodemap[] = {
#include "dstat.map"
    {0, NULL},
};

static const char *
DkimEnum_lookupDkimStatByValue(dkim_stat_t val)
{
    const DkimStatMap *p;
    for (p = dstatcodemap; NULL != p->string; ++p) {
        if (p->val == val) {
            return p->string;
        }   // end if
    }   // end for
    return NULL;
}   // end function : DkimEnum_lookupDkimStatByValue

extern const char *
DKIM_strerror(dkim_stat_t code)
{
    const char *errstr = DkimEnum_lookupDkimStatByValue(code);
    return NULL != errstr ? errstr : "unexpected dkim status";
}   // end function : DKIM_strerror

////////////////////////////////////////////////////////////
