/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: dkimpubkey.c 847 2009-03-29 13:34:25Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: dkimpubkey.c 847 2009-03-29 13:34:25Z takahiko $");

#include <assert.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <strings.h>
#include <stdbool.h>

#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <openssl/x509.h>

#include "dkimlogger.h"
#include "ptrop.h"
#include "xbuffer.h"
#include "xskip.h"
#include "xparse.h"
#include "pstring.h"
#include "intarray.h"
#include "inetdomain.h"
#include "inetmailbox.h"
#include "dnsresolv.h"
#include "dkim.h"
#include "dkimspec.h"
#include "dkimenum.h"
#include "dkimwildcard.h"
#include "dkimtvobj.h"
#include "dkimparse.h"
#include "dkimsignature.h"
#include "dkimpubkey.h"

#ifndef MIN
#define MIN(a,b)    ((a) < (b) ? (a) : (b))
#endif

struct DkimPubkey {
    DkimTvobj_MEMBER;
    dkim_digestalg_t digestalg; // key-h-tag
    dkim_pubkeyalg_t pubkeyalg; // key-k-tag
    dkim_srvtype_t srvtype;     // key-s-tag
    dkim_keyflag_t tflag;       // key-t-tag
    EVP_PKEY *pkey;             // key-p-tag
    char *granularity;          // key-g-tag

    bool parsed_v;
    bool parsed_g;
    bool parsed_h;
    bool parsed_k;
    bool parsed_n;
    bool parsed_p;
    bool parsed_s;
    bool parsed_t;
};

static dkim_stat_t DkimPubkey_parse_v(DkimTvobj *base, const DkimTagParseContext *context,
                                      const char **nextp);
static dkim_stat_t DkimPubkey_parse_g(DkimTvobj *base, const DkimTagParseContext *context,
                                      const char **nextp);
static dkim_stat_t DkimPubkey_parse_h(DkimTvobj *base, const DkimTagParseContext *context,
                                      const char **nextp);
static dkim_stat_t DkimPubkey_parse_k(DkimTvobj *base, const DkimTagParseContext *context,
                                      const char **nextp);
static dkim_stat_t DkimPubkey_parse_p(DkimTvobj *base, const DkimTagParseContext *context,
                                      const char **nextp);
static dkim_stat_t DkimPubkey_parse_s(DkimTvobj *base, const DkimTagParseContext *context,
                                      const char **nextp);
static dkim_stat_t DkimPubkey_parse_t(DkimTvobj *base, const DkimTagParseContext *context,
                                      const char **nextp);

// DKIM 公開鍵レコードのタグに対するパース関数のマップ
static const DkimTvobjFieldMap dkim_pubkey_field_tbl[] = {
    {"v", DkimPubkey_parse_v, false, DKIM1_VERSION_TAG, offsetof(DkimPubkey, parsed_v)},
    {"g", DkimPubkey_parse_g, false, "*", offsetof(DkimPubkey, parsed_g)},
    {"h", DkimPubkey_parse_h, false, "sha1:sha256", offsetof(DkimPubkey, parsed_h)},
    {"k", DkimPubkey_parse_k, false, "rsa", offsetof(DkimPubkey, parsed_k)},
    {"n", NULL, false, NULL, offsetof(DkimPubkey, parsed_n)},
    {"p", DkimPubkey_parse_p, true, NULL, offsetof(DkimPubkey, parsed_p)},
    {"s", DkimPubkey_parse_s, false, "*", offsetof(DkimPubkey, parsed_s)},
    {"t", DkimPubkey_parse_t, false, NULL, offsetof(DkimPubkey, parsed_t)},
    {NULL, NULL, false, NULL, 0},   // sentinel
};

// key-h-tag が指定されていない場合のデフォルトは「全てのアルゴリズムを許可」するなのだが，それを設定できない.
// 仕方がないので現状では "sha1:sha256" としてごまかしている.

////////////////////////////////////////////////////////////////////////
// private functions

/*
 * [RFC4871]
 * key-v-tag    = %x76 [FWS] "=" [FWS] "DKIM1"
 */
dkim_stat_t
DkimPubkey_parse_v(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimPubkey *self = (DkimPubkey *) base;

    if (0 < context->tagno) {   // レコードの先頭 (0) での出現とデフォルト値としての設定 (-1) のみ許可し, 他は文法エラー
        *nextp = context->valuehead;
        DkimLogPermFail(self->policy,
                        "key-v-tag appeared not at the front of public key record: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if

    // compare "DKIM1" tag case-sensitively
    if (0 < XSkip_string(context->valuehead, context->valuetail, DKIM1_VERSION_TAG, nextp)) {
        return DSTAT_OK;
    } else {
        *nextp = context->valuehead;
        DkimLogPermFail(self->policy, "unsupported public key version tag: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_UNSUPPORTED_PUBKEY_VERSION;
    }   // end if
}   // end function : DkimPubkey_parse_v

/*
 * [RFC4871]
 * key-g-tag       = %x67 [FWS] "=" [FWS] key-g-tag-lpart
 * key-g-tag-lpart = [dot-atom-text] ["*" [dot-atom-text] ]
 */
dkim_stat_t
DkimPubkey_parse_g(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimPubkey *self = (DkimPubkey *) base;

    if (NULL != self->granularity) {
        DkimLogImplError(self->policy, "key-g-tag already set");
        return DSTAT_SYSERR_IMPLERROR;
    }   // end if

    // key-g-tag は '*' を1個だけ wildcard として受け取るが,
    // dot-atom-text の構成文字には '*' が含まれる.
    // key-g-tag に複数個の '*' が含まれる場合の動作については RFC4871 に明記されていない.
    // '*' は atext に含まれるので XSkip_looseDotAtomText() 一発でよい.
    // key-g-tag の値は長さが 0 の場合もあるので返値は見ない.
    XSkip_looseDotAtomText(context->valuehead, context->valuetail, nextp);
    self->granularity = strpdup(context->valuehead, *nextp);
    if (NULL == self->granularity) {
        DkimLogNoResource(self->policy);
        return DSTAT_SYSERR_NORESOURCE;
    }   // end if

    return DSTAT_OK;
}   // end function : DkimPubkey_parse_g

/*
 * [RFC4871]
 * key-h-tag       = %x68 [FWS] "=" [FWS] key-h-tag-alg
 *                   0*( [FWS] ":" [FWS] key-h-tag-alg )
 * key-h-tag-alg   = "sha1" / "sha256" / x-key-h-tag-alg
 * x-key-h-tag-alg = hyphenated-word   ; for future extension
 */
dkim_stat_t
DkimPubkey_parse_h(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    const char *p = context->valuehead;
    const char *algtail;
    DkimPubkey *self = (DkimPubkey *) base;

    self->digestalg = DKIM_DIGESTALG_NULL;
    *nextp = context->valuehead;
    do {
        XSkip_fws(p, context->valuetail, &p);
        if (0 >= XSkip_hyphenatedWord(p, context->valuetail, &algtail)) {
            // hyphenated-word が見つからなかった.
            DkimLogPermFail(self->policy, "key-h-tag has no valid digest algorithm: near %.50s",
                            context->valuehead);
            return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
        }   // end if

        dkim_digestalg_t digestalg = DkimEnum_lookupDigestAlgByNameSlice(p, algtail);
        // SPEC: 未知の key-h-tag の値の扱いは定められていないので無視する.
        // SPEC: key-h-tag の値に同一のキーワードが複数回含まれていても気にしない.
        if (DKIM_DIGESTALG_NULL != digestalg) {
            self->digestalg |= digestalg;
        }   // end if

        *nextp = algtail;   // key-h-tag が終端するのはこのタイミング
        XSkip_fws(algtail, context->valuetail, &p);
    } while (0 < XSkip_char(p, context->valuetail, ':', &p));
    return DSTAT_OK;
}   // end function : DkimPubkey_parse_h

/*
 * [RFC4871]
 * key-k-tag        = %x76 [FWS] "=" [FWS] key-k-tag-type
 * key-k-tag-type   = "rsa" / x-key-k-tag-type
 * x-key-k-tag-type = hyphenated-word   ; for future extension
 */
dkim_stat_t
DkimPubkey_parse_k(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimPubkey *self = (DkimPubkey *) base;
    self->pubkeyalg = DkimEnum_lookupPubkeyAlgByNameSlice(context->valuehead, context->valuetail);
    if (DKIM_PUBKEYALG_NULL != self->pubkeyalg) {
        *nextp = context->valuetail;
        return DSTAT_OK;
    } else {
        *nextp = context->valuehead;
        DkimLogPermFail(self->policy, "unsupported public key algorithm: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_UNSUPPORTED_PUBKEYALG;
    }   // end if
}   // end function : DkimPubkey_parse_k

/*
 * [RFC4871]
 * key-n-tag    = %x6e [FWS] "=" [FWS] qp-section
 *
 * key-n-tag は人間が読むことを目的としたタグなので, フォーマットの検証すらおこなわない.
 */

/*
 * [RFC4871]
 * key-p-tag    = %x70 [FWS] "=" [ [FWS] base64string ]
 */
dkim_stat_t
DkimPubkey_parse_p(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimPubkey *self = (DkimPubkey *) base;
    const char *p = context->valuehead;

    SETDEREF(nextp, context->valuehead);
    XSkip_fws(p, context->valuetail, &p);
    if (context->valuetail <= p) {
        // revoke されている
        DkimLogPermFail(self->policy, "public key revoked");
        return DSTAT_PERMFAIL_PUBKEY_REVOKED;
    }   // end if

    dkim_stat_t decode_stat;
    XBuffer *rawpubkey =
        DkimParse_decodeBase64(self->policy, p, context->valuetail, &p, &decode_stat);
    if (NULL == rawpubkey) {
        return decode_stat;
    }   // end if

    const unsigned char *pbuf = XBuffer_getBytes(rawpubkey);
    size_t psize = XBuffer_getSize(rawpubkey);

    // d2i_PUBKEY() の第2引数は (const なのに) 書き換えられる (?) ことに注意
    self->pkey = d2i_PUBKEY(NULL, &pbuf, psize);
    XBuffer_free(rawpubkey);
    if (NULL == self->pkey) {
        DkimLogPermFail(self->policy, "key-p-tag doesn't valid public key record, record=%s",
                        context->valuehead);
        return DSTAT_PERMFAIL_PUBKEY_BROKEN;
    }   // end if

    SETDEREF(nextp, p);
    return DSTAT_OK;
}   // end function : DkimPubkey_parse_p

/*
 * [RFC4871]
 * key-s-tag        = %x73 [FWS] "=" [FWS] key-s-tag-type
 *                     0*( [FWS] ":" [FWS] key-s-tag-type )
 * key-s-tag-type   = "email" / "*" / x-key-s-tag-type
 * x-key-s-tag-type = hyphenated-word   ; for future extension
 */
dkim_stat_t
DkimPubkey_parse_s(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    const char *p = context->valuehead;
    const char *srvtail;
    DkimPubkey *self = (DkimPubkey *) base;

    self->srvtype = DKIM_SRVTYPE_NULL;
    *nextp = context->valuehead;
    do {
        XSkip_fws(p, context->valuetail, &p);
        // hypenated-word には "*" が含まれないことに注意.
        if (0 >= XSkip_hyphenatedWord(p, context->valuetail, &srvtail)
            && 0 >= XSkip_char(p, context->valuetail, '*', &srvtail)) {
            // hypenated-word / "*" が見つからなかった.
            DkimLogPermFail(self->policy, "key-s-tag has no valid service type: near %.50s",
                            context->valuehead);
            return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
        }   // end if

        dkim_srvtype_t srvtype = DkimEnum_lookupServiceTypeByNameSlice(p, srvtail);
        // SPEC: 未知の key-s-tag の値の扱いは定められていないので無視する.
        // SPEC: key-s-tag の値に同一のキーワードが複数回含まれていても気にしない.
        if (DKIM_SRVTYPE_NULL != srvtype) {
            self->srvtype |= srvtype;
        }   // end if

        *nextp = srvtail;   // key-s-tag が終端するのはこのタイミング
        XSkip_fws(srvtail, context->valuetail, &p);
    } while (0 < XSkip_char(p, context->valuetail, ':', &p));
    return DSTAT_OK;
}   // end function : DkimPubkey_parse_s

/*
 * [RFC4871]
 * key-t-tag        = %x74 [FWS] "=" [FWS] key-t-tag-flag
 *                    0*( [FWS] ":" [FWS] key-t-tag-flag )
 * key-t-tag-flag   = "y" / "s" / x-key-t-tag-flag
 * x-key-t-tag-flag = hyphenated-word   ; for future extension
 */
dkim_stat_t
DkimPubkey_parse_t(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimPubkey *self = (DkimPubkey *) base;
    return DkimParse_tflag(self->policy, context->valuehead, context->valuetail, nextp,
                           DKIM_KEYFLAG_PROHIBIT_SUBDOMAIN | DKIM_KEYFLAG_TESTING, &(self->tflag));
}   // end function : DkimPubkey_parse_t

////////////////////////////////////////////////////////////////////////
// public functions

/**
 * @error DSTAT_OK 成功
 * @error DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION tag=value ペアの構造に文法エラーがある
 * @error DSTAT_PERMFAIL_MISSING_REQUIRED_TAG 必須タグがレコード中で指定されていない
 * @error DSTAT_PERMFAIL_TAG_DUPLICATED 既にパース済みのタグに遭遇した
 * @error DSTAT_SYSERR_IMPLERROR デフォルト値として設定されている値がパースできない
 * @error DSTAT_SYSERR_NORESOURCE メモリ確保エラー
 * @error DSTAT_PERMFAIL_UNSUPPORTED_PUBKEY_VERSION
 * @error DSTAT_PERMFAIL_UNSUPPORTED_PUBKEYALG
 * @error DSTAT_PERMFAIL_PUBKEY_REVOKED 公開鍵は破棄されている
 * @error DSTAT_PERMFAIL_PUBKEY_BROKEN 暗号ライブラリによる公開鍵の読み込みに失敗した
 * @error DSTAT_PERMFAIL_PUBKEY_TYPE_MISMATCH
 */
DkimPubkey *
DkimPubkey_build(const DkimPolicy *policy, const char *keyval, const char *domain,
                 dkim_stat_t *dstat)
{
    DkimPubkey *self = (DkimPubkey *) malloc(sizeof(DkimPubkey));
    if (NULL == self) {
        DkimLogNoResource(policy);
        SETDEREF(dstat, DSTAT_SYSERR_NORESOURCE);
        return NULL;
    }   // end if
    memset(self, 0, sizeof(DkimPubkey));
    self->ftbl = dkim_pubkey_field_tbl;
    self->policy = policy;
    dkim_stat_t build_stat = DkimTvobj_build((DkimTvobj *) self, keyval, STRTAIL(keyval), false);
    if (DSTAT_OK != build_stat) {
        DkimLogPermFail(policy, "invalid public key record: domain=%s", domain);
        SETDEREF(dstat, build_stat);
        goto cleanup;
    }   // end if

    // key-k-tag の示す暗号化方式と key-p-tag に格納されている鍵の種類に矛盾がないか確認
    switch (self->pubkeyalg) {
    case DKIM_PUBKEYALG_RSA:
        if (EVP_PKEY_RSA != EVP_PKEY_type(self->pkey->type)) {
            DkimLogPermFail
                (policy,
                 "key-k-tag and key-p-tag doesn't match: domain=%s, keyalg=0x%x, keytype=0x%x",
                 domain, self->pubkeyalg, EVP_PKEY_type(self->pkey->type));
            SETDEREF(dstat, DSTAT_PERMFAIL_PUBKEY_TYPE_MISMATCH);
            goto cleanup;
        }   // end if
        break;
    default:
        DkimLogImplError(policy, "unexpected public key algorithm: pubkeyalg=0x%x",
                         self->pubkeyalg);
        SETDEREF(dstat, DSTAT_SYSERR_IMPLERROR);
        goto cleanup;
    }   // end switch

    SETDEREF(dstat, DSTAT_OK);
    return self;

  cleanup:
    DkimPubkey_free(self);
    return NULL;
}   // end function : DkimPubkey_build

void
DkimPubkey_free(DkimPubkey *self)
{
    assert(NULL != self);
    free(self->granularity);
    if (NULL != self->pkey) {
        EVP_PKEY_free(self->pkey);
    }   // end if
    free(self);
}   // end function : DkimPubkey_free

static bool
DkimPubkey_isDigestAlgMatched(const DkimPubkey *self, dkim_digestalg_t digestalg)
{
    return ((self->digestalg) & digestalg) ? true : false;
}   // end function : DkimPubkey_isDigestAlgMatched

static bool
DkimPubkey_isPubKeyAlgMatched(const DkimPubkey *self, dkim_pubkeyalg_t pubkeyalg)
{
    return self->pubkeyalg == pubkeyalg;
}   // end function : DkimPubkey_isPubKeyAlgMatched

/*
 * @attention service type に "email" を持つレコードのみ採用する.
 *            "email" 以外の用途に使用する場合は改修が必要.
 * @error DSTAT_OK 成功
 * @error DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION tag=value ペアの構造に文法エラーがある
 * @error DSTAT_PERMFAIL_MISSING_REQUIRED_TAG 必須タグがレコード中で指定されていない
 * @error DSTAT_PERMFAIL_TAG_DUPLICATED 既にパース済みのタグに遭遇した
 * @error DSTAT_SYSERR_IMPLERROR デフォルト値として設定されている値がパースできない
 * @error DSTAT_SYSERR_NORESOURCE メモリ確保エラー
 * @error DSTAT_PERMFAIL_UNSUPPORTED_PUBKEY_VERSION
 * @error DSTAT_PERMFAIL_UNSUPPORTED_PUBKEYALG
 * @error DSTAT_PERMFAIL_PUBKEY_REVOKED 公開鍵は破棄されている
 * @error DSTAT_PERMFAIL_PUBKEY_BROKEN 暗号ライブラリによる公開鍵の読み込みに失敗した
 * @error DSTAT_PERMFAIL_PUBKEY_TYPE_MISMATCH
 * @error DSTAT_PERMFAIL_PUBKEY_SRVTYPE_MISMATCH
 * @error DSTAT_PERMFAIL_PUBKEY_DIGESTALG_MISMATCH
 * @error DSTAT_PERMFAIL_PUBKEY_PUBKEYALG_MISMATCH
 * @error DSTAT_PERMFAIL_PUBKEY_SUBDOMAIN_PROHIBITED
 * @error DSTAT_PERMFAIL_PUBKEY_GRANULARITY_MISMATCH
 */
static DkimPubkey *
DkimPubkey_buildFromDnsTxt(const DkimPolicy *policy, const char *record, const char *dkimdomain,
                           const DkimSignature *signature, dkim_stat_t *dstat)
{
    assert(NULL != record);
    assert(NULL != dkimdomain);
    assert(NULL != signature);

    DkimPubkey *self = DkimPubkey_build(policy, record, dkimdomain, dstat);
    if (NULL == self) {
        return NULL;
    }   // end if

    // 得られた公開鍵が署名ヘッダの要求を満たすか確認
    // [RFC4871] 3.6.1. e-mail に使用可能か調べる
    if (!DkimPubkey_isEMailServiceUsable(self)) {
        DkimLogPermFail(policy,
                        "omitting public key record for service type mismatch: pubkey=%s", record);
        SETDEREF(dstat, DSTAT_PERMFAIL_PUBKEY_SRVTYPE_MISMATCH);
        goto cleanup;
    }   // end if

    // [RFC4871] 6.1.2. 7. key-h-tag と sig-a-tag-h (ダイジェストアルゴリズム) の比較
    if (!DkimPubkey_isDigestAlgMatched(self, DkimSignature_getDigestAlg(signature))) {
        DkimLogPermFail
            (policy,
             "omitting public key record for digest algorithm mismatch: digestalg=%s, pubkey=%s",
             DkimEnum_lookupDigestAlgByValue(DkimSignature_getDigestAlg(signature)), record);
        SETDEREF(dstat, DSTAT_PERMFAIL_PUBKEY_DIGESTALG_MISMATCH);
        goto cleanup;
    }   // end if

    // [RFC4871] 6.1.2. 9. key-k-tag と sig-a-tag-k (暗号化アルゴリズム) の比較
    if (!DkimPubkey_isPubKeyAlgMatched(self, DkimSignature_getPubKeyAlg(signature))) {
        DkimLogPermFail
            (policy,
             "omitting public key record for public key algorithm mismatch: pubkeyalg=%s, pubkey=%s",
             DkimEnum_lookupPubkeyAlgByValue(DkimSignature_getPubKeyAlg(signature)), record);
        SETDEREF(dstat, DSTAT_PERMFAIL_PUBKEY_PUBKEYALG_MISMATCH);
        goto cleanup;
    }   // end if

    // [RFC4871] 3.8.
    // If the referenced key record contains the "s" flag as part of the
    // "t=" tag, the domain of the signing identity ("i=" flag) MUST be the
    // same as that of the d= domain.  If this flag is absent, the domain of
    // the signing identity MUST be the same as, or a subdomain of, the d=
    // domain.  Key records that are not intended for use with subdomains
    // SHOULD specify the "s" flag in the "t=" tag.

    // ただし sig-i-tag が sig-d-tag のサブドメインであることは DkimSignature_validate() で検証済み.
    const InetMailbox *identity = DkimSignature_getIdentity(signature);
    if (DkimPubkey_isSubdomainProhibited(self)) {
        // key-t-tag に "s" フラグが含まれている場合,
        // sig-i-tag が sig-d-tag に完全に一致することを確認する. サブドメインは許容されない.
        if (!InetDomain_equals
            (DkimSignature_getSigningDomain(signature), InetMailbox_getDomain(identity))) {
            DkimLogPermFail
                (policy,
                 "omitting public key record for subdomain prohibition: identity=%s, domain=%s",
                 InetMailbox_getDomain(identity), DkimSignature_getSigningDomain(signature));
            SETDEREF(dstat, DSTAT_PERMFAIL_PUBKEY_SUBDOMAIN_PROHIBITED);
            goto cleanup;
        }   // end if
    }   // end if

    // [RFC4871] 6.1.2. 6. key-g-tag と sig-i-tag のローカルパートを比較
    const char *granularity = DkimPubkey_getGranularity(self);
    const char *localpart = InetMailbox_getLocalPart(identity);
    if (!DkimWildcard_matchPubkeyGranularity
        (granularity, STRTAIL(granularity), localpart, STRTAIL(localpart))) {
        DkimLogPermFail
            (policy,
             "omitting public key record for granularity mismatch: identity=%s, granularity=%s",
             localpart, granularity);
        SETDEREF(dstat, DSTAT_PERMFAIL_PUBKEY_GRANULARITY_MISMATCH);
        goto cleanup;
    }   // end if

    SETDEREF(dstat, DSTAT_OK);
    return self;

  cleanup:
    DkimPubkey_free(self);
    return NULL;
}   // end function : DkimPubkey_buildFromDnsTxt

/*
 * @attention 返値は使用後に free すること.
 */
static char *
DkimPubkey_buildQueryDomain(const DkimPolicy *policy, const DkimSignature *signature,
                            dkim_stat_t *dstat)
{
    assert(NULL != signature);

    const char *domain = DkimSignature_getSigningDomain(signature);
    const char *selector = DkimSignature_getSelector(signature);

    size_t buflen = strlen(selector) + sizeof("." DKIM_DNS_NAMESPACE ".") + strlen(domain);
    char *buf = (char *) malloc(buflen);
    if (NULL == buf) {
        DkimLogNoResource(policy);
        SETDEREF(dstat, DSTAT_SYSERR_NORESOURCE);
        return NULL;
    }   // end if

    // クエリ先ドメインの生成
    if ((int) buflen <= snprintf(buf, buflen, "%s." DKIM_DNS_NAMESPACE ".%s", selector, domain)) {
        DkimLogImplError(policy, "allocated memory too small");
        SETDEREF(dstat, DSTAT_SYSERR_IMPLERROR);
        free(buf);
        return NULL;
    }   // end if

    SETDEREF(dstat, DSTAT_OK);
    return buf;
}   // end function : DkimPubkey_buildQueryDomain

/**
 * @error DSTAT_OK 成功
 * @error DSTAT_SYSERR_NORESOURCE メモリ確保エラー
 * @error DSTAT_SYSERR_IMPLERROR
 * @error DSTAT_PERMFAIL_PUBKEY_NOT_EXIST
 * @error DSTAT_TMPERR_DNS_LOOKUP_FAILURE
 */
static DkimPubkey *
DkimPubkey_retrieveDnsTxt(const DkimPolicy *policy, const DkimSignature *signature,
                          DnsResolver *resolver, dkim_stat_t *dstat)
{
    assert(NULL != signature);
    assert(NULL != resolver);

    DkimPubkey *self = NULL;
    DnsTxtResponse *txt_rr = NULL;

    char *domain = DkimPubkey_buildQueryDomain(policy, signature, dstat);
    if (NULL == domain) {
        goto finally;
    }   // end if

    // [RFC4871] 6.1.2. 1.
    // 公開鍵レコードを探す
    dns_stat_t txtquery_stat = DnsResolver_lookupTxt(resolver, domain, &txt_rr);
    switch (txtquery_stat) {
    case DNS_STAT_NOERROR:;
        // [RFC4871] 6.1.2.
        // The verifier MUST validate the key record and MUST
        // ignore any public key records that are malformed.

        // [RFC4871] 6.1.2. 4.
        //  If the query for the public key returns multiple key records, the
        //  verifier may choose one of the key records or may cycle through
        //  the key records performing the remainder of these steps on each
        //  record at the discretion of the implementer.  The order of the
        //  key records is unspecified.  If the verifier chooses to cycle
        //  through the key records, then the "return ..." wording in the
        //  remainder of this section means "try the next key record, if any;
        //  if none, return to try another signature in the usual way".

        int recnum = MIN(DnsTxtResponse_size(txt_rr), DKIM_PUBKEY_CANDIDATE_MAX);   // DoS 対策のために上限を設ける
        for (int i = 0; i < recnum; ++i) {
            dkim_stat_t retr_dstat;
            self =
                DkimPubkey_buildFromDnsTxt(policy, DnsTxtResponse_data(txt_rr, i), domain,
                                           signature, &retr_dstat);
            if (NULL != self) {
                // 公開鍵レコードとして適切だった
                DnsTxtResponse_free(txt_rr);
                SETDEREF(dstat, DSTAT_OK);
                goto finally;   // 正常終了
            } else if (DSTAT_ISCRITERR(retr_dstat)) {
                // システムエラーが発生した場合はすぐに抜ける
                DkimLogSysError
                    (policy,
                     "System error occurred while parsing public key: domain=%s, err=%s, record=%s",
                     domain, DKIM_strerror(retr_dstat), NNSTR(DnsTxtResponse_data(txt_rr, i)));
                DnsTxtResponse_free(txt_rr);
                SETDEREF(dstat, retr_dstat);
                goto finally;
            } else if (DSTAT_ISPERMFAIL(retr_dstat)) {
                // 無効な公開鍵レコード (の候補) を破棄した
                DkimLogDebug(policy,
                             "public key candidate discarded: domain=%s, err=%s, record=%s", domain,
                             DKIM_strerror(retr_dstat), NNSTR(DnsTxtResponse_data(txt_rr, i)));
            }   // end if
        }   // end for
        // 有効な公開鍵レコードはなかった
        DnsTxtResponse_free(txt_rr);
        DkimLogPermFail(policy, "No suitable public key record found from DNS: domain=%s", domain);
        SETDEREF(dstat, DSTAT_PERMFAIL_PUBKEY_NOT_EXIST);
        break;

    case DNS_STAT_NXDOMAIN:
    case DNS_STAT_NODATA:
        // [RFC4871] 6.1.2. 3.
        DkimLogPermFail(policy, "No public key record exists on DNS, domain=%s, err=%s",
                        domain, DnsResolver_getErrorString(resolver));
        SETDEREF(dstat, DSTAT_PERMFAIL_PUBKEY_NOT_EXIST);
        break;

    case DNS_STAT_FORMERR:
    case DNS_STAT_SERVFAIL:
    case DNS_STAT_NOTIMPL:
    case DNS_STAT_REFUSED:
    case DNS_STAT_YXDOMAIN:
    case DNS_STAT_YXRRSET:
    case DNS_STAT_NXRRSET:
    case DNS_STAT_NOTAUTH:
    case DNS_STAT_NOTZONE:
    case DNS_STAT_RESERVED11:
    case DNS_STAT_RESERVED12:
    case DNS_STAT_RESERVED13:
    case DNS_STAT_RESERVED14:
    case DNS_STAT_RESERVED15:
        // [RFC4871] 6.1.2. 2.
        DkimLogInfo(policy, "DNS look-up error for public key record, domain=%s, err=%s",
                    domain, DnsResolver_getErrorString(resolver));
        SETDEREF(dstat, DSTAT_TMPERR_DNS_LOOKUP_FAILURE);
        break;

    default:
        DkimLogImplError(policy,
                         "DnsResolver_lookupTxt returns unexpected value: value=0x%x, domain=%s",
                         txtquery_stat, domain);
        SETDEREF(dstat, DSTAT_SYSERR_IMPLERROR);
        break;
    }   // end switch

  finally:
    free(domain);
    return self;
}   // end function : DkimPubkey_retrieveDnsTxt

/**
 * @error DSTAT_OK 成功
 * @error DSTAT_SYSERR_NORESOURCE メモリ確保エラー
 * @error DSTAT_SYSERR_IMPLERROR
 * @error DSTAT_PERMFAIL_PUBKEY_NOT_EXIST
 * @error DSTAT_TMPERR_DNS_LOOKUP_FAILURE
 */
DkimPubkey *
DkimPubkey_retrieve(const DkimPolicy *policy, const DkimSignature *signature, DnsResolver *resolver,
                    dkim_stat_t *dstat)
{
    assert(NULL != signature);
    assert(NULL != resolver);

    const IntArray *keyretr = DkimSignature_getKeyRetrMethod(signature);
    // TODO: 1つでも失敗したら失敗を返せばよいのではないか?
    // [RFC4871] 3.5. sig-q-tag
    //      ...  If there are multiple query mechanisms listed, the
    // choice of query mechanism MUST NOT change the interpretation of
    // the signature.  Implementations MUST use the recognized query
    // mechanisms in the order presented.
    size_t num = IntArray_getCount(keyretr);
    for (size_t n = 0; n < num; ++n) {
        dkim_keyretr_t keyretr_method = (dkim_keyretr_t) IntArray_get(keyretr, n);
        switch (keyretr_method) {
        case DKIM_KEYRETR_DNS_TXT:;
            dkim_stat_t retr_dstat;
            DkimPubkey *self = DkimPubkey_retrieveDnsTxt(policy, signature, resolver, &retr_dstat);
            if (NULL != self) {
                SETDEREF(dstat, DSTAT_OK);
                return self;
            } else if (DSTAT_ISCRITERR(retr_dstat) || DSTAT_ISTMPERR(retr_dstat)) {
                // システムエラーまたは DNS エラーが発生した場合はすぐに抜ける
                SETDEREF(dstat, retr_dstat);
                return NULL;
            }   // end if
            break;

        case DKIM_KEYRETR_NULL:
        default:
            DkimLogImplError(policy,
                             "unexpected public key retrieving method: keyretr_method=0x%x",
                             keyretr_method);
            SETDEREF(dstat, DSTAT_SYSERR_IMPLERROR);
            return NULL;
        }   // end switch
    }   // end for

    // 有効な公開鍵レコードを取得できなかった
    DkimLogPermFail(policy, "no valid public key record found: domain=%s, selector=%s",
                    DkimSignature_getSigningDomain(signature),
                    DkimSignature_getSelector(signature));
    SETDEREF(dstat, DSTAT_PERMFAIL_PUBKEY_NOT_EXIST);
    return NULL;
}   // end function : DkimPubkey_retrieve

////////////////////////////////////////////////////////////////////////
// accessor

EVP_PKEY *
DkimPubkey_getPublicKey(const DkimPubkey *self)
{
    return self->pkey;
}   // end function : DkimPubkey_getPublicKey

bool
DkimPubkey_isTesing(const DkimPubkey *self)
{
    return ((self->tflag) & DKIM_KEYFLAG_TESTING) ? true : false;
}   // end function : DkimPubkey_isTesing

bool
DkimPubkey_isSubdomainProhibited(const DkimPubkey *self)
{
    return ((self->tflag) & DKIM_KEYFLAG_PROHIBIT_SUBDOMAIN) ? true : false;
}   // end function : DkimPubkey_isSubdomainProhibited

bool
DkimPubkey_isEMailServiceUsable(const DkimPubkey *self)
{
    return ((self->srvtype) & DKIM_SRVTYPE_EMAIL) ? true : false;
}   // end function : DkimPubkey_isEMailServiceUsable

dkim_pubkeyalg_t
DkimPubkey_getPubKeyAlg(const DkimPubkey *self)
{
    return self->pubkeyalg;
}   // end function : DkimPubkey_getPubKeyAlg

const char *
DkimPubkey_getGranularity(const DkimPubkey *self)
{
    return self->granularity;
}   // end function : DkimPubkey_getGranularity
