using System;

namespace gnomeguitar_cs
{
	public class Chord
	{
		string name = "None";
		ChordType type = new ChordType("Unknown");
		Note root = new Note(NoteValue.UNKNOWN);
		ChordShape shape = new ChordShape(ChordShapeValue.UNKNOWN);
		Construction construction = new Construction();
		WrittenGroup writtens = new WrittenGroup();
		VoiceingGroup voiceings = new VoiceingGroup();
		KeyGroup keys = new KeyGroup();

		NoteGroup possibleNotes = new NoteGroup();
//	for (i = 0; i < 37; i++){
//		gpGroup_append(possibleNotes, 
//			       GINT_TO_POINTER((int)i));
//	}
		//These have to be public because xmlChordParser fidles with them 
		public bool keysDirty = false;
		public bool possibleNotesDirty = false;
		//used by xmlChordParser
		public Chord (bool leaveEmpty)
		{
		}
		public Chord () : this (new ChordType(""),new Note(NoteValue.NO_NOTE),new ChordShape(ChordShapeValue.UNKNOWN)){}
		public Chord (ChordType type, Tuning tuning, Note root, ChordShape shape)
		{	

			set_type(type);
			set_root(root);
			set_shape(shape); 
			set_name(root.to_text() + type.to_text());
			set_construction(new Construction("1,3,5"));
			add_voiceing(new Voiceing(tuning));
		}	
 
		public Chord (ChordType type, Note root, ChordShape shape):
		this (type, new Tuning("E,A,D,G,B,E"), root, shape){}	

		public Chord (Note root):
		this (new ChordType("Major"), root, new ChordShape(ChordShapeValue.OPEN)){}

		public bool good_for_scale(Scale scale)
		{
			bool hasNote = true;
			int i = 0;
			Note note;
			int noNotes;

			noNotes = get_noNotes();
			for (; i < noNotes && hasNote; i++){
				note = get_note(i);
				hasNote = scale.has_note(note);
				//System.Console.WriteLine("Chord.good_for_scale: Note no. = {0}, note = {1}, hasNote = {2}", i, note.to_text(), hasNote);
			}
		
			if (hasNote == true && i == noNotes){
				return true;	
			} else {
				return false;
			}
		}	
		
		
		public NoteGroup get_possibleNotes()
		{		

			int noteNo;			
			Note note;
			
			if (!possibleNotesDirty){
				return possibleNotes;
			}
		
			possibleNotes = new NoteGroup();	
		
			foreach (Relation relation in construction){
				//text to note strips sharps and flats
				noteNo = relation.to_noteNo();
				note = get_scale_note(noteNo);
				if(relation.is_sharp()){note.sharpen();}
				else if(relation.is_double_sharp()){note.doubleSharpen();}
				else if(relation.is_flat()){note.flatten();}
				else if(relation.is_double_flat()){note.doubleFlatten();}
				possibleNotes.add(note);
			}	
		
			possibleNotes.add(new Note(NoteValue.NO_NOTE));
			possibleNotes.add(new Note(NoteValue.UNKNOWN));
			
			possibleNotesDirty = false;

			return possibleNotes;
		}
	

	
		public void add_construction(Relation relation)
		{
			construction.add(relation);
			set_dirty(true);
		}	
		
		public void remove_construction(Relation relation)
		{	
			construction.remove(relation);
			remove_relation_from_voiceings(relation);
			set_dirty(true);
		}

		public void remove_relation_from_voiceings(Relation relation)
		{
			Note note;

			note = relation_to_note (relation);

			foreach (Voiceing voiceing in voiceings){
				voiceing.remove_note(note);
			}
		}

		public void flatten_construction(Relation relation)
		{
			construction.flatten(relation);
			remove_relation_from_voiceings(relation);
			set_dirty(true);
		}

		public void sharpen_construction(Relation relation)
		{
			construction.sharpen(relation);
			remove_relation_from_voiceings(relation);	
			set_dirty(true);
		}

		public void double_flatten_construction(Relation relation)
		{
			construction.double_flatten(relation);
			remove_relation_from_voiceings(relation);
			set_dirty(true);
		}

		public void double_sharpen_construction(Relation relation)
		{
			construction.double_sharpen(relation);
			remove_relation_from_voiceings(relation);
			set_dirty(true);
		}

		public void set_generic_name(string genericName)
		{
			string root;

			root = get_root().to_text();
			name = (root + " " + genericName);
		}

		public int get_noNotes()
		{
			//FIXME dose this work with unknown 
			return construction.size();
		}

		public Note get_note(int index)
		{
			Note note;
			Relation relation;
			
			relation = get_nth_construction(index);
			//System.Console.WriteLine("Chord.get_note: Relation {0} = {1}", index, relation.to_text());
			note = relation_to_note(relation);
			return note;
		} 

		public Relation get_nth_construction(int index)
		{
			return (Relation)construction.nth(index);
		}

		public void set_name (string name)
		{
			this.name=name;
		}

		public string get_name()
		{
			return name;
		}
		
		public void set_type (ChordType type)
		{
			this.type = type;
		}

		public ChordType get_chordType ()
		{
			return type;
		}

		public void set_root (Note root)
		{
			this.root = root;
		}

		public Note get_root (){
			return root;
		}

		public void append_key(Note key)
		{
			keys.append(key);
		}
		
		public KeyGroup get_keys ()
		{	
			if(keysDirty){
				auto_set_keys();
			}
			return keys;
		}
		
		public bool has_key (Note key)
		{
			bool hasKey = false;

			foreach (Note currentKey in keys){
				if (key == currentKey){
					hasKey = true;
					break;
				}
			}
			
			return hasKey;
		}
		
		public void set_shape (ChordShape shape)
		{
			this.shape = shape;
		}
		
		public ChordShape get_shape ()
		{
			return shape;
		}
		
		public void set_construction (Construction construction)
		{
			this.construction = construction;
			set_dirty(true);
		}
		
		public Construction get_construction ()
		{
			return construction;
		}
		
		public string get_construction_text()
		{
			string text = null;

			text = construction.to_text();
			if (text == null){text = "";}
			return text;
		}
		
		public void set_writtens (WrittenGroup writtens)
		{
			this.writtens = writtens;
		}

		public void append_written (Written written)
		{
			writtens.append(written);
		}
		
		public void remove_written(Written written)
		{	
			writtens.remove(written);
		}
		
		public WrittenGroup get_writtens ()
		{
			return writtens;			
		}
		
		public bool has_written (Written written)
		{
			return writtens.has_equal(written);
		}

		public void set_voiceings (VoiceingGroup voiceings)
		{
			this.voiceings = voiceings;
		}

		public void add_voiceing (Voiceing voiceing)
		{
			voiceings.add(voiceing);
		}
		
		public void append_voiceing (Voiceing voiceing)
		{
			voiceings.append(voiceing);
		}
		
		public VoiceingGroup get_voiceings ()
		{
			return voiceings;
		}
		
		public string[] get_text ()
		{
			string[] text = new string[5];
	
			text[0] = get_name();	
			text[1] = get_text_type();
			text[2] = get_text_root();
			text[3] = get_text_keys();
			text[4] = get_text_shape();
			
			return text;
		}
		
		public int get_noVoiceings()
		{
			return voiceings.get_no();
		}
		
		public Voiceing get_voiceing(int i)
		{
			return (Voiceing)voiceings.nth(0);
		}
		
		
		public Tuning get_tuning ()
		{	
			Voiceing voiceing;
			Tuning tuning;
			
			voiceing = (Voiceing)voiceings.nth(0);
			tuning = voiceing.get_tuning();
			
			return tuning;
		}
		
		
		public int relation_position (Relation relation)
		{
			int constructionPosition;
			Construction construction;
			
			if (relation.to_text() == "X"){
				return 0;
			}
			construction = get_construction();
			constructionPosition = construction.position_equal(relation);
			return constructionPosition + 1;
		}
		
		public int voiceing_position_equal (Voiceing voiceing)
		{
			return voiceings.position_equal(voiceing);
		}
		
		public int voiceing_position_same (Voiceing voiceing)
		{
			return voiceings.position_same(voiceing);
		}

		public Note note_from_relation(Relation relation)
		{
			return relation_to_note(relation);
		}	
		
		public Relation relation_from_note (Note note)
		{
			Note testNote;
			Relation relation = null;
			
			foreach (Relation tempRelation in construction){
				testNote = relation_to_note(tempRelation);	
				if (testNote == note){
					relation = tempRelation;
					break;
				}	
			}
			
			return relation;
		}
		
		public string to_xml()
		{
			string xml = null;
			
			string name = null;
			string type = null;
			string root = null;
			string keysXml = null;
			string shape = null;
			string writtensXml = null;
			string voiceingsXml = null;
			string construction = null;
			
			name = get_name(); 
			type = get_chordType().to_text();
			root = get_root().to_text();
			keysXml = keys.to_xml();
			shape = get_shape().to_text();
			writtensXml = writtens.to_xml();
			voiceingsXml = voiceings.to_xml();
			construction = get_construction_text();
			
			xml = "<chord name=\"" + name +
				"\"><type>" + type +
					"</type><root>" + root +
					"</root>"+ keysXml +
					"<shape>" + shape +
					"</shape><construction>"+ construction +
					"</construction>" + writtensXml + voiceingsXml+
					"</chord>";
			
			return xml;
		}	

		public bool compare_type(ChordType type)
		{	
			return this.type == type;
		}
		
		public bool compare_root (Note root)
		{
			return this.root == root;
		}	
			
		public bool compare_shape  (ChordShape shape)
		{
			return this.shape == shape;
		}	

		public bool compare_tuning (Tuning tuning)
		{
			Voiceing voiceing;
			
			voiceing = get_voiceing(0);
			return tuning.is_equal_group(voiceing.get_tuning());
		}
		
	
		public bool is_equal (Chord other)
		{	
			ChordType thisType = null;
			ChordType otherType = null;
		
			Note thisRoot;
			Note otherRoot;

			ChordShape thisShape;
			ChordShape otherShape;
			
			bool typeSame;
			
			bool unknownType = true;
			
			thisType = get_chordType();
			otherType = other.get_chordType();
			
			typeSame = thisType == otherType;
			
			if (!typeSame){	
				return false;
			}
			
			thisRoot = get_root();
			otherRoot = other.get_root();
			
			if (thisRoot != otherRoot){
				return false;
			}
			
			thisShape = get_shape();
			otherShape = other.get_shape();
				
			if(thisShape != otherShape){
				return false;
			}
			
			if(thisShape == ChordShapeValue.UNKNOWN 
			   && thisRoot == NoteValue.UNKNOWN 
			   && unknownType){
				Voiceing thisVoiceing = null;
				Voiceing otherVoiceing = null;
					
				thisVoiceing = get_voiceing(0);
				otherVoiceing = other.get_voiceing(0);
					
				return thisVoiceing.is_equal(otherVoiceing);
			} 
			return true;
		}
		
		public bool has_equal_voiceing(Voiceing voiceing)
		{
			VoiceingGroup voiceingGroup = null;
			bool hasEqual = false;
			
			voiceingGroup = get_voiceings();
			hasEqual = voiceingGroup.has_equal(voiceing);
			
			return hasEqual;
		}
		
		public string get_text_type ()
		{
			return get_chordType().to_text();
		}
		
		public string get_text_root ()
		{
			return get_root().to_text();
		}
		
		public string get_text_keys ()
		{
			return keys.to_text();
		}
		
		public string get_text_shape ()
		{
			return get_shape().to_text();
		}
		
		public int get_max_no_frets()
		{
			int maxNoFrets = 0;
			int noFrets;
			
			foreach (Voiceing voiceing in voiceings){
				noFrets = voiceing.get_noFrets();
				if(noFrets > maxNoFrets){
					maxNoFrets = noFrets;
				}
			}

			return maxNoFrets;
		}
		
		public int get_max_no_strings()
		{
			int maxNoStrings = 0;
			int noStrings;
			
			foreach (Voiceing voiceing in voiceings){
				noStrings = voiceing.get_no_strings();
				if(noStrings > maxNoStrings){
					maxNoStrings = noStrings;
				}
			}

			return maxNoStrings;
		}
		
		public int get_max_tuning_chars()
		{
			int maxNoChars = 0;
			int noChars;
			
			foreach (Voiceing voiceing in voiceings){
				noChars = voiceing.get_max_tuning_chars();
				if(noChars > maxNoChars){
					maxNoChars = noChars;
				}
			}

			return maxNoChars;
		}
		
		public int get_max_fretted_chars()
		{
			int maxNoChars = 0;
			int noChars;
			
			foreach (Voiceing voiceing in voiceings){
				noChars = voiceing.get_max_fretted_chars();
				if(noChars > maxNoChars){
					maxNoChars = noChars;
				}
			}

			return maxNoChars;
		}
		
		public int get_max_relation_chars()
		{
			int maxNoChars = 0;
			int noChars;
			
			foreach (Voiceing voiceing in voiceings){
				noChars = voiceing.get_max_relation_chars();
				if(noChars > maxNoChars){
					maxNoChars = noChars;
				}
			}

			return maxNoChars;
		}
		
		/***************************************************************************
		 *******************************PRIVATE STUFF*******************************
		 **************************************************************************/


		private void set_dirty (bool value)
		{	
			keysDirty = value;
			possibleNotesDirty = value;
		}
		
		private void auto_set_keys ()
		{
			int noKeys = 36;
			bool hasNote;
			int i, j;
			Scale scale;
			int noNotes;
			
			keys = new KeyGroup();
			//skip NO_NOTE and UNKNOWN_NOTE
			for(i = 2; i < noKeys; i++){
				
				noNotes = get_noNotes();
				scale = new Scale(new Note((NoteValue)i), 0);
				for (j = 0, hasNote = true; j < noNotes && hasNote; j++){
					Note note;
					note = get_note(j);
					hasNote = scale.has_note(note);
				}
				
				if (j == noNotes && hasNote == true){
					keys.append(i);
				}
				
			}
			
			keysDirty = false;
		}
		
		private Note relation_to_note(Relation relation)
		{
			int noteNo;
			Note note;
		
			//		if (relation.text == "X"){
			//			return new Note(NoteValue.NO_NOTE);
			//		}
			
			noteNo = relation.to_noteNo(); 
			note = get_scale_note(noteNo);
			//System.Console.WriteLine("Chord.relation_to_note: NoteNo = {0}, note = {1}", noteNo, note.to_text());
			if (relation.is_flat()){	
				note.flatten();
			} else if (relation.is_double_flat()){
				note.doubleFlatten();
			} else if (relation.is_sharp()){
				note.sharpen();
			} else if (relation.is_double_sharp()){
				note.doubleSharpen();
			}
			
			return note;
		}
		
		
		
		private Note get_scale_note(int noteNo)
		{
			Note note;
			Scale scale;
			
			//chords use notes up to 14 but this confuses scale
			if (noteNo > 7){
				noteNo -= 7;
			}
			scale = new Scale (get_root().clone(), 0);
			//System.Console.WriteLine("Chord.get_scale_note: {0}", scale.to_text());
			note = scale.get_note(noteNo);
			
			return note;
		}
	}
}