using System.Collections;

namespace gnomeguitar_cs
{
	public class Scale {
		
		Interval[] majorIntervals = new Interval[7] {new Interval(IntervalValue.TONE),
			new Interval(IntervalValue.TONE), new Interval(IntervalValue.SEMI_TONE),
			new Interval(IntervalValue.TONE), new Interval(IntervalValue.TONE),
			new Interval(IntervalValue.TONE), new Interval(IntervalValue.SEMI_TONE)};

		string name;
		ObjGroup notes;
		ScaleType type;
			
		public Scale()
		{
			name = null;
			notes = new ObjGroup();
			type = new ScaleType(ScaleTypeValue.UNKNOWN);	
		}

		public Scale (Note r, ScaleType t, int modeNo) :this ()
		{
		//	System.Console.WriteLine("Scale: {0}", root.to_text() + type.to_text() + modeNo);
			//We clone note and type just to make sure it is safe monkey about with them
			Note root = new Note(r);
			ScaleType type = new ScaleType(t);
			                               
			switch ((ScaleTypeValue)type){
			case ScaleTypeValue.MAJOR: 
				create_major (root, modeNo);
				break;
			case ScaleTypeValue.MELODIC_MINOR: 
				create_melodic_minor(root, modeNo);
				break;
			case ScaleTypeValue.HARMONIC_MINOR: 
				create_harmonic_minor(root, modeNo);
				break;
			case ScaleTypeValue.PENTATONIC:
				create_pentatonic(root, modeNo);
				break;
			case ScaleTypeValue.DOUBLE_HARMONIC:
				create_double_harmonic(root, modeNo);
				break;
			case ScaleTypeValue.CHROMATIC:
				create_chromatic(root, modeNo);
				break;
			case ScaleTypeValue.WHOLE_TONE:
				create_whole_tone(root, modeNo);
				break;
			case ScaleTypeValue.AUGMENTED:
				create_augmented(root, modeNo);
				break;
			case ScaleTypeValue.DIMINISHED:
				create_diminished(root, modeNo);
				break;
			case ScaleTypeValue.BLUES:
				create_blues(root, modeNo);
				break;
			}
	
			name = type.get_modeName(modeNo);
			set_name (name);
			this.type = type;
		}

		public Scale(Note root, int mode): this(root, new ScaleType(ScaleTypeValue.MAJOR), mode)
		{
			
		}

		public void to_mode (int modeNo)
		{
			int i;
			int noNotes;
			ObjGroup newNotes = new ObjGroup();
		 		
			noNotes = get_noNotes();
		
			for (i =  modeNo; i < noNotes; i++){
				newNotes.append(get_note( i));
			}
			for (i = 0; i < modeNo; i++){
				newNotes.append(get_note(i));
			}

			set_notes (newNotes);
		}

		public Interval get_interval(int index)
		{
			//interval 0 is the differnce between note 0 and 1
			Note noteA;
			Note noteB;
			Interval interval;

			noteA = get_note(index);
			noteB = get_note(index + 1);
	
			interval = new Interval(noteA, noteB);

			return interval;
		}

		public void set_notes (ObjGroup notes)
		{
			this.notes = notes;
		}

		public void set_notes (IList notes)
		{
			this.notes = new ObjGroup(notes);
		}
		
		public void set_note (Note note, int index)
		{
			notes.add_by_index(note,index);
		}
 
		public void append_note (Note note)
		{
			notes.append(note);
		}

		public Note get_note (int note)
		{
			return (Note)notes.nth(note);
		}

		public int get_noNotes ()
		{
			return notes.get_no();
		}

		public void delete_note (int noteNo)
		{
			notes.Remove(noteNo);
		}

		public void set_name(string name)
		{
			this.name = name;
		}

		public string get_name ()
		{
			return name;
		}

		public Note get_root()
		{
			return 	get_note(0);
		}

		public ScaleType get_scaleType()
		{
			return type;
		}
		
		public Scale rotate_to_note(Note note)
		{	
			//This is used by the canvas renderer to get a scale that fits nicely on a string
	
			int i = 0;
			int modeNo = 0;
			int thisDifference = 0;
			int leastDifference = 200;
			Scale rotatedScale = null;
		
			foreach (Note currentNote in notes){
				thisDifference = note.get_difference(currentNote);
				if (thisDifference < leastDifference){
					leastDifference = thisDifference;
					modeNo = i;
				}
				i++;
			}
	
			rotatedScale = new Scale(get_root(), 
			                         get_scaleType(),
			                         0);
			rotatedScale.to_mode(modeNo);
	
			return rotatedScale;
		}
	



		private void create_major (Note root, int modeNo)
		{
			//remember notes and modes start at 0

			Interval[] tempIntervals = new Interval[7];
			int i, j;
			Note currentNote;
        
			//adjust intervals for mode 
			for (i = 0, j =  modeNo; j < 7; i++, j++){
                tempIntervals[i] = majorIntervals[j];
			}
			for (j = 0 ; j < modeNo; j++, i++){
				tempIntervals[i] = majorIntervals[j];
			}

			//rember that root isn't real it is mode root
			currentNote = new Note(root); 
			append_note(new Note(currentNote));
			for (i = 0 ; i < 6 ; i++){
				currentNote.plus_scale_interval (tempIntervals[i]);
				append_note(new Note(currentNote));
			}
}

		private void create_melodic_minor(Note root, int modeNo)
		{
			//remember notes and modes start at 0 

			int noteNo;

			noteNo = 2 - modeNo;
			if (noteNo < 0){
				noteNo += 7;
			}
	
			//check to see if we need to compensate for the root being alterd
			if (noteNo == 0){
				root.sharpen();
			}

			create_major(root, modeNo);

			flatten_note (noteNo);
		}

		private void create_harmonic_minor(Note root, int modeNo)
		{
			//remember modes and notes start at 0

			int noteNo1, noteNo2;

			noteNo1 = 2  - modeNo;
			if (noteNo1 < 0){
				noteNo1 += 7;
			}

			noteNo2 = 5  - modeNo;
			if (noteNo2 < 0){
				noteNo2 += 7;
			}

			//check whether we have to compensate for the first note being alted
			if (noteNo1 == 0 || noteNo2 == 0){
				root.sharpen();
			}

			create_major(root, modeNo);

			flatten_note (noteNo1);
			flatten_note (noteNo2);
		}

		private void create_pentatonic(Note root, int modeNo)
		{
			//rember modes and notes start at 0

			int note1, note2;

			//convert modeNo to the major scale modeNo
			if(modeNo == 3){
				modeNo = 4;
			} else if (modeNo == 4){
				modeNo = 5;
			}

			create_major(root, modeNo);
	
			note1 = 3 - modeNo;
			if(note1 < 0){
				note1 += 7;
			}

			note2 = 6 - modeNo;
			if (note2 < 0){
				note2 += 7;
			}
	
			if (note2 > note1){
				delete_note (note2);
				delete_note (note1);
			} else {
				delete_note (note1);
				delete_note (note2);
			}
		}

		private void create_chromatic(Note root, int modeNo)
		{
			Note[] notes = new Note[12];

			create_major(root, modeNo);
	
			notes[0] = get_note(0);
			notes[1] = new Note(notes[0]);
			notes[1].sharpen();
			notes[2] = get_note(1);
			notes[3] = new Note(notes[2]);
			notes[3].sharpen();
			notes[4] = get_note(2);
			notes[5] = get_note(3);
			notes[6] = new Note(notes[5]);
			notes[6].sharpen();
			notes[7] = get_note(4);
			notes[8] = new Note(notes[7]);
			notes[8].sharpen();
			notes[9] = get_note(5);
			notes[10] = new Note(notes[9]);
			notes[10].sharpen();
			notes[11] = get_note(6);

			set_notes(notes);
		}

		private void create_blues (Note root, int modeNo)
		{
			//remember notes and modes start at 0

			Note flatThird;
			Note testRoot;
			
			if(modeNo != 0){
				//ignore NO_NOTE and UNKNOWN_NOTE
				foreach(NoteValue nv in Note.noteValues){
					testRoot = new Note(nv);
					if (testRoot == NoteValue.NO_NOTE
					    || testRoot == NoteValue.UNKNOWN
					    || testRoot.is_doubleFlat() 
					    || testRoot.is_doubleSharp()){
						continue;
					}
					create_blues(testRoot, 0);
					if(get_note(modeNo) == root){
						break;
					}
				}
				to_mode(modeNo);
			} else {
				create_pentatonic(root, 0);
				flatThird = new Note(get_note(2));
				flatThird.flatten();
				set_note(flatThird, 2);
			}
		}

		private void create_double_harmonic (Note root, int modeNo)
		{
			//rember modes and noteNo start at 0
	
			int noteNo1, noteNo2;
	 
			noteNo1 = 1  - modeNo;
			if (noteNo1 < 0){
				noteNo1 += 7;
			}

			noteNo2 = 5  - modeNo ;
			if (noteNo2 < 0){
				noteNo2 += 7;
			}

			//check whether we have to compensate for the first note being alted
			if (noteNo1 == 0 || noteNo2 == 0){
				root.sharpen();
			}

			create_major(root, modeNo);

			flatten_note (noteNo1);
			flatten_note (noteNo2);
		}

		private void create_whole_tone (Note root, int modeNo)
		{
	
			Note[] notes = new Note[6];
			int i ;

			
			notes[0] = root;
			append_note(root);
			for (i = 1; i < 6; i++){
				notes[i] = new Note(notes[i-1]);
				notes[i].plus_scale_tone();
			}
			set_notes(notes);
		}

		private void create_diminished (Note root, int modeNo)
		{
	
			Note[] notes = new Note[8];
			
			notes[0] = root;
			notes[1] = new Note(notes[0]);
			notes[1].plus_scale_tone();
			notes[2] = new Note(notes[1]);
			notes[2].plus_scale_tone();
			notes[3] = new Note(notes[2]);
			notes[3].plus_scale_tone();
			notes[4] = new Note(notes[3]);
			notes[4].plus_scale_tone();
			notes[5] = new Note(notes[4]);
			notes[5].plus_scale_tone();
			notes[6] = new Note(notes[5]);
			notes[6].plus_scale_tone();
			notes[7] = new Note(notes[6]);
			notes[7].plus_scale_tone();

			set_notes(notes);
		}

		private void create_augmented (Note root, int modeNo)
		{
	
			Note[] notes = new Note[6];
	
			notes[0] = root;
			notes[1] = new Note(notes[0]);
			notes[1].plus_interval(new Interval(IntervalValue.TONE_AND_HALF));
			notes[2] = new Note(notes[1]);
			notes[2].plus_interval(new Interval(IntervalValue.SEMI_TONE));
			notes[3] = new Note(notes[2]);
			notes[3].plus_interval(new Interval(IntervalValue.TONE_AND_HALF));
			notes[4] = new Note(notes[3]);
			notes[4].plus_interval(new Interval(IntervalValue.SEMI_TONE));
			notes[5] = new Note(notes[4]);
			notes[5].plus_interval(new Interval(IntervalValue.TONE_AND_HALF));
	
			set_notes(notes);
		}
	
		void flatten_note (int noteNo)
		{
			get_note(noteNo).flatten( );
		}
			
		public bool has_note(Note note)
		{
			
			foreach (Note currentNote in notes){
				if(note == currentNote){
					return true;
				}
			}
			return false;
		}
			
		public ObjGroup get_notes()
		{
			return notes;
		}
		
		public string to_text()
		{
			System.Text.StringBuilder text = new System.Text.StringBuilder(get_name());
			text.Append(" notes = ");
			foreach (Note note in notes){
				text.Append(note.to_text());
				text.Append(",");
			}
			return text.ToString();
		}
	}
}

