// -*- mode: js; js-indent-level: 4; indent-tabs-mode: nil -*-
/*
 * Copyright 2012 Red Hat, Inc.
 * Copyright 2012 Peng Huang <shawn.p.huang@gmail.com>
 * Copyright 2012 Takao Fujiwara <tfujiwar@redhat.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

const GLib = imports.gi.GLib;
const IBus = imports.gi.IBus;
const Lang = imports.lang;

const Config = imports.misc.config;

const ICON_KEYBOARD = 'input-keyboard-symbolic';
const XKB_LAYOUTS_MAX_LENGTH = 4;
const XKB_ENGINE_NAME_PREFIX = 'xkb:';
const XMODMAP_CMD = 'xmodmap';
const XMODMAP_KNOWN_FILES = ['.xmodmap', '.xmodmaprc', '.Xmodmap', '.Xmodmaprc'];


function XKBLayout(config, command) {
    this._init(config, command);
}

XKBLayout.prototype = {
    _init: function(config) {
        this._config = config;
        this._xkbCommand = 'setxkbmap';
        this._useXmodmap = true;
        this._xkbPid = -1;
        this._xmodmapPid = -1;

        this._defaultLayout = "";
        this._defaultVariant = "";
        this._defaultOption = "";

        this._xkbLatinLayouts = [];
        if (this._config != null) {
            let value = this._config.get_value('general',
                                               'xkb_latin_layouts',
                                               null);
            for (let i = 0; value != null && i < value.n_children(); i++) {
                this._xkbLatinLayouts.push(
                    value.get_child_value(i).dup_string()[0]);
            }
            if (this._useXmodmap) {
                this._useXmodmap = this._config.get_value(
                    'general',
                    'use_xmodmap',
                    GLib.Variant.new_boolean(true)).get_boolean();
            }
        }
    },

    /* _getVariantFromLayout:
     * @layout: The format is 'layouts(variants)[options]'
     * @returns: ['layouts[options]', 'variants']
     *
     * Return the array of layouts and variants from the formatted string.
     * Each element can be the comma separated values.
     */
    _getVariantFromLayout: function(layout) {
        let leftBracket = layout.indexOf('(');
        let rightBracket = layout.indexOf(')');
        if (leftBracket >= 0 && rightBracket > leftBracket) {
            return [layout.substring(0, leftBracket) + layout.substring(rightBracket + 1, layout.length),
                    layout.substring(leftBracket + 1, rightBracket)];
        }
        return [layout, 'default'];
    },

    /* _getOptionFromLayout:
     * @layout: The format is 'layouts[options]'
     * @returns: ['layouts', 'options']
     *
     * Return the array of layouts and options from the formatted string.
     * Each element can be the comma separated values.
     */
    _getOptionFromLayout: function(layout) {
        let leftBracket = layout.indexOf('[');
        let rightBracket = layout.indexOf(']');
        if (leftBracket >= 0 && rightBracket > leftBracket) {
            return [layout.substring(0, leftBracket) + layout.substring(rightBracket + 1, layout.length),
                    layout.substring(leftBracket + 1, rightBracket)];
        }
        return [layout, 'default'];
    },

    _getOutputFromCmdline: function(arg, str) {
        let retval = '';
        let argv = [this._xkbCommand, arg];
        let [result, output, std_err, status] = this._spawnWithPipes(argv);
        if (!result) {
            return '';
        }
        let lines = ('' + output).split('\n');
        for (let i = 0; i < lines.length; i++) {
            let line = lines[i];
            if (line.substring(0, str.length) == str) {
                retval = line.substring(str.length);
                if (retval == null) {
                    retval = '';
                } else {
                    retval = retval.trim();
                }
                break;
            }
        }
        return retval;
    },

    _getFullPath: function(command) {
        let paths = GLib.getenv('PATH');
        if (paths != null) {
            paths = paths.split(':');
        } else {
            paths = ['/usr/bin', '/bin'];
        }
        for (let i = 0; paths.length; i++) {
            let dir = paths[i];
            let filepath = dir + '/' + command;
            if (GLib.file_test(filepath, GLib.FileTest.EXISTS)) {
                return filepath;
            }
        }
        return null;
    },

    _getXkbGroupLayout: function(layout, variant, layoutsMaxLength) {
        let groupId = 0;
        let i = 0;
        let layouts = this._defaultLayout.split(',');
        let variants = this._defaultVariant.split(',');
        let groupLayouts = '';
        let groupVariants = '';
        let hasVariant = false;
        let includeKeyMap = false;

        for (i = 0; i < layouts.length; i++) {
            if (i >= layoutsMaxLength - 1) {
                break;
            }

            if (i == 0) {
                groupLayouts = layouts[i];
            } else {
                groupLayouts = groupLayouts + ',' + layouts[i];
            }

            if (i >= variants.length) {
                if (i == 0) {
                    groupVariants = '';
                } else {
                    groupVariants += ',';
                }
                if (layout == layouts[i] && variant == '') {
                    includeKeyMap = true;
                    groupId = i;
                }
                continue;
            }
            if (layout == layouts[i] && variant == variants[i]) {
                includeKeyMap = true;
                groupId = i;
            }

            if (variants[i] != '') {
                hasVariant = true;
            }

            if (i == 0) {
                groupVariants = variants[i];
            } else {
                groupVariants = groupVariants + ',' + variants[i];
            }
        }

        if (variant != '') {
            hasVariant = true;
        }

        if (!includeKeyMap) {
            groupLayouts = groupLayouts + ',' + layout;
            groupVariants = groupVariants + ',' + variant;
            groupId = i;
        }

        if (!hasVariant) {
            groupVariants = null;
        }
        return [groupLayouts, groupVariants, groupId];
    },

    _setLayoutCB: function(pid, status, data) {
        if (this._xkbPid != pid) {
            log('XkbLayout.setLayout has another pid.');
            return;
        }
        this._xkbPid = -1;
        this.setXmodmap();
    },

    _setXmodmapCB: function(pid, status, data) {
        if (this._xmodmapPid != pid) {
            log('XkbLayout.setXmodmap has another pid.');
            return;
        }
        this._xmodmapPid = -1;
    },

    _trySpawnWithPipes: function(argv) {
        let retval = [false, null, null, -1];

        try {
            retval = GLib.spawn_sync(null, argv, null,
                                     GLib.SpawnFlags.SEARCH_PATH,
                                     null, null);
        } catch (err) {
            if (err.code == GLib.SpawnError.G_SPAWN_ERROR_NOENT) {
                err.message = _("Command not found");
            } else {
                // The exception from gjs contains an error string like:
                //   Error invoking GLib.spawn_command_line_async: Failed to
                //   execute child process 'foo' (No such file or directory)
                // We are only interested in the part in the parentheses. (And
                // we can't pattern match the text, since it gets localized.)
                err.message = err.message.replace(/.*\((.+)\)/, '$1');
            }

            throw err;
        }
        return retval;
    },

    _trySpawnAsyncXkb: function(argv) {
        let retval = false;
        let pid = -1;

        try {
            [retval, pid] = GLib.spawn_async(null, argv, null,
                                      GLib.SpawnFlags.SEARCH_PATH |
                                      GLib.SpawnFlags.DO_NOT_REAP_CHILD,
                                      null, null);
            this._xkbPid = pid;
            GLib.child_watch_add(0, this._xkbPid,
                                 Lang.bind(this, this._setLayoutCB),
                                 null);
        } catch (err) {
            if (err.code == GLib.SpawnError.G_SPAWN_ERROR_NOENT) {
                err.message = _("Command not found");
            } else {
                // The exception from gjs contains an error string like:
                //   Error invoking GLib.spawn_command_line_async: Failed to
                //   execute child process 'foo' (No such file or directory)
                // We are only interested in the part in the parentheses. (And
                // we can't pattern match the text, since it gets localized.)
                err.message = err.message.replace(/.*\((.+)\)/, '$1');
            }

            throw err;
        }
        return retval;
    },

    _trySpawnAsyncXmodmap: function(argv) {
        let retval = false;
        let pid = -1;

        try {
            [retval, pid] = GLib.spawn_async(null, argv, null,
                                      GLib.SpawnFlags.SEARCH_PATH |
                                      GLib.SpawnFlags.DO_NOT_REAP_CHILD,
                                      null, null);
            this._xmodmapPid = pid;
            GLib.child_watch_add(0, this._xmodmapPid,
                                 Lang.bind(this, this._setXmodmapCB),
                                 null);
        } catch (err) {
            if (err.code == GLib.SpawnError.G_SPAWN_ERROR_NOENT) {
                err.message = _("Command not found");
            } else {
                // The exception from gjs contains an error string like:
                //   Error invoking GLib.spawn_command_line_async: Failed to
                //   execute child process 'foo' (No such file or directory)
                // We are only interested in the part in the parentheses. (And
                // we can't pattern match the text, since it gets localized.)
                err.message = err.message.replace(/.*\((.+)\)/, '$1');
            }

            throw err;
        }
        return retval;
    },

    _handleSpawnError: function(command, err) {
        let title = _("Execution of '%s' failed:").format(command);
        log(title);
        log(err.message);
    },

    _spawnWithPipes: function(argv) {
        try {
            return this._trySpawnWithPipes(argv);
        } catch (err) {
            this._handleSpawnError(argv[0], err);
            return [false, null, err.message, -1];
        }
    },

    _spawnAsyncXkb: function(argv) {
        try {
            return this._trySpawnAsyncXkb(argv);
        } catch (err) {
            this._handleSpawnError(argv[0], err);
            return false;
        }
    },

    _spawnAsyncXmodmap: function(argv) {
        try {
            return this._trySpawnAsyncXmodmap(argv);
        } catch (err) {
            this._handleSpawnError(argv[0], err);
            return false;
        }
    },

    getLayout: function() {
        return this._getOutputFromCmdline('-query', 'layout: ');
    },

    getVariant: function() {
        return this._getOutputFromCmdline('-query', 'variant: ');
    },

    getOption: function() {
        return this._getOutputFromCmdline('-query', 'options: ');
    },

    setLayout: function(engine) {
        let layout = engine.get_layout();
        let variant = engine.get_layout_variant();
        let option = engine.get_layout_option();
        let xkbGroupId = 0;
        let changedOption = false;

        if (this._xkbPid != -1) {
            return [-1, false];
        }
        if (layout == 'default' &&
            (variant == 'default' || variant == '') &&
            (option == 'default' || option == '')) {
            return [-1, false];
        }

        let needUsLayout = false;
        for (let i = 0; i < this._xkbLatinLayouts.length; i ++) {
            let latinLayout = this._xkbLatinLayouts[i];
            // layout 'in' and variant 'eng' is English layout.
            if (layout == latinLayout && variant != 'eng') {
                needUsLayout = true;
                break;
            }

            if (variant != null && layout + '(' + variant + ')' == latinLayout) {
                needUsLayout = true;
                break;
            }
        }

        let layoutsMaxLength = XKB_LAYOUTS_MAX_LENGTH;
        if (needUsLayout) {
            layoutsMaxLength--;
        }

        if (this._defaultLayout == '') {
            this._defaultLayout = this.getLayout();
        }
        if (this._defaultVariant == '') {
            this._defaultVariant = this.getVariant();
        }
        if (this._defaultOption == '') {
            this._defaultOption = this.getOption();
        }

        if (layout == 'default') {
            layout = this._defaultLayout;
            variant = this._defaultVariant;
        } else {
            if (variant == 'default') {
                variant = '';
            }
            [layout, variant, xkbGroupId] = this._getXkbGroupLayout(layout,
                                                                    variant,
                                                                    layoutsMaxLength);
        }

        if (layout == '') {
            global.log('Could not get the correct layout');
            return [-1, false];
        }

        if (variant == 'default' || variant == '') {
            variant = null;
        }

        if (option == 'default' || option == '') {
            option = this._defaultOption;
        } else {
            let includeOption = false;
            for (let j = 0; j < this._defaultOption.length; j++) {
                if (option == this._defaultOption[j]) {
                    includeOption = true;
                    break;
                }
            }

            if (!includeOption) {
                option = this._defaultOption + ',' + option;
                changedOption = true;
            } else {
                option = this._defaultOption;
            }
        }

        if (option == '') {
            option = null;
        }

        if (needUsLayout) {
            layout = layout + ',us';
            if (variant != null) {
                variant = variant + ',';
            }
        }

        let args = [];
        args.push(this._xkbCommand);
        args.push('-layout');
        args.push(layout);
        if (variant != null) {
            args.push('-variant');
            args.push(variant);
        }
        if (option != null) {
            /* TODO: Need to get the session XKB options */
            args.push('-option');
            args.push('-option');
            args.push(option);
        }

        this._spawnAsyncXkb(args);
        return [xkbGroupId, changedOption];
    },

    resetLayout: function() {
        this._defaultLayout = this.getLayout();
        this._defaultVariant = this.getVariant();
        this._defaultOption = this.getOption();
    },

    setXmodmap: function() {
        if (!this._useXmodmap) {
            return;
        }

        if (this._xmodmapPid != -1) {
            return;
        }

        let xmodmapCmdPath = this._getFullPath(XMODMAP_CMD);
        if (xmodmapCmdPath == null) {
            xmodmapCmdPath = XMODMAP_CMD;
        }

        for (let i = 0; i < XMODMAP_KNOWN_FILES.length; i++) {
            let xmodmapFile = XMODMAP_KNOWN_FILES[i];
            let xmodmapFilePath = GLib.get_home_dir() + '/' + xmodmapFile;
            if (!GLib.file_test(xmodmapFilePath, GLib.FileTest.EXISTS)) {
                continue;
            }

            let args = [];
            args.push(xmodmapCmdPath);
            args.push(xmodmapFilePath);
            this._spawnAsyncXmodmap(args);
            break;
        }
    }
};

function engineDescNew(lang, layout, layoutDesc,
                       variant, variantDesc,
                       name) {
    let longname = layout;
    let desc = null;
    let engineLayout = null;
    let engine = null;

    if (variantDesc != null) {
        longname = variantDesc;
    } else if (layout != null && variant != null) {
        longname = layout + ' - ' + variant;
    } else if (layoutDesc != null) {
        longname = layoutDesc;
    }
    let name_prefix = 'xkb:layout:';
    if (variant != null) {
        if (name == null) {
            name = name_prefix + layout + ':' + variant;
        }
        desc = 'XKB ' + layout + '(' + variant + ') keyboard layout';
        engineLayout = layout + '(' + variant + ')';
    } else {
        if (name == null) {
            name = name_prefix + layout;
        }
        desc = 'XKB ' + layout + ' keyboard layout';
        engineLayout = layout;
    }

    let icon = 'ibus-engine';
    if (name.substring(0, XKB_ENGINE_NAME_PREFIX.length)
        == XKB_ENGINE_NAME_PREFIX) {
        icon = ICON_KEYBOARD;
    }

    engine = new IBus.EngineDesc({ name: name,
                                   longname: longname,
                                   description: desc,
                                   language: lang,
                                   license: 'GPL2',
                                   author: 'Takao Fujiwara <takao.fujiwara1@gmail.com>',
                                   icon: icon,
                                   layout: engineLayout });
    return engine;
}
