/***************************************************************************
 *   Copyright (C) 2005 by SUZUKI Tasuku                                   *
 *   tasuku@linux-life.net                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "kkdtextbase.h"

#include <kiconloader.h>
#include <klocale.h>
#include <kdebug.h>

#include <qregexp.h>

using namespace KKDesigner;

KKDTextBase::KKDTextBase( const QPoint& l
						, const QSize& s
						, const QFont& f
						, const QColor& c
						, Orientation o
						, QObject* parent
						)
	: KKDItemBase( l.x(), l.y(), s.width(), s.height(), parent )
	, m_font( f )
	, m_color( c )
	, m_orientation( o )
{
}

KKDTextBase::KKDTextBase( QDomElement e, QObject* parent )
	: KKDItemBase( e, parent )
	, m_font()
	, m_color( e.attribute( "color" ) )
	, m_orientation( (Orientation)e.attribute( "orientation" ).toInt() )
{
	m_font.fromString( e.attribute( "font" ) );
}

KKDTextBase::KKDTextBase( const KKDTextBase* item, QObject* parent )
	: KKDItemBase( (KKDItemBase*)item, parent )
	, m_font( item->m_font )
	, m_color( item->m_color )
	, m_orientation( item->m_orientation )
{
}

KKDTextBase::~KKDTextBase()
{
}

//BEGIN Properties
void KKDTextBase::setFontFamily( const QString& f )
{
	if( locked() ) return;
	if( m_font.family() == f ) return;
	QRect r = rect();
	m_font.setFamily( f );
	resize();
	emit repaint( r|rect() );
	emit fontFamilyChanged();
}

void KKDTextBase::setFontSize( int s )
{
	if( locked() ) return;
	if( m_font.pointSize() == s ) return;
	QRect r = rect();
	m_font.setPointSize( s );
	resize();
	emit repaint( r|rect() );
	emit fontSizeChanged();
}

void KKDTextBase::setFontBold( bool b )
{
	if( locked() ) return;
	if( m_font.bold() == b ) return;
	QRect r = rect();
	m_font.setBold( b );
	resize();
	emit repaint( r|rect() );
	emit fontBoldChanged();
}

void KKDTextBase::setFontItalic( bool i )
{
	if( locked() ) return;
	if( m_font.italic() == i ) return;
	QRect r = rect();
	m_font.setItalic( i );
	resize();
	emit repaint( r|rect() );
	emit fontItalicChanged();
}

void KKDTextBase::setFontUnderline( bool u )
{
	if( locked() ) return;
	if( m_font.underline() == u ) return;
	QRect r = rect();
	m_font.setUnderline( u );
	resize();
	emit repaint( r|rect() );
	emit fontUnderlineChanged();
}

void KKDTextBase::setFontStrikeOut( bool s )
{
	if( locked() ) return;
	if( m_font.strikeOut() == s ) return;
	QRect r = rect();
	m_font.setStrikeOut( s );
	resize();
	emit repaint( r|rect() );
	emit fontStrikeOutChanged();
}

void KKDTextBase::setColor( const QColor& c )
{
	if( locked() ) return;
	if( m_color == c ) return;
	m_color = c;
	emit repaint( rect() );
	emit colorChanged();
}

void KKDTextBase::setOrientation( Orientation o )
{
	if( locked() ) return;
	if( m_orientation == o ) return;
	QRect r = rect();
	m_orientation = o;
	resize();
	emit repaint( r|rect() );
	emit orientationChanged();
}

//END   Properties

QDomElement KKDTextBase::toElement( QDomDocument& doc, const QString& t )
{
	QString tag( t == QString::null ? tagName() : t );
	QDomElement elem = KKDItemBase::toElement( doc, tag );
	elem.setAttribute( "font", m_font.toString() );
	elem.setAttribute( "color", m_color.name() );
	elem.setAttribute( "orientation", (int)m_orientation );
	return elem;
}

void KKDTextBase::drawItemBefore( QRect& r, QPainter& p, DrawMode m, const KKAddressBook::KKAContactData* d )
{
	KKDItemBase::drawItemBefore( r, p, m, d );
	p.setFont( m_font );
	p.setPen( m_color );
}

void KKDTextBase::drawItemAfter( QRect& r, QPainter& p, DrawMode m, const KKAddressBook::KKAContactData* d )
{
	KKDItemBase::drawItemAfter( r, p, m, d );
}

QString KKDTextBase::toVertical( const QString& str )
{
	QStringList line = QStringList::split( QString(""), str, true );
	line.pop_front();
	line.pop_back();
	QString tmp = line.join( "\n" );

	QRegExp rx( "([0-9])\n([0-9])" );
	while( rx.search( tmp ) != -1 )
	{
		tmp.replace( rx, "\\1\\2" );
	}
	tmp.replace( QString::fromUtf8( "ー" ), QString::fromUtf8( "｜" ) );
	tmp.replace( "-", QString::fromUtf8( "｜" ) );
	return tmp;
}

QSize KKDTextBase::calcSize( const QString& str, int size_decrease )
{
	QSize s;
	QFont f( font() );
	f.setPointSize( f.pointSize() - size_decrease );
	QFontMetrics met( f );

	switch( orientation() )
	{
		case Horizontal:
		{
			QStringList line = QStringList::split( "\n", str, true );
			for( QStringList::Iterator it = line.begin(); it != line.end(); ++it )
			{
				if( s.width() < met.width( *it ) )
				{
					s.setWidth( met.width( *it ) );
				}
				s.setHeight( s.height() + met.height() );
			}
// 			s.setWidth( s.width() + met.width( " " ) );
// 			s.setHeight( s.height() + met.width( " " ) );
			break;
		}
		case Vertical:
		{
			QStringList line = QStringList::split( "\n", str, true );
			for( QStringList::Iterator it = line.begin(); it != line.end(); ++it )
			{
				QSize sz;
				QStringList l = QStringList::split( "\n", toVertical( *it ) );
				for( QStringList::Iterator i = l.begin(); i != l.end(); ++i )
				{
					if( sz.width() < met.width( *i ) * 1.2 )
					{
						sz.setWidth( met.width( *i ) * 1.2 );
					}
					sz.setHeight( sz.height() + met.height() + 1 );
				}
				s.setWidth( s.width() + sz.width() );
				if( s.height() < sz.height() ) s.setHeight( sz.height() );
			}
			s.setWidth( s.width() );
// 			s.setHeight( s.height() + met.width( " " ) );
			break;
		}
	}
	return s;
}

void KKDTextBase::drawText( QRect& r, QPainter& p, AlignmentFlags align, const QString& str, int size_decrease )
{
// 	p.drawRect( r );
	QFont f( font() );
	f.setPointSize( f.pointSize() - size_decrease );
	p.setFont( f );
	switch( orientation() )
	{
		case Horizontal:
			p.drawText( r, align | AlignVCenter, str );
			break;
		case Vertical:
		{
			QStringList line = QStringList::split( "\n", str );
			int width = calcSize( str, size_decrease ).width();
			QRect rct( r );
			rct.setRight( rct.right() - ( rct.width() - width ) / 2 );
			for( QStringList::Iterator it = line.begin(); it != line.end(); ++it )
			{
				int w = calcSize( *it, size_decrease ).width();
				rct.setLeft( rct.right() - w + 1 );
				if( r.contains( rct ) )
				{
					p.drawText( rct, align | AlignHCenter, toVertical( *it ) );
				}
				else
				{
/*					kdDebug() << "Undrawable item info==============" << endl;
					kdDebug() << "r = " << r << endl;
					kdDebug() << "rct = " << rct << endl;
					kdDebug() << "str = " << str << endl;*/
				}
				rct.moveRight( rct.right() - w );
			}
		}
	}
}
