/*
 * Galago Core API
 *
 * Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_CORE_H_
#define _GALAGO_CORE_H_

#include <libgalago/galago-account.h>
#include <libgalago/galago-dbus.h>
#include <libgalago/galago-person.h>
#include <libgalago/galago-service.h>

#define GALAGO_DBUS_CORE_INTERFACE "org.freedesktop.Galago.Core"
#define GALAGO_DBUS_CORE_OBJECT    "/org/freedesktop/Galago/Core"

G_BEGIN_DECLS

#define GALAGO_TYPE_CORE (galago_core_get_type())
#define GALAGO_CORE(obj) \
		(G_TYPE_CHECK_INSTANCE_CAST((obj), GALAGO_TYPE_CORE, \
									GalagoCore))
#define GALAGO_CORE_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_CAST((klass), GALAGO_TYPE_CORE, \
								 GalagoCoreClass))
#define GALAGO_IS_CORE(obj) \
		(G_TYPE_CHECK_INSTANCE_TYPE((obj), GALAGO_TYPE_CORE))
#define GALAGO_IS_CORE_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_TYPE((klass), GALAGO_TYPE_CORE))
#define GALAGO_CORE_GET_CLASS(obj) \
		(G_TYPE_INSTANCE_GET_CLASS((obj), GALAGO_TYPE_CORE, \
								   GalagoCoreClass))

typedef struct _GalagoCore        GalagoCore;
typedef struct _GalagoCoreClass   GalagoCoreClass;
typedef struct _GalagoCorePrivate GalagoCorePrivate;

typedef enum
{
	GALAGO_INIT_CLIENT      = 0,
	GALAGO_INIT_FEED        = 1 << 0,
	GALAGO_INIT_NO_ACTIVATION = 1 << 1

} GalagoInitFlags;


struct _GalagoCore
{
	GalagoObject parent_object;

	GalagoCorePrivate *priv;
};

struct _GalagoCoreClass
{
	GalagoObjectClass parent_class;

	/* Signals */
	void (*registered)(GalagoCore *core);
	void (*unregistered)(GalagoCore *core);
	void (*service_added)(GalagoCore *core, GalagoService *service);
	void (*service_removed)(GalagoCore *core, GalagoService *service);
	void (*person_added)(GalagoCore *core, GalagoPerson *person);
	void (*person_removed)(GalagoCore *core, GalagoPerson *person);
	GalagoAccount *(*calc_priority_account)(GalagoCore *core,
											GalagoPerson *person);

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

GType galago_core_get_type(void);

gboolean galago_init(const char *name, GalagoInitFlags flags);
void galago_uninit(void);

gboolean galago_is_initted(void);
gboolean galago_is_connected(void);
gboolean galago_is_registered(void);
gboolean galago_is_daemon_active(void);
gboolean galago_is_daemon(void);
gboolean galago_is_feed(void);

DBusConnection *galago_get_dbus_conn(void);

const char *galago_get_uid(void);
const char *galago_get_client_obj_path(void);

void galago_set_watch_all(gboolean watch_all);
gboolean galago_get_watch_all(void);

GalagoService *galago_create_service(const char *id, const char *name,
									 GalagoServiceFlags flags);
GalagoService *galago_get_service(const char *id, GalagoOrigin origin,
								  gboolean query);
GList *galago_get_services(GalagoOrigin origin, gboolean query);

GalagoPerson *galago_create_person(const char *uid);
GalagoPerson *galago_get_person(const char *uid, GalagoOrigin origin,
								gboolean query);
GList *galago_get_people(GalagoOrigin origin, gboolean query);
GalagoPerson *galago_get_me(GalagoOrigin origin, gboolean query);

GalagoCore *galago_get_core(void);

G_END_DECLS

#endif /* _GALAGO_CORE_H_ */
