/*
 *  Copyright (C) 2004 UCHINO Satoshi.  All Rights Reserved.
 *  Copyright (C) 2003 Tungsten Graphics, Inc. All Rights Reserved.
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

/*
 * rfbproto.c - functions to deal with client side of RFB protocol.
 *
 * Modular loader donated by Tungsten Graphics, Inc.
 * See http://www.tungstengraphics.com for more information
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <pwd.h>
#include <windowmonitor.h>
#include <vncauth.h>
#include <zlib.h>

#include "rfbmodule.h"
typedef Bool (*encoderFunc_t)(void *msg);
encoderFunc_t encoderFunc[MAX_ENCODINGS];
_vnc_module_close_t _vnc_module_close[MAX_ENCODINGS];
void *_vnc_module_libs[MAX_ENCODINGS];
int encoderType[MAX_ENCODINGS];
int encoders = 0;

int rfbsock;
char *desktopName;
rfbPixelFormat myFormat;
rfbServerInitMsg si;
char *textBuffer = NULL;
int textBufferSize = 0;

int endianTest = 1;

CapsContainer *tunnelCaps;	/* known tunneling/encryption methods */
CapsContainer *authCaps;	/* known authentication schemes       */
CapsContainer *serverMsgCaps;	/* known non-standard server messages */
CapsContainer *clientMsgCaps;	/* known non-standard client messages */
CapsContainer *encodingCaps;	/* known encodings besides Raw        */

static Bool SetupTunneling(void);
static Bool PerformAuthenticationNew(void);
static Bool PerformAuthenticationOld(void);
static Bool AuthenticateVNC(void);
static Bool ReadInteractionCaps(void);
static Bool ReadCapabilityList(CapsContainer *caps, int count);
static void ReadConnFailedReason(void);

/* Note that the CoRRE encoding uses this buffer and assumes it is big enough
   to hold 255 * 255 * 32 bits -> 260100 bytes.  640*480 = 307200 bytes.
   Hextile also assumes it is big enough to hold 16 * 16 * 32 bits.
   Tight encoding assumes BUFFER_SIZE is at least 16384 bytes. */

/* Raw also uses this buffer */

#define BUFFER_SIZE (640*480)
char buffer[BUFFER_SIZE];

/*
 * ConnectToRFBServer.
 */

Bool
ConnectToRFBServer(const char *hostname, int port)
{
  unsigned int host;

  if (!StringToIPAddr(hostname, &host)) {
    fprintf(stderr,"Couldn't convert '%s' to host address\n", hostname);
    return False;
  }

  rfbsock = ConnectToTcpAddr(host, port);

  if (rfbsock < 0) {
    fprintf(stderr,"Unable to connect to VNC server\n");
    return False;
  }

  return SetNonBlocking(rfbsock);
}


/*
 * InitialiseRFBConnection.
 */

Bool
InitialiseRFBConnection(void)
{
  rfbProtocolVersionMsg pv;
  int server_major, server_minor;
  int viewer_major, viewer_minor;
  rfbClientInitMsg ci;

  /* if the connection is immediately closed, don't report anything, so
       that pmw's monitor can make test connections */

  if (!ReadFromRFBServer(pv, sz_rfbProtocolVersionMsg)) return False;

  errorMessageOnReadFailure = True;

  pv[sz_rfbProtocolVersionMsg] = 0;

  if (sscanf(pv,rfbProtocolVersionFormat,&server_major,&server_minor) != 2) {
    fprintf(stderr,"Not a valid VNC server\n");
    return False;
  }

  viewer_major = rfbProtocolMajorVersion;
  viewer_minor = rfbProtocolFallbackMinorVersion;/* use older protocol */

  fprintf(stderr, "Connected to RFB server, using protocol version %d.%d\n",
	  viewer_major, viewer_minor);

  sprintf(pv, rfbProtocolVersionFormat, viewer_major, viewer_minor);
  
  if (!WriteExact(rfbsock, pv, sz_rfbProtocolVersionMsg))
    return False;
  
  if (!PerformAuthenticationOld()) /* authentication in protocol 3.3 */
    return False;

  ci.shared = 1;

  if (!WriteExact(rfbsock, (char *)&ci, sz_rfbClientInitMsg))
    return False;

  if (!ReadFromRFBServer((char *)&si, sz_rfbServerInitMsg))
    return False;

  si.framebufferWidth = Swap16IfLE(si.framebufferWidth);
  si.framebufferHeight = Swap16IfLE(si.framebufferHeight);
  si.format.redMax = Swap16IfLE(si.format.redMax);
  si.format.greenMax = Swap16IfLE(si.format.greenMax);
  si.format.blueMax = Swap16IfLE(si.format.blueMax);
  si.nameLength = Swap32IfLE(si.nameLength);

  desktopName = malloc(si.nameLength + 1);
  if (!desktopName) {
    fprintf(stderr, "Error allocating memory for desktop name, %lu bytes\n",
            (unsigned long)si.nameLength);
    return False;
  }

  if (!ReadFromRFBServer(desktopName, si.nameLength)) return False;

  desktopName[si.nameLength] = 0;

  fprintf(stderr,"Desktop name \"%s\"\n",desktopName);

  return True;
}


/*
 * Negotiate authentication scheme (protocol version 3.3)
 */

static Bool
PerformAuthenticationOld(void)
{
  CARD32 authScheme;

  /* Read the authentication type */
  if (!ReadFromRFBServer((char *)&authScheme, 4))
    return False;

  authScheme = Swap32IfLE(authScheme);

  switch (authScheme) {
  case rfbSecTypeInvalid:
    ReadConnFailedReason();
    return False;
  case rfbSecTypeNone:
    fprintf(stderr, "No authentication needed\n");
    break;
  case rfbSecTypeVncAuth:
    if (!AuthenticateVNC())
      return False;
    break;
  default:
    fprintf(stderr, "Unknown authentication scheme from VNC server: %d\n",
	    (int)authScheme);
    return False;
  }

  return True;
}


/*
 * Standard VNC authentication.
 */

static Bool
AuthenticateVNC(void)
{
  CARD32 authScheme, authResult;
  CARD8 challenge[CHALLENGESIZE];
  char *passwd;

  fprintf(stderr, "Performing standard VNC authentication\n");

  if (!ReadFromRFBServer((char *)challenge, CHALLENGESIZE))
    return False;

  if (appData.passwordFile) {
    passwd = vncDecryptPasswdFromFile(appData.passwordFile);
    if (!passwd) {
      fprintf(stderr, "Cannot read valid password from file \"%s\"\n",
	      appData.passwordFile);
      return False;
    }
  } else {
    passwd = getpass("Password: ");
  }

  if (!passwd || strlen(passwd) == 0) {
    fprintf(stderr, "Reading password failed\n");
    return False;
  }
  if (strlen(passwd) > 8) {
    passwd[8] = '\0';
  }

  vncEncryptBytes(challenge, passwd);

  /* Lose the password from memory */
  memset(passwd, '\0', strlen(passwd));

  if (!WriteExact(rfbsock, (char *)challenge, CHALLENGESIZE))
    return False;

  if (!ReadFromRFBServer((char *)&authResult, 4))
    return False;

  authResult = Swap32IfLE(authResult);

  switch (authResult) {
  case rfbVncAuthOK:
    fprintf(stderr, "VNC authentication succeeded\n");
    break;
  case rfbVncAuthFailed:
    fprintf(stderr, "VNC authentication failed\n");
    return False;
  case rfbVncAuthTooMany:
    fprintf(stderr, "VNC authentication failed - too many tries\n");
    return False;
  default:
    fprintf(stderr, "Unknown VNC authentication result: %d\n",
 	    (int)authResult);
    return False;
  }

  return True;
}

Bool
SetFormatAndEncodings()
{
  rfbSetPixelFormatMsg spf;
  char buf[sz_rfbSetEncodingsMsg + MAX_ENCODINGS * 4];
  rfbSetEncodingsMsg *se = (rfbSetEncodingsMsg *)buf;
  CARD32 *encs = (CARD32 *)(&buf[sz_rfbSetEncodingsMsg]);
  int len = 0;
  Bool requestCompressLevel = False;
  Bool requestQualityLevel = False;
  Bool requestLastRectEncoding = False;

  memset(buf, 0, sz_rfbSetEncodingsMsg + MAX_ENCODINGS * 4);

  spf.type = rfbSetPixelFormat;
  spf.format = myFormat;
  spf.format.redMax = Swap16IfLE(spf.format.redMax);
  spf.format.greenMax = Swap16IfLE(spf.format.greenMax);
  spf.format.blueMax = Swap16IfLE(spf.format.blueMax);

  if (!WriteExact(rfbsock, (char *)&spf, sz_rfbSetPixelFormatMsg))
    return False;

  se->type = rfbSetEncodings;
  se->nEncodings = 0;

  len = sz_rfbSetEncodingsMsg + se->nEncodings * 4;

  se->nEncodings = Swap16IfLE(se->nEncodings);

  if (!WriteExact(rfbsock, buf, len)) return False;

  return True;
}


/*
 * PrintPixelFormat.
 */

static void
ReadConnFailedReason(void)
{
  CARD32 reasonLen;
  char *reason = NULL;

  if (ReadFromRFBServer((char *)&reasonLen, sizeof(reasonLen))) {
    reasonLen = Swap32IfLE(reasonLen);
    if ((reason = malloc(reasonLen)) != NULL &&
        ReadFromRFBServer(reason, reasonLen)) {
      fprintf(stderr,"VNC connection failed: %.*s\n", (int)reasonLen, reason);
      free(reason);
      return;
    }
  }

  fprintf(stderr, "VNC connection failed\n");

  if (reason != NULL)
    free(reason);
}

/*
 * SendClientCutText.
 */

Bool
SendClientCutText(const char *str, int len)
{
  rfbClientCutTextMsg *cct;
  char *buf;
  int buflen = sz_rfbClientCutTextMsg + len;
  Bool result;

  buf = (char *)malloc(buflen);
  if (!buf)
    return False;

  cct = (rfbClientCutTextMsg *)buf;
  cct->type = rfbClientCutText;
  cct->length = Swap32IfLE(len);

  if (len > 0)
    memcpy(buf + sz_rfbClientCutTextMsg, str, len);
  result = WriteExact(rfbsock, buf, buflen);
  free(buf);
  return result;
}

/*
 * SendWindowState
 * (windowname can be null if name is unchanged since last update)
 */

Bool
SendWindowState(unsigned long id, int state, const char *windowname)
{
  rfbWindowStateMsg *wsm;
  int namelen;
  char *buf;
  int buflen;
  Bool result;

  namelen = (windowname ? strlen(windowname) : 0);
  buflen = sz_rfbWindowStateMsg + namelen;

  buf = (char *)malloc(buflen);
  if (!buf)
    return False;

  wsm = (rfbWindowStateMsg *)buf;
  wsm->type = rfbWindowState;
  wsm->state = Swap16IfLE(state);
  wsm->id = Swap32IfLE(id);
  wsm->nameLength = namelen;

  if (namelen > 0)
    memcpy(buf + sz_rfbWindowStateMsg, windowname, namelen);
  result = WriteExact(rfbsock, buf, buflen);
  free(buf);
  return result;
}

/*
 * SendServerData
 */

Bool
SendServerMenuItemPath(unsigned long id, unsigned long clientPtr,
		       unsigned long pathLength, const char *path)
{
  rfbServerDataMsg *sdm;
  rfbServerDataFilePath *sdp;
  char *buf;
  int buflen;
  struct stat file_stat;
  Bool result;

  buflen = sz_rfbServerDataMsg_WM + sz_rfbServerDataFilePath + pathLength;
  buf = (char *)malloc(buflen);
  if (!buf) {
    fprintf(stderr,"%s: memory allocation failed: (%d bytes)\n", __FUNCTION__, buflen);
    return False;
  }
  if (!(stat(path, &file_stat) == 0 &&
	S_ISREG(file_stat.st_mode) && ((file_stat.st_mode & 0444) == 0444))) {
    fprintf(stderr,"%s: %s is not a regular file or does not have read permission\n", __FUNCTION__, path);
    return False;
  }

  sdm = (rfbServerDataMsg *)buf;
  sdm->type       = rfbServerData;
  sdm->dataType   = Swap16IfLE(rfbServerDataMenuItemPath);
  sdm->id         = Swap32IfLE(id);
  sdm->dataLength = Swap32IfLE(sz_rfbServerDataFilePath + pathLength);
  sdm->clientPtr  = clientPtr;

  sdp = (rfbServerDataFilePath *)(buf + sz_rfbServerDataMsg_WM);
  sdp->fileLength = Swap32IfLE(file_stat.st_size);

  if (pathLength > 0)
    memcpy(sdp->filePath, path, pathLength);
  result = WriteExact(rfbsock, buf, buflen);
  free(buf);
  return result;
}

Bool
SendServerIconPath(unsigned long id, int dataType, unsigned long clientPtr,
		   int bpp, int size, int formatType,
		   unsigned long iconPathLength, const char *iconPath)
{
  rfbServerDataMsg *sdm;
  rfbServerDataIconPath *sdi;
  char *buf;
  int buflen;
  struct stat file_stat;
  Bool result;

  if (dataType != rfbServerDataMenuIconPath &&
      dataType != rfbServerDataWindowIconPath)
    return False;

  buflen = sz_rfbServerDataMsg_WM + sz_rfbServerDataIconPath + iconPathLength;
  buf = (char *)malloc(buflen);
  if (!buf) {
    fprintf(stderr,"%s: memory allocation failed: (%d bytes)\n", __FUNCTION__, buflen);
    return False;
  }
  if (!(stat(iconPath, &file_stat) == 0 &&
	S_ISREG(file_stat.st_mode) && ((file_stat.st_mode & 0444) == 0444))) {
    fprintf(stderr,"%s: %s is not a regular file or does not have read permission\n", __FUNCTION__, iconPath);
    return False;
  }

  sdm = (rfbServerDataMsg *)buf;
  sdm->type       = rfbServerData;
  sdm->dataType   = Swap16IfLE(dataType);
  sdm->id         = Swap32IfLE(id);
  sdm->dataLength = Swap32IfLE(sz_rfbServerDataIconPath + iconPathLength);
  sdm->clientPtr  = clientPtr;

  sdi = (rfbServerDataIconPath *)(buf + sz_rfbServerDataMsg_WM);
  sdi->bpp        = bpp;
  sdi->size       = size;
  sdi->formatType = formatType;
  sdi->fileLength = Swap32IfLE(file_stat.st_size);

  if (iconPathLength > 0)
    memcpy(sdi->filePath, iconPath, iconPathLength);
  result = WriteExact(rfbsock, buf, buflen);
  free(buf);
  return result;
}


/*
 * HandleRFBServerMessage.
 */

Bool
HandleRFBServerMessage()
{
  unsigned int i,j;
  rfbServerToClientMsg msg;

  if (!ReadFromRFBServer((char *)&msg, 1))
    return False;

  for (i = 0; i < encoders; i++) {
  	if (encoderFunc[i] && (encoderType[i] == RFB_MODULE_IS_ENCODER) && 
	    encoderFunc[i](&msg)) 
	  return True;
  }

  switch (msg.type) {
  case rfbWindowControl:
    if (!ReadFromRFBServer(((char *)&msg) + 1,
                           sz_rfbWindowControlMsg - 1))
      return False;

    windowControl(Swap32IfLE(msg.wnc.id), Swap16IfLE(msg.wnc.control));
    break;

  case rfbServerDataReq:
    if (!ReadFromRFBServer(((char *)&msg) + 1,
                           sz_rfbServerDataReqMsg_WM - 1))
      return False;

    processServerDataReqMsg((rfbServerDataReqMsg *)&msg);
    break;

  case rfbSetColourMapEntries:
    fprintf(stderr,"Ignoring message 'SetColourMapEntries' from VNC server\n");
    break;

  case rfbFramebufferUpdate:
    fprintf(stderr,"Ignoring message 'FramebufferUpdate' from VNC server\n");
    break;

  case rfbBell:
    fprintf(stderr,"Ignoring message 'Bell' from VNC server\n");
    break;

  case rfbServerCutText:
    if (!ReadFromRFBServer(((char *)&msg) + 1,
			   sz_rfbServerCutTextMsg - 1))
      return False;

    msg.sct.length = Swap32IfLE(msg.sct.length);

    if (textBufferSize < msg.sct.length+1) {
      if (textBuffer)
        free(textBuffer);
      textBuffer = malloc(msg.sct.length+1);
      if (textBuffer == NULL) {
        fprintf(stderr,"Memory allocation error.\n");
        return False;
      }
    }

    if (!ReadFromRFBServer(textBuffer, msg.sct.length))
      return False;

    textBuffer[msg.sct.length] = '\0';
    clipboardSetText(textBuffer, msg.sct.length);
    break;

  default:
    fprintf(stderr,"Unknown message type %d from VNC server\n",msg.type);
    return False;
  }

  return True;
}
