# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2011 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import urlparse
import urllib
import os
import hashlib
import time

import httplib2

from ninix.home import get_normalized_path


class NetworkUpdate:

    __BACKUP_SUFFIX = '.BACKUP'

    def __init__(self, callback):
        self.callback = callback
        self.event_queue = []
        self.state = None
        self.backups = []
        self.newfiles = []
        self.newdirs = []

    def is_active(self):
        return self.state is not None

    def enqueue_event(self, event,
                      ref0=None, ref1=None, ref2=None, ref3=None,
                      ref4=None, ref5=None, ref6=None, ref7=None):
        self.event_queue.append(
            (event, ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7))

    def get_event(self):
        return None if not self.event_queue else self.event_queue.pop(0)

    def has_events(self):
        return 1 if self.event_queue else 0

    def start(self, homeurl, ghostdir, timeout=60):
        url = urlparse.urlparse(homeurl)
        if not (url[0] == 'http' and url[3] == url[4] == url[5] == ''):
            self.enqueue_event('OnUpdateFailure', 'bad home URL')
            self.state = None
            return
        self.homeurl = url.geturl()
        if not self.homeurl.endswith('/'):
            self.homeurl = ''.join((self.homeurl, '/'))
        self.ghostdir = ghostdir
        self.timeout = timeout
        self.state = 0

    def interrupt(self):
        self.event_queue = []
        self.callback['enqueue_event']('OnUpdateFailure', 'artificial')
        self.state = None
        self.stop(revert=1)

    def stop(self, revert=0):
        if revert:
            for path in self.backups:
                if os.path.isfile(path):
                    os.rename(path, path[:-len(self.__BACKUP_SUFFIX)])
            for path in self.newfiles:
                if os.path.isfile(path):
                    os.remove(path)
            for path in self.newdirs:
                if os.path.isdir(path):
                    os.rmdir(path)
        else:
            for path in self.backups:
                if os.path.isfile(path):
                    os.remove(path)
        self.backups = []
        self.newfiles = []
        self.newdirs = []

    LEN_PRE = 3 ## FIXME
    LEN_STATE = 3 ## FIXME

    def run(self):
        if self.state is None or self.callback['check_event_queue']():
            return 0
        elif self.state == 0:
            self.start_updates()
        elif self.state == 1:
            self.wait_response()
        elif self.state == 2: ## FIXME
            self.schedule = self.make_schedule()
            if self.schedule is None:
                return 0
            self.final_state = len(self.schedule) * self.LEN_STATE + self.LEN_PRE
        elif self.state == self.final_state:
            self.end_updates()
        elif (self.state - self.LEN_PRE) % self.LEN_STATE == 0:
            filename, checksum = self.schedule[0]
            print 'UPDATE:', filename, checksum
            self.download(
                urlparse.urljoin(self.homeurl, urllib.quote(filename)),
                event=1)
        elif (self.state - self.LEN_PRE) % self.LEN_STATE == 1:
            self.wait_response()
        elif (self.state - self.LEN_PRE) % self.LEN_STATE == 2:
            filename, checksum = self.schedule.pop(0)
            self.update_file(unicode(filename, 'Shift_JIS'), checksum)
        return 1

    def start_updates(self):
        self.enqueue_event('OnUpdateBegin')
        self.download(urlparse.urljoin(self.homeurl, 'updates2.dau'))

    def download(self, locator, event=0):
        self.locator = self.encode(locator)
        self.http = httplib2.Http('.cache', timeout=self.timeout)
        self.http.force_exception_to_status_code = True
        if event:
            self.enqueue_event('OnUpdate.OnDownloadBegin',
                               os.path.basename(locator),
                               self.file_number, self.num_files)
        self.state += 1

    def encode(self, path):
        return ''.join([self.encode_special(c) for c in path])

    def encode_special(self, c):
        return c if '\x20' < c < '\x7e' else '%%%02x' % ord(c)

    def wait_response(self):
        self.response, self.content = self.http.request(self.locator)
        code = self.response.status
        message = self.response.reason
        ##print 'http:', self.locator, code, message
        if code == 408: ## FIXME
            self.enqueue_event('OnUpdateFailure', 'timeout')
            self.state = None
            self.stop(revert=1)
            return
        elif code == 200:
            pass
        else:
            if self.state == 1: # updates2.dau ## FIXME
                self.enqueue_event('OnUpdateFailure', str(code))
                self.state = None
                return
            filename, checksum = self.schedule.pop(0)
            print 'failed to download %s (%d %s)' % (filename, code, message)
            self.file_number += 1
            self.state += 2
            return
        self.state += 1

    def make_checksum(self, digest):
        return ''.join(['%02x' % ord(x) for x in digest])

    ROOT_FILES = ['install.txt', 'delete.txt', 'readme.txt', 'thumbnail.png']

    def adjust_path(self, filename):
        filename = get_normalized_path(filename)
        if filename in self.ROOT_FILES or os.path.dirname(filename):
            return filename
        return os.path.join('ghost', 'master', filename)

    def make_schedule(self):
        schedule = self.parse_updates2_dau()
        if schedule is not None:
            self.num_files = len(schedule) - 1
            self.file_number = 0
            if self.num_files >= 0:
                self.enqueue_event('OnUpdateReady', self.num_files)
            self.state += 1
        return schedule

    def parse_updates2_dau(self):
        schedule = []
        for line in self.content.splitlines():
            try:
                filename, checksum, newline = line.split('\001', 2)
            except ValueError:
                self.enqueue_event('OnUpdateFailure', 'broken updates2.dau')
                self.state = None
                return None
            if not filename:
                continue
            path = os.path.join(self.ghostdir, self.adjust_path(
                    unicode(filename, 'Shift_JIS')))
            with open(path, 'rb') as f:
                data = f.read()
                m = hashlib.md5()
                m.update(data)
                if checksum == self.make_checksum(m.digest()):
                    continue
            schedule.append((filename, checksum))
        self.updated_files = []
        return schedule

    def update_file(self, filename, checksum):
        m = hashlib.md5()
        m.update(self.content)
        digest = self.make_checksum(m.digest())
        if digest == checksum:
            path = os.path.join(self.ghostdir, self.adjust_path(filename))
            subdir = os.path.dirname(path)
            if not os.path.exists(subdir):
                subroot = subdir
                while 1:
                    head, tail = os.path.split(subroot)
                    if os.path.exists(head):
                        break
                    else:
                        subroot = head
                self.newdirs.append(subroot)
                try:
                    os.makedirs(subdir)
                except OSError:
                    self.enqueue_event(
                        'OnUpdateFailure', ''.join(("can't mkdir ", subdir)))
                    self.state = None
                    self.stop(revert=1)
                    return
            if os.path.exists(path):
                if os.path.isfile(path):
                    backup = ''.join((path, self.__BACKUP_SUFFIX))
                    os.rename(path, backup)
                    self.backups.append(backup)
            else:
                self.newfiles.append(path)
            try:
                with open(path, 'wb') as f:
                    try:
                        f.write(self.content)
                    except IOError:
                        self.enqueue_event(
                            'OnUpdateFailure',
                            ''.join(("can't write ", os.path.basename(path))))
                        self.state = None
                        self.stop(revert=1)
                        return
            except IOError:
                self.enqueue_event(
                    'OnUpdateFailure',
                    ''.join(("can't open ", os.path.basename(path))))
                self.state = None
                self.stop(revert=1)
                return
            self.updated_files.append(filename)
            event = 'OnUpdate.OnMD5CompareComplete'
        else:
            self.enqueue_event('OnUpdate.OnMD5CompareFailure',
                               filename, checksum, digest)
            self.state = None
            self.stop(revert=1)
            return
        self.enqueue_event(event, filename, checksum, digest)
        self.file_number += 1
        self.state += 1

    def end_updates(self):
        filelist = self.parse_delete_txt()
        if filelist:
            for filename in filelist:
                path = os.path.join(self.ghostdir, filename)
                if os.path.exists(path) and os.path.isfile(path):
                    try:
                        os.unlink(path)
                        print 'deleted', path
                    except OSError, e:
                        print e
        update_list = ','.join(self.updated_files)
        if not update_list:
            self.enqueue_event('OnUpdateComplete', 'none')
        else:
            self.enqueue_event('OnUpdateComplete', 'changed', update_list)
        self.state = None
        self.stop()

    def parse_delete_txt(self):
        filelist = []
        try:
            with open(os.path.join(self.ghostdir, 'delete.txt')) as f:
                for line in f:
                    line = line.strip()
                    if not line:
                        continue
                    filename = unicode(line, 'Shift_JIS')
                    filelist.append(get_normalized_path(filename))
        except IOError:
            return None
        return filelist


def test():
    import sys
    if len(sys.argv) != 3:
        raise SystemExit, 'Usage: update.py homeurl ghostdir\n'
    update = NetworkUpdate({'enqueu_event': lambda *a: None,
                            'check_event_queue': lambda *a: None,})
    update.start(sys.argv[1], sys.argv[2], timeout=60)
    while 1:
        state = update.state
        s = time.time()
        code = update.run()
        e = time.time()
        delta = e - s
        if delta > 0.1:
            print 'Warning: state = %d (%f sec)' % (state, delta)
        while 1:
            event = update.get_event()
            if not event:
                break
            print event
        if code == 0:
            break
        if update.state == 5 and update.schedule:
            print 'File(s) to be update:'
            for filename, checksum in update.schedule:
                print '   ', filename
    update.stop()


if __name__ == '__main__':
    test()
