
/* qmprint.c: unparsing of Q expressions */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qdefs.h"

/* expression outline parameters */

int maxlevel;	/* maximum expression depth */
int maxlist;	/* maximum list/tuple members */
int maxchars;	/* maximum string size */

/* format parameters */

static char    *my_fformat = "%.17g"; /* floating point format */
static int      my_imode = 0; /* integer format */

/* internal state */

static char    *bufp;
static int      bufleng, abufleng; /* buffer for unparsed expression */
static FILE    *fp;	/* output file for unparsed expression */
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
static iconv_t *ic;	/* iconv conversion handle for file output */
#endif
static int      (*putstr) (); /* string put routine */

/* 1-level cache for custom unparsings */

typedef struct {
  EXPR *x, *y;
} cache_t;

static cache_t cache = { NULL, NULL };

static inline void clear_cache(THREAD *thr)
{
  if (cache.x) qmfree(thr, cache.x);
  if (cache.y) qmfree(thr, cache.y);
  cache.x = NULL; cache.y = NULL;
}

/* handle recursive invokations of the unparser (through unparse methods) */

#define MAX_STATE_STACK 1024

static struct {
  char *bufp;
  int bufleng, abufleng;
  FILE *fp;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  iconv_t *ic;
#endif
  int (*putstr) ();
  cache_t cache;
} state_stack[MAX_STATE_STACK];

static int state_stack_size;

static inline int push_state(void)
{
  if (state_stack_size >= MAX_STATE_STACK) return 0;
  state_stack[state_stack_size].bufp = bufp;
  state_stack[state_stack_size].bufleng = bufleng;
  state_stack[state_stack_size].abufleng = abufleng;
  state_stack[state_stack_size].fp = fp;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  state_stack[state_stack_size].ic = ic;
#endif
  state_stack[state_stack_size].putstr = putstr;
  state_stack[state_stack_size].cache = cache;
  cache.x = cache.y = NULL;
  state_stack_size++;
  return 1;
}

static inline void pop_state(THREAD *thr)
{
  if (state_stack_size <= 0) return;
  state_stack_size--;
  bufp = state_stack[state_stack_size].bufp;
  bufleng = state_stack[state_stack_size].bufleng;
  abufleng = state_stack[state_stack_size].abufleng;
  fp = state_stack[state_stack_size].fp;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  ic = state_stack[state_stack_size].ic;
#endif
  putstr = state_stack[state_stack_size].putstr;
  clear_cache(thr);
  cache = state_stack[state_stack_size].cache;
}

/* keep track of punctuation operators, to pad with spaces as necessary */

#ifdef HAVE_UNICODE
static inline long
u8decode(char *s)
{
  size_t n;
  unsigned p = 0, q = 0;
  unsigned long c = 0;
  if (s[0] == 0)
    return -1;
  else if (s[1] == 0)
    return (unsigned char)s[0];
  for (n = 0; n == 0 && *s; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)*s) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  c = uc & 0x1f;
	  break;
	case 0xe0:
	  q = 2;
	  c = uc & 0xf;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0) {
	    q = 3;
	    c = uc & 0x7;
	  } else
	    c = uc;
	  break;
	default:
	  c = uc;
	  break;
	}
      } else
	c = uc;
      p = 0;
      if (q == 0) n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      c = c << 6 | (uc & 0x3f);
      if (--q == 0)
	n++;
      else
	p++;
    } else {
      /* malformed char */
      return -1;
    }
  }
  if (n == 1 && *s == 0)
    return c;
  else
    return -1;
}
#endif

static inline int ispunctsym(char *s)
{
  if (!*s || strstr(s, "::"))
    return 0;
  else {
#ifdef HAVE_UNICODE
    long c = u8decode(s);
    if (c >= 0)
      return u_ispunct(c);
    else
#endif      
      return ispunct(s[0]);
  }
}

#define DELIM "\"()[]{},;_"

static inline int ispunctsym2(char *s)
{
  return *s && !strchr(DELIM, *s) && ispunctsym(s);
}

static inline int ispunctsym3(char *s)
{
  int l = strlen(s);
#ifdef HAVE_UNICODE
  while (l > 0 && (((unsigned char)s[--l])&0xc0) == 0xc0) ;
#else
  if (l > 0) --l;
#endif
  return ispunctsym2(s+l);
}

static char tmpbuf[2*MAXSTRLEN];

static inline checktmpbuf(void)
{
  return tmpbuf[0] != 0;
}

static inline void inittmpbuf(void)
{
  tmpbuf[0] = 0;
}

static inline writetmpbuf(char *s)
{
  if (strlen(s) < 2*MAXSTRLEN) {
    strcpy(tmpbuf, s);
    return 1;
  } else
    return 0;
}

static inline int flushtmpbuf(int (*putstr) ())
{
  if (tmpbuf[0] != 0) {
    int res = (*putstr)(tmpbuf);
    tmpbuf[0] = 0;
    return res;
  } else
    return 1;
}

static int lastsymf, marksymf;

static inline checklastsym(char *s)
{
  int res = lastsymf && ispunctsym2(s);
  if (res && checktmpbuf()) res = -1;
  lastsymf = ispunctsym3(s);
  return res;
}

static inline checkmarksym(void)
{
  int res = marksymf;
  marksymf = 0;
  return res;
}

static inline marksym(int flag)
{
  marksymf = flag;
  return 1;
}

/* sputstr(): write string to buffer */

static sputstr1(char *s)
{
  int l = strlen(s);

  while (bufleng >= abufleng-l) {
    /* try to enlarge the buffer */
    char *bufp1;
    if (bufp1 = (char*) arealloc(bufp, abufleng, MAXSTRLEN,
				 sizeof(char))) {
      bufp = bufp1;
      abufleng += MAXSTRLEN;
    } else {
      free(bufp);
      return (0);
    }
  }
  strcpy(bufp+bufleng, s);
  bufleng += l;
  return (1);
}

static inline sputstr(char *s)
{
  int ret;
  if ((ret = checklastsym(s))) {
    if (ret < 0) {
      if (!sputstr1(" ")) return 0;
      if (!flushtmpbuf(sputstr1)) return 0;
    }
    if (!sputstr1(" ")) return 0;
  } else if (!flushtmpbuf(sputstr1))
    return 0;
  if (checkmarksym())
    return writetmpbuf(s);
  else
    return sputstr1(s);
}

/* fputstr(): write string to file */

#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)

#define CHUNKSZ 128

static inline char *
icfromutf8(iconv_t ic[2], char *s)
{
  if (ic[1] == (iconv_t)-2) {
    char *codeset = default_encoding();
    if (codeset && strcmp(codeset, "UTF-8"))
      ic[1] = iconv_open(codeset, "UTF-8");
    else
      ic[1] = (iconv_t)-1;
  }
  if (ic[1] == (iconv_t)-1)
    return NULL;
  else {
    /* Here the input buffer may be NULL, to emit a terminating shift
       sequence. In this case we initialize an output buffer of size
       CHUNKSZ. */
    size_t l = s?strlen(s):0, m = s?l:CHUNKSZ;
    char *t = malloc(m+1), *t1;
    char *inbuf = s, *outbuf = t;
    size_t inbytes = l, outbytes = m;

    while (iconv(ic[1], &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, m+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  m += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return NULL;
      }
    /* terminate the output string */
    *outbuf = 0;
    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

#endif

static fputstr1(char *s)
{
  char *t = s;
  int ret;

  if (checkbrk && fp == stdout)
    return (0);
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  t = icfromutf8(ic, s);
  if (!t) t = s;
#endif
  ret = fputs(t, fp);
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  if (t != s) free(t);
#endif
  if (ret == EOF)
    return (0);
  else
    return (1);
}

static inline fputstr(char *s)
{
  int ret;
  if ((ret = checklastsym(s))) {
    if (ret < 0) {
      if (!fputstr1(" ")) return 0;
      if (!flushtmpbuf(fputstr1)) return 0;
    }
    if (!fputstr1(" ")) return 0;
  } else if (!flushtmpbuf(fputstr1))
    return 0;
  if (checkmarksym())
    return writetmpbuf(s);
  else
    return fputstr1(s);
}

/* opstr(): return print representation of an operator symbol */

static char *pad(char *s, int l, int r)
{
  static char buffer[MAXSTRLEN];
  char *template;
  if (l && r)
    template = " %s ";
  else if (l)
    template = " %s";
  else if (r)
    template = "%s ";
  else
    return s;
  sprintf(buffer, template, s);
  return buffer;
}

static char *
opstr(int fno, int l, int r)
{
  static char s[MAXSTRLEN];
  int padding = 0, k;
  pname(s, fno); k = strlen(s);
  if (!*s)
    return " ??? "; /* this can't happen !? */
  /* these need special treatment */
  switch (fno) {
  case SEQOP:
  case RAPPOP:
  case COMPOP:
    padding = 1;
    break;
  case UMINOP:
    return "-";
  case APPOP:
    return " ";
  }
  s[k-1] = 0;
  if (symtb[fno].prec == 5 || symtb[fno].prec == 9) {
    if (!ispunctsym(s+1) || padding)
      s[k-1] = ' ';
    return s+1;
  } else if (ispunctsym(s+1) && !padding)
    return s+1;
  else
    return pad(s+1,l,r);
}

/* NOTE: SEQOP and friends are mapped to precedence -1 here, so that we make
   room for an extra precedence level for if-then-else which get a precedence
   of 0. Lambdas are at precedence level -2. Symbols with a precedence of NONE
   (-1) are remapped to the NOPREC value. */

#define NOPREC (-99)

/* opprec(): return precedence of operator symbol */

static opprec(int fno)
{
  switch (fno) {
  case SEQOP:
    return -1;	/* || */
  case RAPPOP:
    return 1;	/* $ */
  case LEOP:
  case GROP:
  case EQOP:
  case LEQOP:
  case GEQOP:
  case NEQOP:
  case IDOP:
    return 2;	/* relational operators */
  case CATOP:
  case ADDOP:
  case MINOP:
  case OROP:
  case ORELSEOP:
    return 3;	/* addition operators */
  case MULOP:
  case FDIVOP:
  case DIVOP:
  case MODOP:
  case ANDOP:
  case ANDTHENOP:
    return 4;	/* multiplication operators */
  case UMINOP:
  case NOTOP:
  case HASHOP:
    return 5;	/* prefix operators */
  case POWOP:
  case IDXOP:
    return 6;	/* super/subscript */
  case COMPOP:
    return 7;	/* composition */
  case APPOP:
    return 8;	/* @ */
  case QUOTEOP:
  case UNQUOTEOP:
  case FORCEOP:
  case MEMOP:
    return 9;	/* quotation */
  default:
    /* ordinary function symbol or user-defined operator */
    if (symtb[fno].prec == 0)
      return -1;
    else if (symtb[fno].prec < 0)
      return NOPREC;
    else
      return symtb[fno].prec;
  }
}

/* islsect(), isrsect(): check for left and right operator sections */

static inline islsect(EXPR *x)
{
  int p;
  return (x->fno == APPOP && (p = opprec(x->data.args.x1->fno)) >= -1 &&
	  p != 5 && p != 8 && p != 9);
}

static inline isrsect(EXPR *x)
{
  int p;
  return (x->fno == APPOP && x->data.args.x1->fno == APPOP &&
	  x->data.args.x1->data.args.x1->fno == FLIPOP &&
	  x->data.args.x1->data.args.x2->fno != MINOP &&
	  (p = opprec(x->data.args.x1->data.args.x2->fno)) >= -1 &&
	  p != 5 && p != 8 && p != 9);
}

static inline isenum(EXPR *x)
{
  if (!(x->fno == APPOP && x->data.args.x1->fno == APPOP &&
	x->data.args.x1->data.args.x1->fno == ENUMOP))
    return 0;
  x = x->data.args.x1->data.args.x2;
  if (x->fno == CONSOP) {
    while (x->fno == CONSOP)
      x = x->data.args.x2;
    return x->fno == NILOP;
  } else
    return x->fno != NILOP;
}

static inline isenum1(EXPR *x)
{
  if (!(x->fno == APPOP && x->data.args.x1->fno == ENUM1OP))
    return 0;
  x = x->data.args.x2;
  if (x->fno == CONSOP) {
    while (x->fno == CONSOP)
      x = x->data.args.x2;
    return x->fno == NILOP;
  } else
    return x->fno != NILOP;
}

static inline issenum(EXPR *x)
{
  if (!(x->fno == APPOP && x->data.args.x1->fno == APPOP &&
	x->data.args.x1->data.args.x1->fno == SENUMOP))
    return 0;
  x = x->data.args.x1->data.args.x2;
  if (x->fno == CONSOP) {
    while (x->fno == CONSOP)
      x = x->data.args.x2;
    return x->fno == NILOP;
  } else
    return x->fno != NILOP;
}

static inline issenum1(EXPR *x)
{
  if (!(x->fno == APPOP && x->data.args.x1->fno == SENUM1OP))
    return 0;
  x = x->data.args.x2;
  if (x->fno == CONSOP) {
    while (x->fno == CONSOP)
      x = x->data.args.x2;
    return x->fno == NILOP;
  } else
    return x->fno != NILOP;
}

static inline istenum(EXPR *x)
{
  if (!(x->fno == APPOP && x->data.args.x1->fno == APPOP &&
	x->data.args.x1->data.args.x1->fno == TENUMOP))
    return 0;
  x = x->data.args.x1->data.args.x2;
  if (x->fno == CONSOP) {
    while (x->fno == CONSOP)
      x = x->data.args.x2;
    return x->fno == NILOP;
  } else
    return x->fno != NILOP;
}

static inline istenum1(EXPR *x)
{
  if (!(x->fno == APPOP && x->data.args.x1->fno == TENUM1OP))
    return 0;
  x = x->data.args.x2;
  if (x->fno == CONSOP) {
    while (x->fno == CONSOP)
      x = x->data.args.x2;
    return x->fno == NILOP;
  } else
    return x->fno != NILOP;
}

static inline isstream(EXPR *x)
{
  return x->fno == SNILOP ||
    x->fno == APPOP && x->data.args.x1->fno == APPOP &&
    x->data.args.x1->data.args.x1->fno == SCONSOP;
}

static inline isscons(EXPR *x, EXPR **y, EXPR **z)
{
  if (x->fno == APPOP && x->data.args.x1->fno == APPOP &&
      x->data.args.x1->data.args.x1->fno == SCONSOP) {
    *y = x->data.args.x1->data.args.x2;
    *z = x->data.args.x2;
    return 1;
  } else
    return 0;
}

static inline islistof(EXPR *x)
{
  static char sym[20];
  int fno = mksym(strcpy(sym, "cond::listof"));
  return (fno != NONE && x->fno == APPOP && x->data.args.x1->fno == APPOP &&
	  x->data.args.x1->data.args.x1->fno == fno &&
	  x->data.args.x2->fno == VECTOP && x->data.args.x2->data.vect.n >= 1);
}

static inline istupleof(EXPR *x)
{
  static char sym[20];
  int fno = mksym(strcpy(sym, "cond::tupleof"));
  return (fno != NONE && x->fno == APPOP && x->data.args.x1->fno == APPOP &&
	  x->data.args.x1->data.args.x1->fno == fno &&
	  x->data.args.x2->fno == VECTOP && x->data.args.x2->data.vect.n >= 1);
}

static inline isstreamof(EXPR *x)
{
  static char sym[20];
  int fno = mksym(strcpy(sym, "cond::streamof"));
  return (fno != NONE && x->fno == APPOP && x->data.args.x1->fno == APPOP &&
	  x->data.args.x1->data.args.x1->fno == fno &&
	  x->data.args.x2->fno == VECTOP && x->data.args.x2->data.vect.n >= 1);
}

static inline islambda(EXPR *x)
{
  return (x->fno == APPOP && x->data.args.x1->fno == APPOP &&
	  x->data.args.x1->data.args.x1->fno == LAMBDAOP);
}

static inline isifelse(EXPR *x)
{
  static char sym[20];
  int fno = mksym(strcpy(sym, "cond::ifelse"));
  return (fno != NONE && x->fno == APPOP && x->data.args.x1->fno == APPOP &&
	  x->data.args.x1->data.args.x1->fno == APPOP &&
	  x->data.args.x1->data.args.x1->data.args.x1->fno == fno);
}

static inline iswhen(EXPR *x)
{
  static char sym[20];
  int fno = mksym(strcpy(sym, "cond::when"));
  return (fno != NONE && x->fno == APPOP && x->data.args.x1->fno == APPOP &&
	  x->data.args.x1->data.args.x1->fno == fno);
}

/* unparse(): use custom unparser if present */

bool unparseflag = 1;

static inline EXPR *unparse(THREAD *thr, EXPR *x, int cached)
{
  EXPR *f, *y;
  int type;
  int _mode;
  if (!unparseflag) return NULL;
  if (x == cache.x) return qmnew(cache.y);
  type = (x->argc||x->virt)?0:x->type;
  if (!type || !(symtb[type].flags&VIRT)) return NULL;
  _mode = thr->mode;
  thr->mode = 1;
  if (!(f = funexpr(thr, UNPARSEOP))) {
    thr->mode = _mode;
    return NULL;
  }
  if ((y = qmnew(consexpr(thr, APPOP, f, x)))) {
    int res = 0;
    bool _brkdbg = thr->brkdbg, _debug = thr->debug;
    thr->mode = _mode;
    thr->brkdbg = thr->debug = 0;
    if (push_state()) {
      res = eval(thr, y);
      pop_state(thr);
    }
    thr->brkdbg = _brkdbg; thr->debug = _debug;
    qmfree(thr, y);
    if (res)
      y = *--thr->xsp;
    else
      return NULL;
  } else {
    thr->mode = _mode;
    qmfree(thr, qmnew(f));
    return NULL;
  }
  if (y->fno == APPOP && y->data.args.x1->fno == QUOTEOP) {
    EXPR *z = qmnew(y->data.args.x2);
    qmfree(thr, y);
    if (cached) {
      if (cache.x) qmfree(thr, cache.x);
      if (cache.y) qmfree(thr, cache.y);
      cache.x = qmnew(x); cache.y = qmnew(z);
    }
    return z;
  } else {
    qmfree(thr, y);
    return NULL;
  }
}

/* exprprec(): return precedence of an expression */

static exprprec(THREAD *thr, EXPR *x)
{
  int p;
  EXPR *y = unparse(thr, x, 1);

  if (y) {
    p = exprprec(thr, y);
    qmfree(thr, y);
    return p;
  }
  switch (x->fno) {
    /* applications */
  case APPOP:
    if (islsect(x) || isrsect(x) || isenum(x) || issenum(x) || istenum(x) ||
	issenum1(x) || islistof(x) || isstreamof(x) || istupleof(x) ||
	isstream(x))
      return 9; /* section, enumeration, comprehension or stream */
    else if (islambda(x))
      return -2; /* lambda */
    else if (isifelse(x) || iswhen(x))
      return 0; /* if-then-else */
    else if (x->data.args.x1->fno == APPOP)
      if ((p = opprec(x->data.args.x1->data.args.x1->fno)) >= -1)
	return (p);	/* binary op */
      else
	return (8);	/* @ */
    else if ((p = opprec(x->data.args.x1->fno)) == 5 || p == 9)
      return (p);		/* unary op */
    else
      return (8);		/* @ */
    /* atoms: */
  case INTVALOP:
    return (mpz_sgn(x->data.z) < 0 ? 5 : 9);
    /* 5 = prec. of unary minus */
  case FLOATVALOP:
    return (x->data.f < 0.0 ? 5 : 9);
  default:
    return (9);
  }
}

/* pretty-printing operations: */

static print(THREAD *thr, int level, EXPR *x);

static inline printparen(THREAD *thr, int level, EXPR *x)
{
  return ((*putstr) ("(") && print(thr, level+1, x) && (*putstr) (")"));
}

static printlprec(THREAD *thr, int level, int fno, EXPR *x)
{
  if (x)
    if (opprec(fno) > exprprec(thr, x))
      return (printparen(thr, level, x));
    else
      return (print(thr, level, x));
  else
    return 1;
}

static printrprec(THREAD *thr, int level, int fno, EXPR *x)
{
  if (x)
    if (opprec(fno) >= exprprec(thr, x))
      return (printparen(thr, level, x));
    else
      return (print(thr, level, x));
  else
    return 1;
}

static print1(THREAD *thr, int level, int fno, EXPR *x)
{
  return ((*putstr) (opstr(fno,0,0)) &&
	  printlprec(thr, level, fno, x));
}

static print2l(THREAD *thr, int level, int fno, EXPR *x, EXPR *y)
{
  return (printlprec(thr, level, fno, x) &&
	  marksym(x!=NULL) &&
	  (*putstr) (opstr(fno,x!=NULL,y!=NULL)) &&
	  printrprec(thr, level, fno, y));
}

static print2r(THREAD *thr, int level, int fno, EXPR *x, EXPR *y)
{
  return (printrprec(thr, level, fno, x) &&
	  marksym(x!=NULL) &&
	  (*putstr) (opstr(fno,x!=NULL,y!=NULL)) &&
	  printlprec(thr, level, fno, y));
}

static print2n(THREAD *thr, int level, int fno, EXPR *x, EXPR *y)
{
  return (printrprec(thr, level, fno, x) &&
	  marksym(x!=NULL) &&
	  (*putstr) (opstr(fno,x!=NULL,y!=NULL)) &&
	  printrprec(thr, level, fno, y));
}

static print2(THREAD *thr, int level, int fno, EXPR *x, EXPR *y)
{
  switch (opprec(fno)) {
    /* relational ops (non-associative): */
  case 2:
    return (print2n(thr, level, fno, x, y));
    /* ($), sub/superscript (right-associative): */
  case 1:
  case 6:
    return (print2r(thr, level, fno, x, y));
    /* others (left-associative): */
  default:
    return (print2l(thr, level, fno, x, y));
  }
}

/* print an expression (outline only if maxlevel > 0), also truncate strings
   and lists/tuples to length maxchars and maxlist if nonzero,
   respectively) */

static char	s1[MAXSTRLEN], s2[MAXSTRLEN];

#ifdef HAVE_UNICODE

static inline size_t
u8strlen(char *s)
{
  size_t n = 0;
  unsigned p = 0, q = 0;
 start:
  for (; *s; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  break;
	case 0xe0:
	  q = 2;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0)
	    q = 3;
	  break;
	}
      }
      p = 0; n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      if (--q == 0)
	p = 0;
      else
	p++;
    } else {
      /* malformed char */
      s -= p+1; p = q = 0;
    }
  }
  if (q > 0) {
    /* unterminated char */
    s -= p; p = q = 0;
    goto start;
  }
  return n;
}

static inline char *
u8strind(char *s, size_t i)
{
  unsigned p = 0, q = 0;
 start:
  for (; *s && i > 0; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  break;
	case 0xe0:
	  q = 2;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0)
	    q = 3;
	  break;
	}
      }
      p = 0; if (q == 0) i--;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      if (--q == 0) {
	p = 0; i--;
      } else
	p++;
    } else {
      /* malformed char */
      i--; s -= p+1; p = q = 0;
    }
  }
  if (q > 0) {
    /* unterminated char */
    i--; s -= p; p = q = 0;
    goto start;
  }
  return s;
}

#endif

static char *strunc(char *s)
{
  /* truncate a string to maximum length */
#ifdef HAVE_UNICODE
  if (maxchars > 0 && u8strlen(s) > maxchars)
    strcpy(u8strind(s, maxchars-3), "...");
#else
  if (maxchars > 0 && strlen(s) > maxchars)
    strcpy(s+(maxchars-3), "...");
#endif
  return s;
}

static print(THREAD *thr, int level, EXPR *x)
{
  EXPR *y, *z;
  char test;
  extern int stack_dir;
  if (cstackmax > 0 && stack_dir*(&test - thr->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr->qmstat = AST_OVF;
    return 0;
  }
  if (maxlevel > 0 && level >= maxlevel)
    return (*putstr) ("...");
  if ((y = unparse(thr, x, 0))) {
    int res = print(thr, level, y);
    qmfree(thr, y);
    //    clear_cache(thr);
    return res;
  }
  switch (x->fno) {
  /* atoms: */
  case INTVALOP: {
    int l = mpz_sizeinbase(x->data.z, (my_imode==1)?16:(my_imode==2)?8:10)+1;
    if (mpz_sgn(x->data.z) < 0) l++;
    if (l >= MAXSTRLEN) {
      /* LARGE buffer needed */
      char *s1 = malloc(l*sizeof(char));
      if (s1) {
	int ret = (*putstr) (pmpz(s1, my_imode, x->data.z));
	free(s1);
	return ret;
      } else {
	thr->qmstat = MEM_OVF;
	return 0;
      }
    } else
      return ((*putstr) (pmpz(s1, my_imode, x->data.z)));
  }
  case FLOATVALOP:
    return ((*putstr) (pfloat(s1, my_fformat, x->data.f)));
  case STRVALOP:
    if (strlen(x->data.s)*4+2 >= MAXSTRLEN) {
      /* LARGE buffer needed */
      /* Note: Since s may contain non-printable characters,
	 the representation of s may be as large as
	 strlen(s)*4+2 + 1 for \0 */
      char *s1 = malloc((strlen(x->data.s)*4+3)*
			sizeof(char));
      char *s2 = malloc((strlen(x->data.s)*4+3)*
			sizeof(char));
      if (!s1 || !s2) {
	if (s1) free(s1);
	thr->qmstat = MEM_OVF;
	return (0);
      } else if ((*putstr) ((sprintf(s1, "\"%s\"",
				     strunc(pstr(s2, x->data.s))), s1))) {
	free(s1); free(s2);
	return (1);
      } else {
	free(s1); free(s2);
	return (0);
      }
    } else
      /* default buffer is large enough */
      return ((*putstr) ((sprintf(s1, "\"%s\"",
				  strunc(pstr(s2, x->data.s))), s1)));
  case FILEVALOP:
  case BADFILEVALOP:
    return ((*putstr) ((sprintf(s1, "<<%s>>",
				x->type?pname(s2, x->type):
				"???"), s1)));
  case USRVALOP:
    return ((*putstr) ((sprintf(s1, "<<%s>>",
				x->type?pname(s2, x->type):
				"???"), s1)));
  /* {}: */
  case SNILOP:
    return (*putstr) ("{}");
  /* [|]: */
  case CONSOP: {
    int count = 1;
    if (maxlevel > 0 && level+1 >= maxlevel)
      return (*putstr) ("[...]");
    if (!(*putstr) ("[") || !print(thr, level+1, x->data.args.x1))
      return (0);
    x = x->data.args.x2;
    if (maxlist > 0)
      while (x->fno == CONSOP && count < maxlist) {
	if (!(*putstr) (",") || !print(thr, level+1, x->data.args.x1))
	  return (0);
	x = x->data.args.x2;
	count++;
      }
    else
      while (x->fno == CONSOP) {
	if (!(*putstr) (",") || !print(thr, level+1, x->data.args.x1))
	  return (0);
	x = x->data.args.x2;
      }
    if (x->fno == NILOP)
      return ((*putstr) ("]"));
    else if (maxlist > 0 && count == maxlist)
      if (x->fno == CONSOP)
	return ((*putstr) (",...]"));
      else
	return ((*putstr) ("|...]"));
    else
      return ((*putstr) ("|") && print(thr, level+1, x) && (*putstr) ("]"));
  }
  /* (|): */
  case PAIROP: {
    int count = 1;
    char *lb = "(", *rb = ")";
    if (x->data.args.x2->fno == VOIDOP) {
      lb = "("; rb = ",)";
    }
    if (maxlevel > 0 && level+1 >= maxlevel)
      return (*putstr) ("(...)");
    if (!(*putstr) (lb) || !print(thr, level+1, x->data.args.x1))
      return (0);
    x = x->data.args.x2;
    if (maxlist > 0)
      while (x->fno == PAIROP && count < maxlist) {
	if (!(*putstr) (",") || !print(thr, level+1, x->data.args.x1))
	  return (0);
	x = x->data.args.x2;
	count++;
      }
    else
      while (x->fno == PAIROP) {
	if (!(*putstr) (",") || !print(thr, level+1, x->data.args.x1))
	  return (0);
	x = x->data.args.x2;
      }
    if (x->fno == VOIDOP)
      return ((*putstr) (rb));
    else if (maxlist > 0 && count == maxlist)
      if (x->fno == PAIROP)
	return ((*putstr) (",...") && (*putstr) (rb));
      else
	return ((*putstr) ("|...") && (*putstr) (rb));
    else
      return ((*putstr) ("|") && print(thr, level+1, x) && (*putstr) (rb));
  }
  /* @: */
  case APPOP:
    /* syntactic sugar */
    if (islsect(x))
      /* left sections */
      if (maxlevel > 0 && level+1 >= maxlevel)
	return (*putstr) ("(...)");
      else
	return (*putstr) ("(") &&
	  print2(thr, level+1, x->data.args.x1->fno, x->data.args.x2, NULL) &&
	  (*putstr) (")");
    else if (isrsect(x))
      /* right sections */
      if (maxlevel > 0 && level+1 >= maxlevel)
	return (*putstr) ("(...)");
      else
	return (*putstr) ("(") &&
	  print2(thr, level+1, x->data.args.x1->data.args.x2->fno, NULL,
		 x->data.args.x2) &&
	  (*putstr) (")");
    else if (isscons(x, &y, &z)) {
      /* {|} */
      int count = 1;
      if (maxlevel > 0 && level+1 >= maxlevel)
	return (*putstr) ("{...}");
      if (!(*putstr) ("{") || !print(thr, level+1, y))
	return (0);
      x = z;
      if (maxlist > 0)
	while (isscons(x, &y, &z) && count < maxlist) {
	  if (!(*putstr) (",") || !print(thr, level+1, y))
	    return (0);
	  x = z;
	  count++;
	}
      else
	while (isscons(x, &y, &z)) {
	  if (!(*putstr) (",") || !print(thr, level+1, y))
	    return (0);
	  x = z;
	}
      if (x->fno == SNILOP)
	return ((*putstr) ("}"));
      else if (maxlist > 0 && count == maxlist)
	if (isscons(x, &y, &z))
	  return ((*putstr) (",...}"));
	else
	  return ((*putstr) ("|...}"));
      else
	return ((*putstr) ("|") && print(thr, level+1, x) && (*putstr) ("}"));
    } else if (islambda(x))
      /* lambda */
      return ((*putstr) ("\\") &&
	      printrprec(thr, level, APPOP, x->data.args.x1->data.args.x2) &&
	      (*putstr) (" . ") && print(thr, level, x->data.args.x2));
    else if (iswhen(x)) {
      /* if-then */
      if (!((*putstr) ("if ") &&
	    printlprec(thr, level, RAPPOP,
		       x->data.args.x1->data.args.x2) &&
	    (*putstr) (" then ")))
	return 0;
      if (exprprec(thr, x->data.args.x2) < 0)
	return ((*putstr) ("(") && print(thr, level+1, x->data.args.x2) &&
		(*putstr) (")"));
      else
	return print(thr, level, x->data.args.x2);
    } else if (isifelse(x)) {
      /* if-then-else */
      if (!((*putstr) ("if ") &&
	    printlprec(thr, level, RAPPOP,
		       x->data.args.x1->data.args.x1->data.args.x2) &&
	    (*putstr) (" then ")))
	return 0;
      if (iswhen(x->data.args.x1->data.args.x2) ||
	  exprprec(thr, x->data.args.x1->data.args.x2) < 0) {
	if (!((*putstr) ("(") &&
	      print(thr, level+1, x->data.args.x1->data.args.x2) &&
	      (*putstr) (")")))
	  return 0;
      } else if (!print(thr, level, x->data.args.x1->data.args.x2))
	return 0;
      if (!((*putstr) (" else ")))
	return 0;
      if (exprprec(thr, x->data.args.x2) < 0)
	return ((*putstr) ("(") && print(thr, level+1, x->data.args.x2) &&
		(*putstr) (")"));
      else
	return print(thr, level, x->data.args.x2);
    } else if (isenum(x))
      /* list enumerations */
      if (maxlevel > 0 && level+1 >= maxlevel)
	return (*putstr) ("[...]");
      else {
	EXPR *y = x->data.args.x2;
	x = x->data.args.x1->data.args.x2;
	if (x->fno == CONSOP) {
	  if (!(*putstr) ("[") || !print(thr, level+1, x->data.args.x1))
	    return (0);
	  x = x->data.args.x2;
	  while (x->fno == CONSOP) {
	    if (!(*putstr) (",") || !print(thr, level+1, x->data.args.x1))
	      return (0);
	    x = x->data.args.x2;
	  }
	} else if (!(*putstr) ("[") || !print(thr, level+1, x))
	  return (0);
	return ((*putstr) ("..") && print(thr, level+1, y) && (*putstr) ("]"));
      }
    else if (istenum(x))
      /* tuple enumerations */
      if (maxlevel > 0 && level+1 >= maxlevel)
	return (*putstr) ("(...)");
      else {
	EXPR *y = x->data.args.x2;
	x = x->data.args.x1->data.args.x2;
	if (x->fno == CONSOP) {
	  if (!(*putstr) ("(") || !print(thr, level+1, x->data.args.x1))
	    return (0);
	  x = x->data.args.x2;
	  while (x->fno == CONSOP) {
	    if (!(*putstr) (",") || !print(thr, level+1, x->data.args.x1))
	      return (0);
	    x = x->data.args.x2;
	  }
	} else if (!(*putstr) ("(") || !print(thr, level+1, x))
	  return (0);
	return ((*putstr) ("..") && print(thr, level+1, y) && (*putstr) (")"));
      }
    else if (issenum(x))
      /* stream enumerations */
      if (maxlevel > 0 && level+1 >= maxlevel)
	return (*putstr) ("{...}");
      else {
	EXPR *y = x->data.args.x2;
	x = x->data.args.x1->data.args.x2;
	if (x->fno == CONSOP) {
	  if (!(*putstr) ("{") || !print(thr, level+1, x->data.args.x1))
	    return (0);
	  x = x->data.args.x2;
	  while (x->fno == CONSOP) {
	    if (!(*putstr) (",") || !print(thr, level+1, x->data.args.x1))
	      return (0);
	    x = x->data.args.x2;
	  }
	} else if (!(*putstr) ("{") || !print(thr, level+1, x))
	  return (0);
	return ((*putstr) ("..") && print(thr, level+1, y) && (*putstr) ("}"));
      }
    else if (isenum1(x))
      /* half-open enumerations */
      if (maxlevel > 0 && level+1 >= maxlevel)
	return (*putstr) ("[...]");
      else {
	x = x->data.args.x2;
	if (x->fno == CONSOP) {
	  if (!(*putstr) ("[") || !print(thr, level+1, x->data.args.x1))
	    return (0);
	  x = x->data.args.x2;
	  while (x->fno == CONSOP) {
	    if (!(*putstr) (",") || !print(thr, level+1, x->data.args.x1))
	      return (0);
	    x = x->data.args.x2;
	  }
	} else if (!(*putstr) ("[") || !print(thr, level+1, x))
	  return (0);
	return ((*putstr) ("..]"));
      }
    else if (istenum1(x))
      /* half-open tuple enumerations */
      if (maxlevel > 0 && level+1 >= maxlevel)
	return (*putstr) ("(...)");
      else {
	x = x->data.args.x2;
	if (x->fno == CONSOP) {
	  if (!(*putstr) ("(") || !print(thr, level+1, x->data.args.x1))
	    return (0);
	  x = x->data.args.x2;
	  while (x->fno == CONSOP) {
	    if (!(*putstr) (",") || !print(thr, level+1, x->data.args.x1))
	      return (0);
	    x = x->data.args.x2;
	  }
	} else if (!(*putstr) ("(") || !print(thr, level+1, x))
	  return (0);
	return ((*putstr) ("..)"));
      }
    else if (issenum1(x))
      /* infinite stream enumerations */
      if (maxlevel > 0 && level+1 >= maxlevel)
	return (*putstr) ("{...}");
      else {
	x = x->data.args.x2;
	if (x->fno == CONSOP) {
	  if (!(*putstr) ("{") || !print(thr, level+1, x->data.args.x1))
	    return (0);
	  x = x->data.args.x2;
	  while (x->fno == CONSOP) {
	    if (!(*putstr) (",") || !print(thr, level+1, x->data.args.x1))
	      return (0);
	    x = x->data.args.x2;
	  }
	} else if (!(*putstr) ("{") || !print(thr, level+1, x))
	  return (0);
	return ((*putstr) ("..}"));
      }
    else if (islistof(x))
      /* list comprehensions */
      if (maxlevel > 0 && level+1 >= maxlevel)
	return (*putstr) ("[...]");
      else {
	EXPR **xv = x->data.args.x2->data.vect.xv;
	int i, n = x->data.args.x2->data.vect.n;
	if (!(*putstr) ("[") ||
	    !print(thr, level+1, x->data.args.x1->data.args.x2) ||
	    !(*putstr) (":") || !print(thr, level+1, xv[0]))
	  return (0);
	for (i = 1; i < n; i++)
	  if (!(*putstr) (",") || !print(thr, level+1, xv[i]))
	    return (0);
	return ((*putstr) ("]"));
      }
    else if (istupleof(x))
      /* tuple comprehensions */
      if (maxlevel > 0 && level+1 >= maxlevel)
	return (*putstr) ("(...)");
      else {
	EXPR **xv = x->data.args.x2->data.vect.xv;
	int i, n = x->data.args.x2->data.vect.n;
	if (!(*putstr) ("(") ||
	    !print(thr, level+1, x->data.args.x1->data.args.x2) ||
	    !(*putstr) (":") || !print(thr, level+1, xv[0]))
	  return (0);
	for (i = 1; i < n; i++)
	  if (!(*putstr) (",") || !print(thr, level+1, xv[i]))
	    return (0);
	return ((*putstr) (")"));
      }
    else if (isstreamof(x))
      /* stream comprehensions */
      if (maxlevel > 0 && level+1 >= maxlevel)
	return (*putstr) ("{...}");
      else {
	EXPR **xv = x->data.args.x2->data.vect.xv;
	int i, n = x->data.args.x2->data.vect.n;
	if (!(*putstr) ("{") ||
	    !print(thr, level+1, x->data.args.x1->data.args.x2) ||
	    !(*putstr) (":") || !print(thr, level+1, xv[0]))
	  return (0);
	for (i = 1; i < n; i++)
	  if (!(*putstr) (",") || !print(thr, level+1, xv[i]))
	    return (0);
	return ((*putstr) ("}"));
      }
    else if (x->data.args.x1->fno == APPOP)
      switch (opprec(x->data.args.x1->data.args.x1->fno)) {
	/* @: */
      case NOPREC: case 5: case 8: case 9:
	return print2l(thr, level, APPOP, x->data.args.x1,
		       x->data.args.x2);
      default:
	return print2(thr, level, x->data.args.x1->data.args.x1->fno,
		      x->data.args.x1->data.args.x2,
		      x->data.args.x2);
      }
    else if (opprec(x->data.args.x1->fno) == 5 ||
	     opprec(x->data.args.x1->fno) == 9)
      /* unary operators: */
      return (print1(thr, level, x->data.args.x1->fno,
		     x->data.args.x2));
    else
      /* @: */
      return (print2l(thr, level, APPOP, x->data.args.x1,
		      x->data.args.x2));
  /* vectors: */
  case VECTOP: {
    int n = x->data.vect.n;
    if (n == 0)
      return ((*putstr) (pname(s1, VOIDOP)));
    else if (maxlevel > 0 && level+1 >= maxlevel)
      return (*putstr) ("(...)");
    else if (n == 1)
      return ((*putstr) ("(") && print(thr, level+1, x->data.vect.xv[0]) &&
	      (*putstr) (",)"));
    else {
      int i;
      if (!(*putstr) ("(") || !print(thr, level+1, x->data.vect.xv[0]))
	return 0;
      if (maxlist > 0) {
	for (i = 1; i < maxlist && i < n; i++)
	  if (!(*putstr) (",") || !print(thr, level+1, x->data.vect.xv[i]))
	    return 0;
	if (i < n)
	  return ((*putstr) (",...)"));
	else
	  return ((*putstr) (")"));
      } else {
	for (i = 1; i < n; i++)
	  if (!(*putstr) (",") || !print(thr, level+1, x->data.vect.xv[i]))
	    return 0;
	return ((*putstr) (")"));
      }
    }
  }
  /* other symbols: */
  default:
    return ((*putstr) (pname(s1, x->fno)));
  }
}

static inline printchk(THREAD *thr, int level, EXPR *x)
{
  int res;
  char base, *baseptr = thr->baseptr;
  extern int stack_dir;
  if (!baseptr)
    thr->baseptr = &base;
  if (cstackmax > 0 && stack_dir*(&base - thr->baseptr) >= cstackmax) {
    /* C stack overflow -- bail out */
    thr->qmstat = AST_OVF;
    return 0;
  }
  lastsymf = marksymf = 0;
  inittmpbuf();
  res = print(thr, level, x);
  if (!baseptr)
    thr->baseptr = NULL;
  return res;
}

/* sprintx() unparses an expression to a null-terminated string. */

char *sprintx(EXPR *x)
{
  THREAD *thr = get_thr();
  int retval;
  if (!(bufp = malloc(sizeof(char)*MAXSTRLEN)))
    return (NULL);
  bufleng = 0;
  abufleng = MAXSTRLEN;
  putstr = sputstr;
  retval = printchk(thr, 0, x) && flushtmpbuf(sputstr1);
  clear_cache(thr);
  if (retval) {
    if (!(bufp = realloc(bufp, sizeof(char)*(strlen(bufp)+1))))
      fatal(qmmsg[THIS_CANT_HAPPEN]);
    return (bufp);
  } else {
    return (NULL);
  }
}

/* fprintx() unparses an expression to a file. It returns zero unless
   an error is encountered writing to the file. */

#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
int fprintx(FILE *_fp, iconv_t *_ic, EXPR *x)
#else
int fprintx(FILE *_fp, EXPR *x)
#endif
{
  THREAD *thr = get_thr();
  int retval;
  fp = _fp;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  ic = _ic;
#endif
  putstr = fputstr;
  retval = printchk(thr, 0, x) && flushtmpbuf(fputstr1);
  clear_cache(thr);
  return (retval || checkbrk);
}

/* printx() unparses an expression to stdout. */

int printx(EXPR *x)
{
  char *save_fformat = my_fformat;
  int save_imode = my_imode;
  int ret;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  static iconv_t ic[2] = { (iconv_t)-2, (iconv_t)-2 };
  EXPR *out = (EXPR*)symtb[OUTPUTOP].x;
  my_fformat = fformat;
  my_imode = imode;
  if (out && out->fno == FILEVALOP)
    ret = fprintx(stdout, out->data.fargs.ic, x);
  else
    ret = fprintx(stdout, ic, x);
#else
  my_fformat = fformat;
  my_imode = imode;
  ret = fprintx(stdout, x);
#endif
  my_fformat = save_fformat;
  my_imode = save_imode;
  return ret;
}
