
/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "sys.h"

#ifndef HAVE_STRDUP

char *strdup(char *s)
{
  char *t;
  return ((t=malloc(strlen(s)+1))?strcpy(t, s):NULL);
}

#endif

#ifndef HAVE_STRSTR

char *strstr(char *s, char *t)
{
  char *s1, *t1;
  while (strlen(t) <= strlen(s)) {
    for (s1 = s, t1 = t; *t1 && *s1 == *t1; s1++, t1++)
      ;
    if (!*t1)
      return s;
    else
      s++;
  }
  return NULL;
}

#endif

#ifndef HAVE_MEMCPY

#if __GNUC__ > 1
#define memcpy(TO,FROM,COUNT)	__builtin_memcpy(TO,FROM,COUNT)
#else
void *memcpy (char *to, char *from, int count)
{
  register char *f = from;
  register char *t = to;
  register int i = count;

  while (i-- > 0)
    *t++ = *f++;
  return (void*)to;
}
#endif

#endif

#ifndef HAVE_MEMMOVE

#if __GNUC__ > 1
#define memmove(TO,FROM,COUNT)	__builtin_memmove(TO,FROM,COUNT)
#else
void *memmove (char *to, char *from, int count)
{
  register char *f = from;
  register char *t = to;
  register int i = count;

  if (from > to) {
    while (i-- > 0)
      *t++ = *f++;
  } else if (from < to) {
    from += count; to += count;
    while (i-- > 0)
      *--t = *--f;
  }
  return (void*)to;
}
#endif

#endif

#ifndef HAVE_MEMSET

#if __GNUC__ > 1
#define memset(TO,C,COUNT)	__builtin_memset(TO,C,COUNT)
#else
void *memset (char *to, int c, int count)
{
  register char f = (char)c;
  register char *t = to;
  register int i = count;

  while (i-- > 0)
    *t++ = f;
  return (void*)to;
}
#endif

#endif

#ifndef HAVE_MEMCMP

#if __GNUC__ > 1
#define memcmp(S1,S2,COUNT)	__builtin_memcmp(S1,S2,COUNT)
#else
int memcmp (char *s1, char *s2, int count)
{
  register char *t1 = s1;
  register char *t2 = s2;
  register int i = count;
  register int c;

  while (i-- > 0)
    if ((c = *t1++ - *t2++) != 0)
      return c;
  return 0;
}
#endif

#endif

sighandler_t syssignal(int sig, sighandler_t handler)
{
#ifdef HAVE_POSIX_SIGNALS
  struct sigaction new_action, old_action;
  new_action.sa_handler = handler;
  sigemptyset(&new_action.sa_mask);
  new_action.sa_flags = 0;
  sigaction(sig, &new_action, &old_action);
  return old_action.sa_handler;
#else
  return signal(sig, handler);
#endif
}

#if defined (HAVE_POSIX_SIGNALS)
static sigset_t saved_sigmask;
#endif

void save_sigmask(void)
{
#if defined (HAVE_POSIX_SIGNALS)
  sigprocmask(0, 0, &saved_sigmask);
#endif
}

void restore_sigmask(void)
{
#if defined (HAVE_POSIX_SIGNALS)
  sigprocmask(SIG_SETMASK, &saved_sigmask, 0);
#endif
}

#ifdef HAVE_CLOCK_GETTIME

double systime(void)
{
  struct timespec tv;
  int res = clock_gettime(CLOCK_REALTIME, &tv);
  if (res)
    return 0.0;
  else
    return ((double)tv.tv_sec)+((double)tv.tv_nsec)*1e-9;
}

#else

#ifdef HAVE_GETTIMEOFDAY

double systime(void)
{
  struct timeval tv;
  gettimeofday(&tv, NULL);
  return ((double)tv.tv_sec)+((double)tv.tv_usec)*1e-6;
}

#else

#ifdef HAVE_FTIME

#include <sys/timeb.h>

double systime(void)
{
  struct timeb tb;
  ftime(&tb);
  return ((double)tb.time)+((double)tb.millitm)*1e-3;
}

#else

double systime(void)
{
  return (double)time(NULL);
}

#endif

#endif

#endif

#ifndef _WIN32

void syssleep(double t)
{
  if (t > 0.0) {
    double ip, fp;
    unsigned long secs;
#ifdef HAVE_NANOSLEEP
    unsigned long nsecs;
    struct timespec req;
    fp = modf(t, &ip);
    if (ip > LONG_MAX) { ip = (double)LONG_MAX; fp = 0.0; }
    secs = (unsigned long)ip;
    nsecs = (unsigned long)(fp*1e9);
    req.tv_sec = secs; req.tv_nsec = nsecs;
    nanosleep(&req, NULL);
#else
#ifdef HAVE_USLEEP
    unsigned long usecs;
    if (t > LONG_MAX) t = LONG_MAX;
    fp = modf(t, &ip);
    secs = (unsigned long)ip;
    usecs = (unsigned long)(fp*1e6);
    usleep(secs*1000000U+usecs);
#else
    fp = modf(t, &ip);
    if (ip > LONG_MAX) ip = (double)LONG_MAX;
    secs = (unsigned long)ip;
    sleep(secs);
#endif
#endif
  }
}

#else

#include <windows.h>

void syssleep(double t)
{
  if (t > 0.0) {
    unsigned long secs;
    unsigned short msecs;
    double ip, fp;
    if (t > LONG_MAX) t = LONG_MAX;
    fp = modf(t, &ip);
    secs = (unsigned long)ip;
    msecs = (unsigned short)(fp*1e3);
    Sleep(secs*1000U+msecs);
  }
}

/* simplistic "emulation" of lt_dlopen and friends with the Windows API */

/* Windoze error messages (as returned by FormatMessage()) are worse than
   nothing, we do our own error messages. */

static char errstr[1024], *dlerrstr = NULL, *lt_path = NULL;
static int counter = 0;

int lt_dlinit(void)
{
  counter++;
  return 0;
}

int lt_dlexit(void)
{
  if (counter <= 0) return 1;
  if (lt_path) free(lt_path);
  lt_path = NULL;
  counter--;
  return 0;
}

int lt_dlsetsearchpath(char *path)
{
  if (!(path = strdup(path))) return 1;
  if (lt_path) free(lt_path);
  lt_path = path;
  return 0;
}

static char lt_dirstr[] = DIRSTR, lt_volstr[] = VOLSTR;

extern char *expand(), *absname();
extern int absolute(), dirprefix();

#define tilde(s) (s[0] == '~' && (!s[1] || strchr(lt_dirstr, s[1]) && !strchr(lt_volstr, s[1])))

static char*
searchpath(char *s1, char *s2)
{
  char           *s, *t;

  if (tilde(s2))
    return expand(s1, s2);
  else if (absolute(s2) || dirprefix(s2, ".") || dirprefix(s2, ".."))
    return strcpy(s1, s2);
  else if (lt_path)
    for (s = lt_path; *s; s = t) {
      int l;
      FILE *fp;
      char p[1024];
      if (!(t = strchr(s, PATHDELIM)))
	t = strchr(s, 0);
      if (s == t) goto next;
      if (s[0] == '.')
	if (t == s+1)
	  s = t;
	else if (strchr(lt_dirstr, s[1]) &&
		 !strchr(lt_volstr, s[1]))
	  s += 2;
      l = t-s;
      strncpy(p, s, l);
      p[l] = 0;
      expand(s1, p);
      l = strlen(s1);
      if (l > 0 && (!strchr(lt_dirstr, s1[l-1]) || 
		    strchr(lt_volstr, s1[l-1])))
	s1[l] = *lt_dirstr, l++;
      strcpy(s1+l, s2);
      if (fp = fopen(s1, "rb")) {
	fclose(fp);
	return s1;
      }
    next:
      if (*t) t++;
    }
  return strcpy(s1, s2);
}

void *lt_dlopen(char *filename)
{
  char fname[1024], aname[1024];
  void *handle = LoadLibrary(absname(aname, searchpath(fname, filename)));
  if (!handle) {
    sprintf(errstr, "%s: error loading module", filename);
    dlerrstr = errstr;
  } else
    dlerrstr = NULL;
  return handle;
}

void *lt_dlopenext(char *filename)
{
  char fname[1024], aname[1024];
  void *handle = LoadLibrary(absname(aname, searchpath(fname, filename)));
  if (!handle) {
    char extfilename[1024];
    strcat(strcpy(extfilename, filename), ".dll");
    handle = LoadLibrary(absname(aname, searchpath(fname, extfilename)));
  }
  if (!handle) {
    sprintf(errstr, "%s: error loading module", filename);
    dlerrstr = errstr;
  } else
    dlerrstr = NULL;
  return handle;
}

int lt_dlclose(void *handle)
{
  int res = FreeLibrary((HMODULE)handle);
  if (!res) {
    char modname[1024];
    GetModuleFileName((HMODULE)handle, modname, 1024);
    sprintf(errstr, "%s: error unloading module", modname);
    dlerrstr = errstr;
  } else
    dlerrstr = NULL;
  return !res;
}

void *lt_dlsym(void *handle, char *symbol)
{
  void *proc = GetProcAddress(handle, symbol);
  if (!proc) {
    char modname[1024];
    GetModuleFileName((HMODULE)handle, modname, 1024);
    sprintf(errstr, "%s: symbol `%s' not found", modname, symbol);
    dlerrstr = errstr;
  } else
    dlerrstr = NULL;
  return proc;
}

char *lt_dlerror(void)
{
  char *preverr = dlerrstr;
  dlerrstr = NULL;
  return preverr;
}

/* support for handling "sigint" and "sigterm" events sent by qpad */

static HANDLE hSigInt, hSigTerm, hSignalHandler;

static DWORD WINAPI SignalHandler(LPVOID dummy)
{
  HANDLE hEvents[2] = { hSigInt, hSigTerm };
  while (1) {
    switch (WaitForMultipleObjects(2, hEvents, FALSE, INFINITE)) {
    case WAIT_OBJECT_0:
      raise(SIGINT);
      break;
    case WAIT_OBJECT_0+1:
      raise(SIGTERM);
      break;
    default:
      ExitProcess(0);
    }
  }
}

int InstallSignalHandler(void)
{
  TCHAR szSigInt[MAX_PATH], szSigTerm[MAX_PATH];
  DWORD dwSignalHandler;
  sprintf(szSigInt, "Q_SIGINT-%u", GetCurrentProcessId());
  sprintf(szSigTerm, "Q_SIGTERM-%u", GetCurrentProcessId());
  hSigInt = OpenEvent(EVENT_ALL_ACCESS, FALSE, szSigInt);
  hSigTerm = OpenEvent(EVENT_ALL_ACCESS, FALSE, szSigTerm);
  if (hSigInt != NULL && hSigTerm != NULL) {
    hSignalHandler = CreateThread(NULL, 0, SignalHandler, NULL,
				  0, &dwSignalHandler);
    return hSignalHandler != NULL;
  } else
    return hSigInt == hSigTerm;
}

#endif /* _WIN32 */

