/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#ifndef RAST_DB_H
#define RAST_DB_H

/**
 * @file db.h db
 */

#include "rast/error.h"
#include "rast/rast.h"
#include "rast/text_index.h"
#include "rast/result.h"
#include "rast/macros.h"

RAST_EXTERN_C_BEGIN

/**
 * @defgroup db db
 * @{
 */

/** Options for rast_db_create() */
typedef struct {
    /** The byte order of the database */
    rast_byte_order_e byte_order;
    /** The encoding of the database */
    const char *encoding;
    /** Whether the whole text should be preseved in the database */
    int preserve_text;
    /** The property definitions */
    rast_property_t *properties;
    /** The number of the properties */
    int num_properties;
    /** The block size of .pos files */
    rast_size_t pos_block_size;
} rast_db_create_option_t;

/** Options for rast_db_optimize() */
typedef struct {
    int squeeze_doc_id;
} rast_db_optimize_option_t;

/**
 * Create options for rast_db_create().
 * @param pool The pool to allocate the memory out of
 * @return The created options
 */
rast_db_create_option_t *rast_db_create_option_create(apr_pool_t *pool);

/** Options for rast_db_open() */
typedef struct {
    /** The number of registered characters to start sync */
    int sync_threshold_chars;
} rast_db_open_option_t;

/**
 * Create options for rast_db_open().
 * @param pool The pool to allocate the memory out of
 * @return The created options
 */
rast_db_open_option_t *rast_db_open_option_create(apr_pool_t *pool);

#define RAST_DB_RDWR    0       /**< Opening database in read-write mode */
#define RAST_DB_RDONLY  1       /**< Opening database in read-only mode */

/**
 * Create options for rast_db_optimize().
 * @param pool The pool to allocate the memory out of
 * @return The created options
 */
rast_db_optimize_option_t *rast_db_optimize_option_create(apr_pool_t *pool);

typedef struct rast_document_t rast_document_t;
typedef struct rast_db_t rast_db_t;

struct rast_document_t {
    rast_db_t *db;
    apr_pool_t *pool;

    rast_error_t *(*add_text)(rast_document_t *doc,
                              const char *text, int nbytes);
    rast_error_t *(*set_property)(rast_document_t *doc,
                                  const char *name,
                                  const rast_value_t *value);
    rast_error_t *(*commit)(rast_document_t *doc);
    rast_error_t *(*abort)(rast_document_t *doc);
    rast_error_t *(*get_doc_id)(rast_document_t *doc, rast_doc_id_t *doc_id);
};

struct rast_db_t {
    /** The pool associated with the database */
    apr_pool_t *pool;

    rast_error_t *(*db_sync)(rast_db_t *db);
    rast_error_t *(*db_close)(rast_db_t *db);
    rast_error_t *(*db_register)(rast_db_t *db,
                                 const char *text, rast_size_t nbytes,
                                 rast_value_t *properties,
                                 rast_doc_id_t *doc_id);
    rast_error_t *(*db_create_document)(rast_db_t *db, rast_document_t **doc);
    rast_error_t *(*db_search)(rast_db_t *db, const char *query,
                               rast_search_option_t *options,
                               rast_result_t **result, apr_pool_t *pool);
    rast_error_t *(*db_delete)(rast_db_t *db, rast_doc_id_t doc_id);
    rast_error_t *(*db_update)(rast_db_t *db,
                               rast_doc_id_t old_doc_id,
                               const char *text, rast_size_t nbytes,
                               rast_value_t *properties,
                               rast_doc_id_t *new_doc_id);
    rast_error_t *(*db_get_text)(rast_db_t *db,
                                 rast_doc_id_t doc_id,
                                 char **s, rast_size_t *nbytes,
                                 apr_pool_t *pool);
    rast_byte_order_e (*db_byte_order)(rast_db_t *db);
    const char *(*db_encoding)(rast_db_t *base);
    const rast_property_t *(*db_properties)(rast_db_t *db,
                                            int *num_properties);
    int (*db_sync_threshold_chars)(rast_db_t *db);
};

static inline rast_db_t *
rast_document_get_db(rast_document_t *doc)
{
    return doc->db;
}

/**
 * Add text to the document.
 * @param doc The document
 * @param text The text to add the document
 * @param nbytes The length of text to add in bytes
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_document_add_text(rast_document_t *doc,
                                     const char *text, int nbytes);

/**
 * Set property to the document.
 * @param doc The document
 * @param name The property name
 * @param value The property value
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_document_set_property(rast_document_t *doc,
                                         const char *name,
                                         const rast_value_t *value);

/**
 * Commit the document to database.
 * @param doc The document
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_document_commit(rast_document_t *doc);

/**
 * Abort the document to database.
 * @param doc The document
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_document_abort(rast_document_t *doc);

/**
 * Get the document ID of the document.
 * @param doc The document
 * @param doc_id The document ID registerd
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_document_get_doc_id(rast_document_t *doc,
                                       rast_doc_id_t *doc_id);

typedef rast_error_t *rast_open_func_t(rast_db_t **db,
                                       const char *name, int flags,
                                       rast_db_open_option_t *options,
                                       apr_pool_t *pool);

/**
 * Add a function for rast_db_open.  If you want to know how to use this
 * function, please read xmlrpc_client.c :>.
 * @param open_func The function of rast_db_open
 */
void rast_add_open_function(rast_open_func_t *open_func);

/**
 * Create a database.  This function do not open the created database.
 * Use rast_db_open() to open the database.
 * @param name The name of the database
 * @param options The options to create a database.
 * @param pool The pool to allocate the memory out of
 * @return RAST_OK if succeeded, error otherwise
 * @see rast_db_create_option_create()
 */
rast_error_t *rast_db_create(const char *name,
                             rast_db_create_option_t *options,
                             apr_pool_t *pool);

/**
 * Optimize the database.
 * @param name The name of the database
 * @param options The options to optimize a database.
 * @param pool The pool to allocate the memory out of
 * @return RAST_OK if succeeded, error otherwise
 * @see rast_db_optimize_option_create()
 */
rast_error_t *rast_db_optimize(const char *name,
                               const rast_db_optimize_option_t *options,
                               apr_pool_t *pool);

/**
 * Open the specified database.
 * @param db The database opened
 * @param name The name of the database
 * @param flags How to open
 *          - RAST_RDRW
 *            - open for reading and writing
 *          - RAST_RDONLY
 *            - open for reading
 * @param options The options to open the database.
 * @param pool The pool to allocate the memory out of
 * @return RAST_OK if succeeded, error otherwise
 * @see rast_db_open_option_create()
 */
rast_error_t *rast_db_open(rast_db_t **db, const char *name, int flags,
                           rast_db_open_option_t *options, apr_pool_t *pool);

/**
 * Sync registered data to files.
 * @param db The database to sync
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_db_sync(rast_db_t *db);

/**
 * Close the database.
 * @param db The database to close
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_db_close(rast_db_t *db);

/**
 * Register a document to the database.
 * @param db The database
 * @param text The text of the document
 * @param nbytes The length of text in bytes
 * @param properties The property values of documents
 * @param doc_id If non-NULL, the registered document ID
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_db_register(rast_db_t *db,
                               const char *text, rast_size_t nbytes,
                               rast_value_t *properties,
                               rast_doc_id_t *doc_id);

/**
 * Create a document object to register the database.
 * @param db The database
 * @param doc The document for register
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_db_create_document(rast_db_t *db, rast_document_t **doc);

/**
 * Search documents in the database.
 * @param db The database
 * @param query The query string
 * @param options search options
 * @param result The search result
 * @param pool The pool to allocate the memory out of
 * @return RAST_OK if succeeded, error otherwise
 * @see rast_search_option_create()
 */
rast_error_t *rast_db_search(rast_db_t *db,
                             const char *query, rast_search_option_t *options,
                             rast_result_t **result, apr_pool_t *pool);

/**
 * Delete the document from the database.
 * @param db The database
 * @param doc_id The document ID
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_db_delete(rast_db_t *db, rast_doc_id_t doc_id);

/**
 * Update the document in the database.
 * @param db The database
 * @param old_doc_id The document ID
 * @param text The text of the document
 * @param nbytes The length of text in bytes
 * @param properties The property values of documents
 * @param new_doc_id If non-NULL, Updated document ID
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_db_update(rast_db_t *db,
                             rast_doc_id_t old_doc_id,
                             const char *text, rast_size_t nbytes,
                             rast_value_t *properties,
                             rast_doc_id_t *new_doc_id);

/**
 * Get the text of the document from the database.
 * @param db The database
 * @param doc_id The document ID
 * @param s The text of the document
 * @param nbytes The length of the text in bytes
 * @param pool The pool to allocate the memory out of
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_db_get_text(rast_db_t *db, rast_doc_id_t doc_id,
                               char **s, rast_size_t *nbytes,
                               apr_pool_t *pool);

/**
 * Return the byte order of the database.
 * @param db The database
 * @return The byte order of the database
 * @see rast_local_db_byte_order()
 */
rast_byte_order_e rast_db_byte_order(rast_db_t *db);

/**
 * Return the encoding of the database.
 * @param db The database
 * @return The encoding of the database
 * @see rast_local_db_encoding()
 */
const char *rast_db_encoding(rast_db_t *db);

/**
 * Return the property definitions of the database.
 * @param db The database
 * @param num_properties The number of properties
 * @return The property definitions
 */
const rast_property_t *rast_db_properties(rast_db_t *db, int *num_properties);

/**
 * Return the number of registered characters to start sync.
 * @param db The database
 * @return The number of registered characters to start sync
 * @see rast_local_db_sync_threshold_chars()
 */
int rast_db_sync_threshold_chars(rast_db_t *db);

/** @} */

RAST_EXTERN_C_END

#endif
