# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "tempfile"
require "fileutils"
require File.join(File.dirname(__FILE__), "read-buckets-to-file")

class ApplicationTar
  SUPPORTED_VERSION = 1
  MIME_TYPE = "application/x-tar"

  include ReadBucketsToFile

  private

  def process_file(filter, mime_type, path)
    tmp_dir = mkdtemp("tar")

    begin
      `cd #{tmp_dir}; tar xvf #{path}`.each_line do |name|
        filename = File.join(tmp_dir, name.chomp)
        if File.file?(filename)
          File.open(filename) do |f|
            brigade = Rast::Brigade.new
            brigade.insert_tail(Rast::FileBucket.new(f))
            brigade.insert_tail(Rast::EOSBucket.new)
            filter.pass(brigade, nil, filename)
          end
        end
      end
    ensure
      rm_rf(tmp_dir)
    end
  end

  def mkdtemp(prefix, mode = 0700)
    retry_count = 0
    begin
      dir = File.join(Dir.tmpdir, 
                      "#{prefix}-#{$$}.#{rand(10000)}")
      Dir.mkdir(dir, mode)
      return dir
    rescue Errno::EEXIST
      if retry_count < 3
        retry_count += 1
        retry
      else
        raise "can't create #{dir}"
      end
    end
  end

  def rm_rf(filename)
    count = 0
    begin
      FileUtils.rm_rf(filename)
    rescue Errno::ENOTEMPTY, Errno::EBADF
      if count >= 30
        raise
      end
      count += 1
      retry
    end
  end
end
