# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "test/unit"
require "yaml"

require "rast"
require "test-utility"
require "make-variable"
require "rast/database-generatable"

class RastStatTest < Test::Unit::TestCase
  include Rast::DatabaseGeneratable
  COMMAND = File.join(MakeVariable::TOP_BUILDDIR, "src", "rast-stat")

  def test_simple
    options = {
      "encoding" => "utf8",
      "properties" => [],
    }
    db_name = generate_db_name
    Rast::DB.create(db_name, options)
    result = YAML.load(`#{COMMAND} #{db_name}`.to_s)
    assert_equal(db_name, result["db_name"])
    assert_equal(0, result["num_register"])
    assert_equal("utf8", result["encoding_module"])
    assert_equal(false, result["preserve_text"])
    assert_equal(true, result["is native"])
    assert_equal(512, result["pos block size"])
    assert_equal([], result["properties"])
  end

  def test_with_summary
    options = {
      "encoding" => "utf8",
      "preserve_text" => true,
      "properties" => [],
    }
    db_name = generate_db_name
    Rast::DB.create(db_name, options)
    result = YAML.load(`#{COMMAND} #{db_name}`.to_s)
    assert_equal(db_name, result["db_name"])
    assert_equal(0, result["num_register"])
    assert_equal("utf8", result["encoding_module"])
    assert_equal(true, result["preserve_text"])
    assert_equal(true, result["is native"])
    assert_equal(512, result["pos block size"])
    assert_equal([], result["properties"])
  end

  def test_with_property
    options = {
      "encoding" => "utf8",
      "preserve_text" => true,
      "properties" => [
        {
          "name" => "title",
          "type" => Rast::PROPERTY_TYPE_STRING,
          "search" => true,
          "text_search" => true,
          "full_text_search" => true,
          "unique" => false,
        },
        {
          "name" => "count",
          "type" => Rast::PROPERTY_TYPE_UINT,
          "search" => true,
          "text_search" => false,
          "full_text_search" => false,
          "unique" => true,
        },
        {
          "name" => "date",
          "type" => Rast::PROPERTY_TYPE_DATE,
          "search" => false,
          "text_search" => false,
          "full_text_search" => false,
          "unique" => false,
        },
      ],
    }
    db_name = generate_db_name
    Rast::DB.create(db_name, options)
    result = YAML.load(`#{COMMAND} #{db_name}`.to_s)
    assert_equal(db_name, result["db_name"])
    assert_equal(0, result["num_register"])
    assert_equal("utf8", result["encoding_module"])
    assert_equal(true, result["preserve_text"])
    assert_equal(true, result["is native"])
    assert_equal(512, result["pos block size"])

    assert_equal("title", result["properties"][0]["name"])
    assert_equal("RAST_TYPE_STRING", result["properties"][0]["type"])
    assert_equal("RAST_PROPERTY_FLAG_SEARCH",
                 result["properties"][0]["flags"][0])
    assert_equal("RAST_PROPERTY_FLAG_TEXT_SEARCH",
                 result["properties"][0]["flags"][1])
    assert_equal("RAST_PROPERTY_FLAG_FULL_TEXT_SEARCH",
                 result["properties"][0]["flags"][2])
    assert_equal(3, result["properties"][0]["flags"].length)

    assert_equal("count", result["properties"][1]["name"])
    assert_equal("RAST_TYPE_UINT", result["properties"][1]["type"])
    assert_equal("RAST_PROPERTY_FLAG_SEARCH",
                 result["properties"][1]["flags"][0])
    assert_equal("RAST_PROPERTY_FLAG_UNIQUE",
                 result["properties"][1]["flags"][1])
    assert_equal(2, result["properties"][1]["flags"].length)

    assert_equal("date", result["properties"][2]["name"])
    assert_equal("RAST_TYPE_DATE", result["properties"][2]["type"])
    assert_equal([], result["properties"][2]["flags"])
  end
end
