# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "test/unit"

require "rast_test"
require "rast/filter"

module Rast
  module Filter
    load_mime_filter("postscript.rb")

    class ApplicationPostScriptTest < Test::Unit::TestCase
      def test_invoke
        invoke_test_simple_en
        invoke_test_simple_ja
      end

      def invoke_test_simple_en
        filter = SpoofFilter.new("UTF-8")
        brigade = Brigade.new

        path = File.join(MakeVariable::TOP_SRCDIR, "tests", "data",
                         "filter", "en", "postscript-a2psj.ps")
        file = File.open(path)
        bucket = FileBucket.new(file)
        brigade.insert_tail(bucket)
        brigade.insert_tail(EOSBucket.new)
        filter_module = ApplicationPostScript.new
        filter_module.invoke(filter, brigade, "application/postscript")

        brigades = []
        filter.passed.each do |passed|
          buckets = []
          passed.brigade.each do |bucket|
            buckets.push(bucket)
          end
          brigades.push(buckets)
        end

        assert_equal("text/plain", filter.passed[0].mime_type)
        buf = brigades[0][0].read
        assert_match(/PostScript test./, buf)
        assert_match(/This is PostScript file./, buf)
        assert_match(/This page is Page 1./, buf)
        assert_equal(1, brigades[0].length)
        assert_match(/This page is Page 2./, brigades[1][0].read)
        assert_equal(true, brigades[1][1].eos?)
        assert_equal(2, brigades[1].length)
        assert_equal(2, brigades.length)
        assert_equal("PostScript test", filter.properties["title"])
        assert_equal("author@example.com (author)", filter.properties["for"])
        assert_match(/a2ps/, filter.properties["creator"])
        match_data = /%%CreationDate: (.*)$/.match(File.read(path))
        assert_equal(match_data[1], filter.properties["creationdate"])
        assert_equal("(atend)", filter.properties["pages"])
        assert_equal("Ascend", filter.properties["pageorder"])
        assert_equal("A4", filter.properties["documentpapersizes"])
        assert_equal("Portrait", filter.properties["orientation"])
        assert_equal("author@example.com (author)",
                     filter.properties["author"])
        assert_equal(9, filter.properties.length)
        assert_equal(2, filter.passed.length)
      end

      def invoke_test_simple_ja
        filter = SpoofFilter.new("UTF-8")
        brigade = Brigade.new

        path = File.join(MakeVariable::TOP_SRCDIR, "tests", "data",
                         "filter", "ja", "postscript-a2psj.ps")
        file = File.open(path)
        bucket = FileBucket.new(file)
        brigade.insert_tail(bucket)
        brigade.insert_tail(EOSBucket.new)
        filter_module = ApplicationPostScript.new
        filter_module.invoke(filter, brigade, "application/postscript")

        brigades = []
        filter.passed.each do |passed|
          buckets = []
          passed.brigade.each do |bucket|
            buckets.push(bucket)
          end
          brigades.push(buckets)
        end

        assert_equal("text/plain", filter.passed[0].mime_type)
        buf = NKF.nkf("-Ew", brigades[0][0].read)
        assert_match(/PostScript テスト/, buf)
        assert_match(/これはPostScriptファイルです/, buf)
        assert_match(/1ページ目です。/, buf)
        assert_equal(1, brigades[0].length)
        assert_match(/2ページ目です。/, NKF.nkf("-Ew", brigades[1][0].read))
        assert_equal(true, brigades[1][1].eos?)
        assert_equal(2, brigades[1].length)
        assert_equal(2, brigades.length)
        assert_equal(2, filter.passed.length)
        assert_equal("PostScript test", filter.properties["title"])
        assert_equal("author@example.com (author)", filter.properties["for"])
        assert_match(/a2ps/, filter.properties["creator"])
        match_data = /%%CreationDate: (.*)$/.match(File.read(path))
        assert_equal(match_data[1], filter.properties["creationdate"])
        assert_equal("(atend)", filter.properties["pages"])
        assert_equal("Ascend", filter.properties["pageorder"])
        assert_equal("A4", filter.properties["documentpapersizes"])
        assert_equal("Portrait", filter.properties["orientation"])
        assert_equal("author@example.com (author)",
                     filter.properties["author"])
        assert_equal(9, filter.properties.length)
      end
    end
  end
end
