#include "config.h"

#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <signal.h>
#include <dirent.h>

#include <limits.h>

#include "saphire/saphire.h"
#include "saphire/saphire_inner.h"

// 内部コマンド用のリダイレクト処理
static BOOL statment_tree_redirect(sCommand* command, sRFd ** nextin, sWFd** nextout, int* nexterr, sRFd* pipein, sWFd* pipeout, int pipeerr, char* sname, int sline)
{
    int k;
    for(k=0; k<vector_size(command->mRedirects); k++) {
        sRedirect* redirect = (sRedirect*)vector_item(command->mRedirects, k);
        if(redirect->mType == kRedirectErrAndOutput) {
            err_msg("inner command can't take error and output redirect", sname, sline);
            return FALSE;
        }
        else {
            mode_t mode;
            switch(redirect->mType) {
                case kRedirectInput:
                    mode = O_RDONLY;
                    break;
                
                case kRedirectOverwrite:
                    mode = O_WRONLY | O_CREAT | O_TRUNC;
                    break;
                    
                case kRedirectAppend:
                    mode = O_WRONLY | O_CREAT | O_APPEND;
                    break;
            }

            int openfd = open(string_c_str(redirect->mFName), mode, 0666);
            if(openfd < 0) {
                char buf[PATH_MAX];
                snprintf(buf, PATH_MAX, "open %s is err1\n"
                            , string_c_str(redirect->mFName));
                err_msg(buf, sname, sline);
                return FALSE;
            }

            switch(redirect->mType) {
                case kRedirectInput:
                    if(*nextin == pipein) {
                        *nextin = RFD_NEW(openfd);
                    }
                    else {
                        if(!sRFd_close(*nextin)) {
                            return FALSE;
                        }
                        *nextin = RFD_NEW(openfd);
                    }
                    break;
                
                case kRedirectOverwrite:
                    if(redirect->mFd == 1) {
                        if(*nextout == pipeout) {
                            *nextout = WFD_NEW(openfd);
                        }
                        else {
                            if((*nextout)->mFd != -1 && (*nextout)->mFd != 1) {
                                (void)close((*nextout)->mFd);
                            }
                            (*nextout)->mFd = openfd;
                        }
                    }
                    else {
                        if(*nexterr != 2 && *nexterr != pipeerr) (void)close(*nexterr);
                        
                        *nexterr = openfd;
                    }
                    break;
                    
                case kRedirectAppend:
                    if(redirect->mFd == 1) {
                        if(*nextout == pipeout) {
                            *nextout = WFD_NEW(openfd);
                        }
                        else {
                            if((*nextout)->mFd != -1 && (*nextout)->mFd != 1) {
                                (void)close((*nextout)->mFd);
                            }
                            (*nextout)->mFd = openfd;
                        }
                    }
                    else {
                        if(*nexterr != 2 && *nexterr != pipeerr) (void)close(*nexterr);
                        
                        *nexterr = openfd;
                    }
                    break;
            }
        }
    }

    return TRUE;
}

// forkした子供(外部コマンド)をexecさせたい関数
static BOOL statment_tree_exec_cprog(sCommand* command, int nextin, int nextout, int nexterr, vector_obj* argv, char* sname, int sline)
{
    /// リダイレクト ///
    int err_and_output = FALSE;
    int k;
    for(k=0; k<vector_size(command->mRedirects); k++) {
        sRedirect* redirect = (sRedirect*)vector_item(command->mRedirects, k);

        if(redirect->mType == kRedirectErrAndOutput) {
            err_and_output = TRUE;
        }
        else {
            mode_t mode;
            switch(redirect->mType) {
                case kRedirectInput:
                    mode = O_RDONLY;
                    break;
                
                case kRedirectOverwrite:
                    mode = O_RDWR | O_CREAT | O_TRUNC;
                    break;
                    
                case kRedirectAppend:
                    mode = O_RDWR | O_CREAT | O_APPEND;
                    break;
            }

            int openfd = open(string_c_str(redirect->mFName), mode, 0666);
            if(openfd < 0) {
                char buf[PATH_MAX];
                snprintf(buf, PATH_MAX, "open %s is err2\n"
                                 , string_c_str(redirect->mFName));
                err_msg(buf, sname, sline);
                return FALSE; 
            }
            
            switch(redirect->mFd) {
                case 0:
                    if(nextin != 0 && nextin != -1) {
                        (void)close(nextin);
                    }
                    nextin = openfd;
                    break;
                    
                case 1:
                    if(nextout != 1 && nextout != -1) {
                        (void)close(nextout);
                    }
                    nextout = openfd;
                    break;
                    
                case 2:
                    if(nexterr != 2) {
                        (void)close(nexterr);
                    }
                    nexterr = openfd;
                    break;
            }
        }
    }

    /// パイプ ///
    if(err_and_output) {
        if(dup2(nextout, 2) < 0) {
            perror("dup2 1");
            exit(1);
        }
    }
    
    if(nextin != 0) {
        if(dup2(nextin, 0) < 0) {
            char buf[32];
            snprintf(buf, 32, "dup2 3 nextin %d", nextin);
            perror(buf);
            exit(1);
        }
        if(close(nextin) < 0) { return FALSE; }
    }
    
    if(nextout != 1) {
        if(dup2(nextout, 1) < 0) {
            char buf[128];
            snprintf(buf, 128, "dup2 nextout (%d)", nextout);
            perror(buf);
            exit(1);
        }
        
        if(close(nextout) < 0) { return FALSE; }
    }
    
    if(nexterr != 2) {
        if(dup2(nexterr, 2) < 0) {
            perror("dup2 5");
            exit(1);
        }
        
        if(close(nexterr) < 0) { return FALSE; }
    }

    /// char** に変換 ///
    char** argv2 = (char**)malloc(sizeof(char*)*(vector_size(argv)+1));

    int i;
    for(i=0; i<vector_size(argv); i++) {
        string_obj* item = (string_obj*)vector_item(argv, i);
        argv2[i] = string_c_str(item);
    }
    argv2[i] = NULL;

    /// exec ///
    execvp(argv2[0], argv2);
    fprintf(stderr, "exec('%s') error\n", argv2[0]);
    kill(getppid(), SIGUSR1);
    exit(1);
}

ssize_t bufsiz_write(int fd, char* buf, int sigint)
{
    int len = strlen(buf);
    char* p = buf;
    while(*p) {
        int size;
        if(len - (p - buf) < BUFSIZ) {
            size = len - (p-buf);
        }
        else {
            size = BUFSIZ;
        }

        ssize_t ret = write(fd, p, size);
        if(ret < 0) {
            return ret;
        }

        if(sigint) {
            if(gKitutukiSigInt) {
                errno = EINTR;
                return 1;
            }
        }

        p+=size;
    }

    return 0;
}

// 外部コマンドの処理
static BOOL statment_tree_external_command(sRunInfo* runinfo, sStatment* statment, vector_obj* argv, sRFd* nextin, sWFd* nextout, int nexterr, sWFd* pipeout, sCommand* command, BOOL last_program, int* last_program_pid, char* sname, int sline, vector_obj* blocks, sRFd* pipein, vector_obj* psubs)
{
//printf("nextin %d nextout %d\n", nextin->mFd, nextout->mFd);
    char* argv0 = string_c_str(vector_item(argv, 0));

    /// メモリーパイプをパイプに変換 ///
    int mpipefds[2] = { -1, -1 };
    int mpipefds2[2] = { -1, -1 };
    int nextin2 = nextin->mFd;
    int nextout2 = nextout->mFd;
    if(nextin2 == -1) {
        if(pipe(mpipefds) < 0) {
            perror("pipe");
            exit(1);
        }
        nextin2 = mpipefds[0];
        mpipefds[0] = -1;
    }
    if(nextout2 == -1) {
        if(pipe(mpipefds2) < 0) {
            perror("pipe");
            exit(1);
        }
        
        nextout2 = mpipefds2[1];
        mpipefds2[1] = -1;
    }
 
    /// fork ///
    pid_t pid = fork();
    if(pid < 0) {
        perror("fork");
        exit(1);
    }

    /// 子プロセスの処理 ///
    if(pid == 0) {
        /// 環境変数 ///
        int i;
        for(i=0; i<vector_size(runinfo->envs); i+=2) {
            char* name = string_c_str(vector_item(runinfo->envs, i));
            char* value = string_c_str(vector_item(runinfo->envs, i+1));

            setenv(name, value, 1);
        }
        
        if(mpipefds[0] != -1) (void)close(mpipefds[0]);
        if(mpipefds[1] != -1) (void)close(mpipefds[1]);
        if(mpipefds2[0] != -1) (void)close(mpipefds2[0]);
        if(mpipefds2[1] != -1) (void)close(mpipefds2[1]);

        pid = getpid();

        // プログラムは固有のプロセスグループに属する
        if(gAppType != kATOptC) {
            (void)setpgid(pid, pid);
        }
        /// 前に持ってくる
        if((gAppType == kATCursesApp 
                || gAppType == kATConsoleApp)
            && !statment->mBackground 
            && pipeout->mFd == STDOUT_FILENO) 
        {
            if(tcsetpgrp(0, pid) < 0) {
                perror("tcsetpgrp(child)");
                exit(1);
            }
        }

        saphire_restore_signal_default();
        sigchld_block(0);

        if(!statment_tree_exec_cprog(command, nextin2, nextout2, nexterr , argv ,sname, sline))
        {
            return FALSE;
        }

        return TRUE;
    }
    /// 親プロセスの処理 ///
    else {
        if(last_program) *last_program_pid = pid;

        // プログラムはすべて固有のプロセスグループに属する
        if(gAppType != kATOptC) {
            (void)setpgid(pid, pid);
        }

        /// 子へのメモリパイプのバッファ渡しがあって
        /// 子がTSTP食らうとsaphire本体のパイプの書き込みがブロックし
        /// waitpidもできないためデッドロックする。
        /// そのためバッファを書き込むプロセスを用意してsaphire
        /// 本体はバッファの書き込みをしない
        if(mpipefds[1] != -1) {
            int pid2 = fork();
            if(pid2 < 0) {
                perror("fork2");
                exit(1);
            }
            /// writer process ///
            if(pid2 == 0) {             /// 子プロセス
                pid2 = getpid();

                if(gAppType != kATOptC) {
                    (void)setpgid(pid2, pid);
                }

                if(mpipefds[1] != -1) {
                    (void)close(nextin2);
                    BOOL epipe = FALSE;
                    if(!gSigUser && bufsiz_write(mpipefds[1], nextin->mBuffer, 0) < 0)
                    {
                        if(errno != EPIPE ) {
                            perror("write memory pipe");
                            exit(1);
                        }
                    }
                    (void)close(mpipefds[1]);
                }
                exit(0);
            }
            /// 親プロセス ///
            else {
                if(gAppType != kATOptC) {
                    (void)setpgid(pid2, pid);
                }

                if(mpipefds[1] != -1) {
                    (void)close(nextin2);
                    (void)close(mpipefds[1]);
                }

                if(mpipefds2[0] != -1) {
                    (void)close(nextout2);

                    if(!gSigUser) {
                        char buf[BUFSIZ+1];
                        while(1) {
                            if(gKitutukiSigInt) {
                                err_msg("signal interrupt", sname, sline);
                                gKitutukiSigInt = FALSE;
                                return FALSE;
                            }
                            int r = read(mpipefds2[0], buf, BUFSIZ);
                            if(r < 0) {
                                if(errno == EINTR) {
                                    buf[0] = 0;
                                    sWFd_push_back(nextout, buf);
                                    break;
                                }
                                else {
                                    perror("read mpipe");
                                    exit(1);
                                }
                            }
                            buf[r] = 0;

                            sWFd_push_back(nextout, buf);

                            if(r == 0) {
                                break;
                            }
                        }
                    }

                    (void)close(mpipefds2[0]);
                }
            }
            if(gAppType == kATOptC) {
                while(1) {
                    int status;
                    pid2 = waitpid(pid2, &status, WUNTRACED);

                    if(pid2 < 0 && errno == EINTR) {
                    }
                    else if(WIFSTOPPED(status)) {
                        kill(pid2, SIGCONT);
                    }
                    else if(WIFSIGNALED(status)) {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                    else {
                        break;
                    }
                }
            }
            else {
                int status;
                pid2 = waitpid(pid2, &status, WUNTRACED);

                if(WIFSIGNALED(status) || WIFSTOPPED(status))
                {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
            }
        }
        else {
            if(mpipefds2[0] != -1) {
                (void)close(nextout2);

                if(!gSigUser) {
                    char buf[BUFSIZ+1];
                    while(1) {
                        if(gKitutukiSigInt) {
                            err_msg("signal interrupt", sname, sline);
                            gKitutukiSigInt = FALSE;
                            return FALSE;
                        }
                        int r = read(mpipefds2[0], buf, BUFSIZ);
                        if(r < 0) {
                            if(errno == EINTR) {
                                buf[0] = 0;
                                sWFd_push_back(nextout, buf);
                                break;
                            }
                            else {
                                perror("read mpipe");
                                exit(1);
                            }
                        }
                        buf[r] = 0;

                        sWFd_push_back(nextout, buf);

                        if(r == 0) {
                            break;
                        }
                    }
                }

                (void)close(mpipefds2[0]);
            }
        }

        // 毎回waitする
        if(!last_program) {
            if(gAppType == kATOptC) {
                while(1) {
                    int status;
                    pid = waitpid(pid, &status, WUNTRACED);

                    if(pid < 0 && errno == EINTR) {
                    }
                    else if(WIFSTOPPED(status)) {
                        kill(pid, SIGCONT);
                    }
                    else if(WIFSIGNALED(status)) {
                        err_msg("signal interrupt", sname, sline);
                        return FALSE;
                    }
                    else {
                        break;
                    }
                }
            }
            else {
                int status;
                pid = waitpid(pid, &status, WUNTRACED);

                if(WIFSIGNALED(status) || WIFSTOPPED(status)) {
                    err_msg("signal interrupt", sname, sline);
                    return FALSE;
                }
            }
        }

        return TRUE;
    }
}

static void statment_tree_make_argv_make_randomize_file_name(char* result)
{
    while(1) {
        int i;
        for(i=0; i<64; i++) {
            unsigned char c = random() % (0x7a - 0x61) + 0x61;
            char str[2];
            str[0] = c;
            str[1] = 0;
            xstrncat(result, str, PATH_MAX);
            if(i%16 == 0) {
                srandom((unsigned)time(NULL));
            }
        }
        int flg = 0;
        for(i=0; i<vector_size(gTmpFiles); i++) {
            char* fname2 = string_c_str(vector_item(gTmpFiles, i));
            if(strcmp(result, fname2) == 0) {
                flg++;
            }
        }

        if(flg == 0) break;
    }
}

/// argvを作る
static BOOL statment_tree_make_argv(sRunInfo* runinfo, sCommand* command, vector_obj* argv, vector_obj* blocks, sRFd* nextin, sRFd* pipein, char* sname, int sline, vector_obj* psubs, BOOL* psub_security_umask)
{
    sRedirect* redirect = NULL;

    int ii;
    for(ii=0; ii<vector_size(command->mArgs); ii++) {
        sArg* arg = vector_item(command->mArgs, ii);
        switch(arg->mKind) {
            /// string ///
            case 0:
                if(redirect) {
                    if(!string_put_cancelable(redirect->mFName, string_c_str(arg->mBody)))
                    {
                        return FALSE;
                    }
                    redirect = NULL;
                }
                else {
                    vector_add(argv, STRING_NEW(string_c_str(arg->mBody)));
                }
                break;

            /// @ のコマンド展開　///
            case 2: {
                sStatments* statments = arg->mBody;
                sWFd* pipeout = WFD_NEW(-1);
                int rcode = run(runinfo, statments, "quick command expansion", pipeout, pipein, 2, FALSE);

                if(rcode < 0) {
                    sWFd_delete(pipeout);
                    return FALSE;
                }

                if(redirect) {
                    if(!string_put_cancelable(redirect->mFName, pipeout->mBuffer))
                    {
                        sWFd_delete(pipeout);
                        return FALSE;
                    }
                    redirect = NULL;
                    sWFd_delete(pipeout);
                }
                else {
                    string_obj* str = string_new_cancelable(pipeout->mBuffer);
                    if(str) {
                        vector_add(argv, str);
                    }
                    else {
                        sWFd_delete(pipeout);
                        return FALSE;
                    }

                    sWFd_delete(pipeout);
                }
                }
                break;

            /// @@のコマンド展開 ///
            case 4: {
                sAtCommand* at_command = arg->mBody;
                sStatments* statments = at_command->mStatments;
                sWFd* pipeout = WFD_NEW(-1);
                int rcode = run(runinfo, statments, "quick command expansion", pipeout, pipein, 2, FALSE);

                if(rcode < 0) {
                    sWFd_delete(pipeout);
                    return FALSE;
                }

                sRFd* tmp = RFD_NEW3(-1, pipeout->mBuffer, pipeout->mMallocSize); 
                pipeout->mBuffer = NULL;
                sWFd_delete(pipeout);
                while(1) {
                    string_obj* str = STRING_NEW("");
                    int ret = sRFd_read_oneline(tmp, str, at_command->mLineField);

                    if(ret < 0) {
                        string_delete(str);
                        sRFd_delete(tmp);
                        return FALSE;
                    }
                    else if(ret == 1) {
                        string_delete(str);
                        break;
                    }

                    string_chomp(str);
                    if(redirect) {
                        if(!string_put_cancelable(redirect->mFName, string_c_str(str)))
                        {
                            string_delete(str);
                            sRFd_delete(tmp);
                            return FALSE;
                        }
                        string_delete(str);
                        redirect = NULL;
                    }
                    else {
                        vector_add(argv, str);
                    }
                }
                sRFd_delete(tmp);

                }
                break;

            /// ブロック ///
            case 5: {
                vector_add(blocks, sBlock_new2(arg->mBody));
                }
                break;

            /// プロセス置換 ///
            case 6: {
                if(tcsetpgrp(0, getpid()) < 0) {
                    perror("tcsetpgrp(inner command)");
                    exit(1);
                }

                sStatments* statments = arg->mBody;
                sWFd* pipeout = WFD_NEW(-1);
                int rcode = run(runinfo, statments, "process substitution", pipeout, pipein, 2, FALSE);

                if(rcode < 0) {
                    sWFd_delete(pipeout);
                    return FALSE;
                }

                /// 一時ファイルを作る ///
                char fname[PATH_MAX];
                snprintf(fname, PATH_MAX, "%s/saphire_psub2", gTmpDir);
                statment_tree_make_argv_make_randomize_file_name(fname);
                vector_add(gTmpFiles, STRING_NEW(fname));
                
                int fd = open(fname, O_WRONLY|O_TRUNC|O_CREAT, 0600);
                int ret = bufsiz_write(fd, pipeout->mBuffer, 1);
                if(ret < 0)
                {
                    perror("write");
                    exit(1);
                }
                if(ret == 1) {
                    err_msg("signal interrupt!", sname, sline);
                    sWFd_delete(pipeout);
                    gKitutukiSigInt = FALSE;
                    return FALSE;
                }

                (void)close(fd);

                /// md5
                
                if(redirect) {
                    string_put(redirect->mFName, fname);
                    redirect = NULL;
                }
                else {
                    vector_add(argv, STRING_NEW(fname));
                }

                sWFd_delete(pipeout);
                }
                break;

            /// ヘアドキュメント ///
            case 7: {
                sHereDoc* hdoc = arg->mBody;
                string_obj* text = hdoc->mText;

                /// 環境変数展開 ///
                if(hdoc->mExpandEnv) {
                    string_obj* text2 = STRING_NEW("");
                    if(!expand_env(string_c_str(text), text2, pipein, sname, &sline, runinfo->object))
                    {
                        string_delete(text2);
                        return FALSE;
                    }
                    text = text2;
                }

                /// 一時ファイルを作る ///
                char fname[PATH_MAX];
                snprintf(fname, PATH_MAX, "%s/saphire_here_doc", gTmpDir);
                statment_tree_make_argv_make_randomize_file_name(fname);
                vector_add(gTmpFiles, STRING_NEW(fname));

                int f = open(fname, O_WRONLY|O_TRUNC|O_CREAT, 0600);
                int ret = bufsiz_write(f, string_c_str(text), 1);
                if(ret < 0) {
                    perror("write on here document");
                    exit(1);
                }
                if(ret == 1) {
                    err_msg("signal interrupt!", sname, sline);
                    gKitutukiSigInt = FALSE;
                    if(hdoc->mExpandEnv) { string_delete(text); }
                    return FALSE;
                }
                (void)close(f);

                if(hdoc->mExpandEnv) { string_delete(text); }

                if(redirect) {
                    string_put(redirect->mFName, fname);
                    redirect = NULL;
                }
                else {
                    vector_add(argv, STRING_NEW(fname));
                }

                }
                break;

            /// プロセス置換2 ///
            case 8: {
                sStatments* statments = arg->mBody;

                /// 一時ファイルを作る ///
                char fname[PATH_MAX];
                snprintf(fname, PATH_MAX, "%s/saphire_psub", gTmpDir);
                statment_tree_make_argv_make_randomize_file_name(fname);
                vector_add(gTmpFiles, STRING_NEW(fname));
                vector_add(psubs, sPSub_new(fname, statments));

                if(redirect) {
                    string_put(redirect->mFName, fname);
                    redirect = NULL;
                }
                else {
                    vector_add(argv, STRING_NEW(fname));
                }
                *psub_security_umask = TRUE;
                }
                break;

            /// リダイレクト ///
            case 9: {
                sRedirect* redirect2 = sRedirect_new2(arg->mBody);

                vector_add(command->mRedirects, redirect2);

                if(redirect2->mType == kRedirectErrAndOutput) {
                    if(redirect) {
                        err_msg("invalid redirect file name", sname, sline);
                        return FALSE;
                    }
                    else {
                        redirect = NULL;
                    }
                }
                else {
                    if(redirect) {
                        err_msg("invalid redirect file name", sname, sline);
                        return FALSE;
                    }
                    else {
                        redirect = redirect2;
                    }
                }
                }
                break;
        }
    }

    if(redirect) {
        err_msg("redirect file name doesn't exist" , sname, sline);
        return FALSE;
    }

    return TRUE;
}

// 文を処理したい関数。本体。
// 文を回す
// TRUE 正常終了
// FALSE エラー
static BOOL statment_tree(sRunInfo* runinfo, sStatment* statment, int* last_program_pid, sWFd* pipeout, sRFd* pipein, int pipeerr, char* title, int* rcode, char* sname, int sline, vector_obj* psubs) 
{
    sRFd* nextin;
    int nexterr = pipeerr;

    /// パイプ準備 ///
    int exist_globalin = statment->mGlobalPipeIn & 0x0F;
    BOOL exist_stdinpipe = statment->mSTDINPipe;

    if(exist_globalin == 1) {
        nextin = RFD_NEW(-1);
        sRFd_set_buffer(nextin, string_c_str(gGlobalPipe));
    }
    else if(exist_globalin == 4) {
        int num = statment->mGlobalPipeInNum;
        nextin = RFD_NEW(-1);
        sRFd_set_buffer(nextin, string_c_str(gGlobalPipeNum[num]));
    }
    else if(exist_stdinpipe) {
        if(pipein->mFd == -1) {
            nextin = RFD_NEW(-1);
            sRFd_set_buffer(nextin, pipein->mBuffer);
        }
        else {
            sRFd_read_all_to_buffer(pipein);
            nextin = RFD_NEW2(pipein->mFd, pipein->mBuffer);
        }
    }
    else {
        nextin = pipein;
    }

    int exist_globalout = 0;
    int exist_globalout_append = 0;

    /// 回す ///
    int j;
    for(j=0; j<vector_size(statment->mCommands); j++) {
        sCommand* command = vector_item(statment->mCommands, j);

        const BOOL first_program = j == 0;
        const BOOL last_program = j == (vector_size(statment->mCommands)-1);

        /// パイプ ///
        sWFd* nextout;
        if(last_program) {
            exist_globalout = statment->mGlobalPipeOut & 0x0F;
            exist_globalout_append = statment->mGlobalPipeOut & 0xF0;

            if(exist_globalout) {
                nextout = WFD_NEW(-1);
            }
            else {
                nextout = pipeout;
            }
        }
        else {
            nextout = WFD_NEW(-1);
        }

        // argvを作成 sArgから
        vector_obj* argv = VECTOR_NEW(30);
        vector_obj* blocks = VECTOR_NEW(30);
        BOOL psub_security_umask = FALSE;
        if(!statment_tree_make_argv(runinfo, command, argv, blocks, nextin, pipein, sname, sline, psubs, &psub_security_umask)) 
        {
            if(nextout != pipeout) sWFd_delete(nextout);
            if(nextin != pipein) sRFd_delete(nextin);
            int i;
            for(i=0; i<vector_size(argv); i++) {
                string_delete(vector_item(argv, i));
            }
            vector_delete(argv);
            for(i=0; i<vector_size(blocks); i++) {
                sBlock_delete(vector_item(blocks, i));
            }
            vector_delete(blocks);
            return FALSE;
        }

        /// プロセス置換で作る一時ファイルが/tmpに作られるのでパーミッションは制限したい
        char* home = getenv("HOME");
        mode_t umask_before;
        if(psub_security_umask && (home == NULL || *home == 0))
        {
            umask_before = umask(0077);
        }

        if(*runinfo->return_) {
            if(nextout != pipeout) sWFd_delete(nextout);
            if(nextin != pipein) sRFd_delete(nextin);
            int i;
            for(i=0; i<vector_size(argv); i++) {
                string_delete(vector_item(argv, i));
            }
            vector_delete(argv);
            for(i=0; i<vector_size(blocks); i++) {
                sBlock_delete(vector_item(blocks, i));
            }
            vector_delete(blocks);
            if(psub_security_umask && (home == NULL || *home == 0))
            {
                umask(umask_before);
            }
            return TRUE;
        }

        const BOOL input = !first_program || exist_globalin || exist_stdinpipe;
        const BOOL output = !last_program;

        int ii;

        char* arg0 = string_c_str(vector_item(argv, 0));
        sObject* object2;
        sFunction* method;
        sFunction* fun;
        sClass* klass;
        sInnerCommand* inner_fun;
        char* tmp = NULL;
        char* p = arg0;
        while(*p) {
            if(*p == '-' && *(p+1) == '>') {
                tmp = p;
                break;
            }
            else {
                p++;
            }
        }
        BOOL method_class = FALSE;
        if(tmp == arg0) {
            string_obj* str = STRING_NEW("");
            int ret = sRFd_read_oneline(nextin, str, kLF);

            if(ret == 1 || ret < 0) {
                err_msg("invalid address of object", sname, sline);
                string_delete(str);
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }

            void* address = (void*)(unsigned long)strtoll(string_c_str(str), NULL, 16);

            string_delete(str);

            int kind = memchecker_is_enable_mem(address);
            if(kind == kMCObject) {
                object2 = address;
            }
            else {
                err_msg("invalid address of object", sname, sline);
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }

            tmp += 2;
            char* tmp3 = MALLOC(arg0 + strlen(arg0) - tmp + 1);
            xstrncpy(tmp3, tmp, arg0 + strlen(arg0) - tmp + 1);

            if(strcmp(tmp3, "class") == 0) {
                method_class = TRUE;
            }
            else {
                method = hash_item(object2->mMethods, tmp3);

                if(method == NULL) {
                    char str[1024];
                    snprintf(str, 1024, "method(%s) is not found", tmp3);
                    err_msg(str, sname, sline);
                    if(nextout != pipeout) sWFd_delete(nextout);
                    if(nextin != pipein) sRFd_delete(nextin);
                    int i;
                    for(i=0; i<vector_size(argv); i++) {
                        string_delete(vector_item(argv, i));
                    }
                    vector_delete(argv);
                    for(i=0; i<vector_size(blocks); i++) {
                        sBlock_delete(vector_item(blocks, i));
                    }
                    vector_delete(blocks);
                    FREE(tmp3);
                    if(psub_security_umask && (home == NULL || *home == 0))
                    {
                        umask(umask_before);
                    }
                    return FALSE;
                }
            }
            fun = NULL;
            inner_fun = NULL;
            klass = NULL;

            FREE(tmp3);
        }
        else if(tmp) {
            char* tmp2 = MALLOC(tmp - arg0 + 1);
            memcpy(tmp2, arg0, tmp-arg0);
            tmp2[tmp - arg0] = 0;

            if(strcmp(tmp2, "self") == 0) {
                object2 = runinfo->object;
            }
            else {
                object2 = saphire_get_object(tmp2, runinfo->object, TRUE, FALSE);
            }

            if(object2 == NULL) {
                char str[1024];
                snprintf(str, 1024, "object(%s) is not found", tmp2);
                err_msg(str, sname, sline);
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                FREE(tmp2);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }

            tmp += 2;
            char* tmp3 = MALLOC(arg0 + strlen(arg0) - tmp + 1);
            xstrncpy(tmp3, tmp, arg0 + strlen(arg0) - tmp + 1);

            if(strcmp(tmp3, "class") == 0) {
                method_class = TRUE;
            }
            else {
                method = hash_item(object2->mMethods, tmp3);

                if(method == NULL) {
                    char str[1024];
                    snprintf(str, 1024, "method(%s) is not found", tmp3);
                    err_msg(str, sname, sline);
                    if(nextout != pipeout) sWFd_delete(nextout);
                    if(nextin != pipein) sRFd_delete(nextin);
                    int i;
                    for(i=0; i<vector_size(argv); i++) {
                        string_delete(vector_item(argv, i));
                    }
                    vector_delete(argv);
                    for(i=0; i<vector_size(blocks); i++) {
                        sBlock_delete(vector_item(blocks, i));
                    }
                    vector_delete(blocks);
                    FREE(tmp3);
                    FREE(tmp2);
                    if(psub_security_umask && (home == NULL || *home == 0))
                    {
                        umask(umask_before);
                    }
                    return FALSE;
                }
            }
            fun = NULL;
            inner_fun = NULL;
            klass = NULL;

            FREE(tmp3);

            FREE(tmp2);
        }
        else {
            object2 = NULL;
            method = NULL;
            fun = saphire_get_fun(arg0, runinfo->object, TRUE);
            klass = saphire_get_class(arg0, runinfo->object, TRUE);
            inner_fun = hash_item(gInnerCommands, arg0);
        }

        if(fun || inner_fun) {
            command->mKind = kInnerCommand;
        }

        /// クラス ///
        if(klass) {
            klass->input = input;
            klass->output = output;

            if(vector_size(argv) >= 2) {
                BOOL global = FALSE;
                BOOL local = FALSE;
                int i;
                for(i=1; i<vector_size(argv); i++) {
                    char* arg = string_c_str(vector_item(argv, i));
                    if(strcmp(arg, "-l") == 0) {
                        string_delete(vector_item(argv, i));
                        vector_erase(argv, i);
                        i--;
                        continue;
                    }
                    else if(strcmp(arg, "-g") == 0) {
                        string_delete(vector_item(argv, i));
                        vector_erase(argv, i);
                        i--;
                        continue;
                    }
                }
                char* objname = string_c_str(vector_item(argv, 1));

                sObject* object2 = sObject_new(objname, string_c_str(klass->name), runinfo->object, 1);
                if(local) {
                    hash_obj* top_stack
                         = vector_item(gStackFrameObjects, vector_size(gStackFrameObjects)-1);

                    sObject* obj = hash_item(top_stack, objname);
                    if(obj) {
                        err_msg("same name object exists", sname, sline);
                        sObject_delete(object2);
                        sObject_delete(object2);
                        if(nextout != pipeout) sWFd_delete(nextout);
                        if(nextin != pipein) sRFd_delete(nextin);
                        int i;
                        for(i=0; i<vector_size(argv); i++) {
                            string_delete(vector_item(argv, i));
                        }
                        vector_delete(argv);
                        for(i=0; i<vector_size(blocks); i++) {
                            sBlock_delete(vector_item(blocks, i));
                        }
                        vector_delete(blocks);
                        if(psub_security_umask && (home == NULL || *home == 0))
                        {
                            umask(umask_before);
                        }
                        return FALSE;
                    }
                    hash_put(top_stack, objname, object2);
                }
                else if(runinfo->object && !global) {
                    sObject* obj = hash_item(runinfo->object->mObjects, objname);
                    if(obj) {
                        err_msg("same name object exists", sname, sline);
                        sObject_delete(object2);
                        if(nextout != pipeout) sWFd_delete(nextout);
                        if(nextin != pipein) sRFd_delete(nextin);
                        int i;
                        for(i=0; i<vector_size(argv); i++) {
                            string_delete(vector_item(argv, i));
                        }
                        vector_delete(argv);
                        for(i=0; i<vector_size(blocks); i++) {
                            sBlock_delete(vector_item(blocks, i));
                        }
                        vector_delete(blocks);
                        if(psub_security_umask && (home == NULL || *home == 0))
                        {
                            umask(umask_before);
                        }
                        return FALSE;
                    }
                    hash_put(runinfo->object->mObjects, objname, object2);
                }
                else {
                    sObject* obj = hash_item(gObjects, objname);
                    if(obj) {
                        err_msg("same name object exists", sname, sline);
                        sObject_delete(object2);
                        if(nextout != pipeout) sWFd_delete(nextout);
                        if(nextin != pipein) sRFd_delete(nextin);
                        int i;
                        for(i=0; i<vector_size(argv); i++) {
                            string_delete(vector_item(argv, i));
                        }
                        vector_delete(argv);
                        for(i=0; i<vector_size(blocks); i++) {
                            sBlock_delete(vector_item(blocks, i));
                        }
                        vector_delete(blocks);
                        if(psub_security_umask && (home == NULL || *home == 0))
                        {
                            umask(umask_before);
                        }
                        return FALSE;
                    }

                    hash_put(gObjects, objname, object2);
                }

                // スタックを掘る
                sAry* argv2 = sAry_new(30, 1);
                int k;
                for(k=2; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    saphire_push_back_array_element(argv2, STRING_NEW(value));
                }
                sAry* fun_argv_before;
                dig_stack_frame(sname, sline, &fun_argv_before, argv2);

                /// 実行
                int enable_return = runinfo->enable_return;
                runinfo->enable_return = TRUE;

                int enable_break = runinfo->enable_break;
                runinfo->enable_break = FALSE;

                sObject* object = runinfo->object;
                runinfo->object = object2;

                sFunction* running_fun = runinfo->running_fun;
                runinfo->running_fun = NULL;

                sClass* running_class = runinfo->running_class;
                runinfo->running_class = klass;

                *rcode = run(runinfo, klass->statments, title, nextout, nextin, nexterr, FALSE);
                *runinfo->return_ = FALSE; // ユーザー関数が復帰したらreturn終わり
                *runinfo->break_ = FALSE;

                runinfo->enable_return = enable_return;
                runinfo->enable_break = enable_break;
                runinfo->object = object;
                runinfo->running_fun = running_fun;
                runinfo->running_class = running_class;

                // スタックを埋める
                string_obj* sname_strace;
                int line_strace;
                burry_stack_frame(fun_argv_before, &sname_strace, &line_strace);

                if(*rcode < 0) {
                    if(nextout != pipeout) sWFd_delete(nextout);
                    if(nextin != pipein) sRFd_delete(nextin);
                    string_push_back(gErrMsg, string_c_str(sname_strace));
                    string_delete(sname_strace);
                    char tmp[1024];
                    snprintf(tmp, 1024, ":%d\n", line_strace);
                    string_push_back(gErrMsg, tmp);
                    int i;
                    for(i=0; i<vector_size(argv); i++) {
                        string_delete(vector_item(argv, i));
                    }
                    vector_delete(argv);
                    for(i=0; i<vector_size(blocks); i++) {
                        sBlock_delete(vector_item(blocks, i));
                    }
                    vector_delete(blocks);

                    if(psub_security_umask && (home == NULL || *home == 0))
                    {
                        umask(umask_before);
                    }
                    return FALSE;
                }
                else {
                    string_delete(sname_strace);
                }

                /// 終了コード ///
                if(statment->mRCodeReverse) {
                    if(*rcode == 0) 
                        *rcode = 1;
                    else
                        *rcode = 0;
                }

                char buf2[256];
                snprintf(buf2, 256, "%d", *rcode);
                (void)saphire_set_global_var("RCODE", buf2, NULL);
            }
            else {
                err_msg("reuire object name", sname, sline);
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
        }
        /// classメソッド ///
        else if(method_class) {
            if(vector_size(blocks) == 1) {
                // スタックを掘る
                sAry* argv2 = sAry_new(30, 1);
                int k;
                for(k=1; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    saphire_push_back_array_element(argv2, STRING_NEW(value));
                }
                sAry* fun_argv_before;
                dig_stack_frame(sname, sline, &fun_argv_before, argv2);

                /// 実行
                sBlock* block = vector_item(blocks, 0);
                sStatments* statments = block->mStatments;
                
                int enable_return = runinfo->enable_return;
                runinfo->enable_return = TRUE;

                int enable_break = runinfo->enable_break;
                runinfo->enable_break = FALSE;

                sObject* object = runinfo->object;
                runinfo->object = object2;

                sFunction* running_fun = runinfo->running_fun;
                runinfo->running_fun = NULL;

                sClass* running_class = runinfo->running_class;
                runinfo->running_class = NULL;
                
                *rcode = run(runinfo, statments, title, nextout, nextin, nexterr, FALSE);
                *runinfo->return_ = FALSE; // ユーザー関数が復帰したらreturn終わり
                *runinfo->break_ = FALSE;

                runinfo->enable_return = enable_return;
                runinfo->enable_break = enable_break;
                runinfo->object = object;
                runinfo->running_fun = running_fun;
                runinfo->running_class = running_class;

                // スタックを埋める
                string_obj* sname_strace;
                int line_strace;
                burry_stack_frame(fun_argv_before, &sname_strace, &line_strace);

                if(*rcode < 0) {
                    string_push_back(gErrMsg, string_c_str(sname_strace));
                    string_delete(sname_strace);
                    char tmp[1024];
                    snprintf(tmp, 1024, ":%d\n", line_strace);
                    string_push_back(gErrMsg, tmp);

                    if(nextout != pipeout) sWFd_delete(nextout);
                    if(nextin != pipein) sRFd_delete(nextin);
                    int i;
                    for(i=0; i<vector_size(argv); i++) {
                        string_delete(vector_item(argv, i));
                    }
                    vector_delete(argv);
                    for(i=0; i<vector_size(blocks); i++) {
                        sBlock_delete(vector_item(blocks, i));
                    }
                    vector_delete(blocks);
                    if(psub_security_umask && (home == NULL || *home == 0))
                    {
                        umask(umask_before);
                    }
                    return FALSE;
                }
                else {
                    string_delete(sname_strace);
                }

                /// 終了コード ///
                if(statment->mRCodeReverse) {
                    if(*rcode == 0) 
                        *rcode = 1;
                    else
                        *rcode = 0;
                }

                char buf2[256];
                snprintf(buf2, 256, "%d", *rcode);
                (void)saphire_set_global_var("RCODE", buf2, NULL);
            }
            else if(vector_size(argv) >= 2) {
                char* klassname = string_c_str(vector_item(argv, 1));

                // スタックを掘る
                sAry* argv2 = sAry_new(30, 1);
                int k;
                for(k=2; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    saphire_push_back_array_element(argv2, STRING_NEW(value));
                }
                sAry* fun_argv_before;
                dig_stack_frame(sname, sline, &fun_argv_before, argv2);

                //// 実行
                if(runinfo->object) {
                    klass = hash_item(runinfo->object->mClasses, klassname);
                    if(klass == NULL) {
                        sObject* parent = runinfo->object->mParent;
                        while(parent) {
                            klass = hash_item(parent->mClasses, klassname);
                            if(klass) break;
                            parent = parent->mParent;
                        }
                    }
                    if(klass == NULL) {
                        klass = hash_item(gClasses, klassname);
                    }
                }
                else {
                    klass = hash_item(gClasses, klassname);
                }

                if(klass == NULL) {
                    err_msg("class name is not found", sname, sline);
                    if(nextout != pipeout) sWFd_delete(nextout);
                    if(nextin != pipein) sRFd_delete(nextin);
                    int i;
                    for(i=0; i<vector_size(argv); i++) {
                        string_delete(vector_item(argv, i));
                    }
                    vector_delete(argv);
                    for(i=0; i<vector_size(blocks); i++) {
                        sBlock_delete(vector_item(blocks, i));
                    }
                    vector_delete(blocks);
                    if(psub_security_umask && (home == NULL || *home == 0))
                    {
                        umask(umask_before);
                    }
                    return FALSE;
                }

                klass->input = input;
                klass->output = output;
                
                int enable_return = runinfo->enable_return;
                runinfo->enable_return = TRUE;

                int enable_break = runinfo->enable_break;
                runinfo->enable_break = FALSE;

                sObject* object = runinfo->object;
                runinfo->object = object2;

                sFunction* running_fun = runinfo->running_fun;
                runinfo->running_fun = NULL;

                sClass* running_class = runinfo->running_class;
                runinfo->running_class = klass;

                *rcode = run(runinfo, klass->statments, title, nextout, nextin, nexterr, FALSE);
                *runinfo->return_ = FALSE; // ユーザー関数が復帰したらreturn終わり
                *runinfo->break_ = FALSE;

                runinfo->enable_return = enable_return;
                runinfo->enable_break = enable_break;
                runinfo->object = object;
                runinfo->running_fun = running_fun;
                runinfo->running_class = running_class;

                string_obj* sname_strace;
                int line_strace;
                burry_stack_frame(fun_argv_before, &sname_strace, &line_strace);

                if(*rcode < 0) {
                    string_push_back(gErrMsg, string_c_str(sname_strace));
                    string_delete(sname_strace);
                    char tmp[1024];
                    snprintf(tmp, 1024, ":%d\n", line_strace);
                    string_push_back(gErrMsg, tmp);
                    if(nextout != pipeout) sWFd_delete(nextout);
                    if(nextin != pipein) sRFd_delete(nextin);
                    int i;
                    for(i=0; i<vector_size(argv); i++) {
                        string_delete(vector_item(argv, i));
                    }
                    vector_delete(argv);
                    for(i=0; i<vector_size(blocks); i++) {
                        sBlock_delete(vector_item(blocks, i));
                    }
                    vector_delete(blocks);

                    if(psub_security_umask && (home == NULL || *home == 0))
                    {
                        umask(umask_before);
                    }
                    return FALSE;
                }
                else {
                    string_delete(sname_strace);
                }

                /// 終了コード ///
                if(statment->mRCodeReverse) {
                    if(*rcode == 0) 
                        *rcode = 1;
                    else
                        *rcode = 0;
                }

                char buf2[256];
                snprintf(buf2, 256, "%d", *rcode);
                (void)saphire_set_global_var("RCODE", buf2, NULL);
            }
            else {
                err_msg("reuire class name or a block", sname, sline);
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
        }
        /// メソッド ///
        else if(method) {
            method->input = input;
            method->output = output;

            if(!statment_tree_redirect(command, &nextin, &nextout, &nexterr, pipein, pipeout, pipeerr, sname, sline))
            {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
            if(*runinfo->return_) {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return TRUE;
            }

            /// 引数名が指定されていなければスタックフレームを増やす ///
            sAry* fun_argv_before;
            sAry* fun_argv;
            if(strcmp(string_c_str(method->arg_name), "") == 0) {
                // スタックを掘る
                int k;
                sAry* argv2 = sAry_new(30, 1);
                for(k=1; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    saphire_push_back_array_element(argv2, STRING_NEW(value));
                }
                dig_stack_frame(sname, sline, &fun_argv_before, argv2);
                fun_argv = NULL;
            }
            /// 引数名が指定されていなければスタックフレームは増やさない
            else {
                /// 前の引数を保存 ///
                fun_argv_before = hash_item(gArrays, string_c_str(method->arg_name));

                /// 引数を渡す ///
                int k;
                fun_argv = sAry_new(30, 1);
                for(k=1; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    saphire_push_back_array_element(fun_argv, STRING_NEW(value));
                }
                
                hash_put(gArrays, string_c_str(method->arg_name), fun_argv);

                /// スタックフレームの記録を付ける
                vector_add(gStackTraceFName, STRING_NEW(sname));
                vector_add(gStackTraceLineNum, (void*)sline);
            }

            /// 実行 ///
            int enable_return = runinfo->enable_return;
            runinfo->enable_return = TRUE;

            int enable_break = runinfo->enable_break;
            runinfo->enable_break = FALSE;

            sObject* object = runinfo->object;
            runinfo->object = object2;

            sFunction* running_fun = runinfo->running_fun;
            runinfo->running_fun = method;

            vector_obj* parent_blocks = runinfo->parent_blocks;
            runinfo->parent_blocks = blocks;

            *rcode = run(runinfo, method->statments, title, nextout, nextin, nexterr, FALSE);
            *runinfo->break_ = FALSE;

            runinfo->enable_return = enable_return;
            runinfo->enable_break = enable_break;
            runinfo->object = object;
            runinfo->running_fun = running_fun;
            runinfo->parent_blocks = parent_blocks;

            if(strcmp(string_c_str(method->arg_name), "") == 0) {
                *runinfo->return_ = FALSE; // ユーザー関数が復帰したらreturn終わり
            }
            else {
                if(runinfo->enable_return && *runinfo->return_) {
                    if(nextout != pipeout) sWFd_delete(nextout);
                    if(nextin != pipein) sRFd_delete(nextin);
                    int i;
                    for(i=0; i<vector_size(argv); i++) {
                        string_delete(vector_item(argv, i));
                    }
                    vector_delete(argv);
                    for(i=0; i<vector_size(blocks); i++) {
                        sBlock_delete(vector_item(blocks, i));
                    }
                    vector_delete(blocks);
                    if(psub_security_umask && (home == NULL || *home == 0))
                    {
                        umask(umask_before);
                    }
                    return TRUE;
                }
            }

            string_obj* sname_strace;
            int line_strace;
            if(fun_argv == NULL) {
                burry_stack_frame(fun_argv_before, &sname_strace, &line_strace);
            }
            else {
                int k;
                sAry_delete(fun_argv);
                
                if(fun_argv_before) {
                    hash_put(gArrays, string_c_str(method->arg_name)
                                , fun_argv_before);
                }
                else {
                    hash_erase(gArrays, string_c_str(method->arg_name));
                }

                /// スタックフレームの記録を消す
                sname_strace = vector_pop_back(gStackTraceFName);
                line_strace = (int)vector_pop_back(gStackTraceLineNum);
            }

            if(*rcode < 0) {
                string_push_back(gErrMsg, string_c_str(sname_strace));
                string_delete(sname_strace);
                char tmp[1024];
                snprintf(tmp, 1024, ":%d\n", line_strace);
                string_push_back(gErrMsg, tmp);
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
            else {
                string_delete(sname_strace);
            }

            /// 終了コード ///
            if(statment->mRCodeReverse) {
                if(*rcode == 0) 
                    *rcode = 1;
                else
                    *rcode = 0;
            }

            char buf2[256];
            snprintf(buf2, 256, "%d", *rcode);
            (void)saphire_set_global_var("RCODE", buf2, NULL);
        }

        /// ユーザーコマンド ///
        else if(fun) {
            fun->input = input;
            fun->output = output;

            if(!statment_tree_redirect(command, &nextin, &nextout, &nexterr, pipein, pipeout, pipeerr, sname, sline))
            {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
            if(*runinfo->return_) {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return TRUE;
            }

            /// 引数名が指定されていなければスタックフレームを増やす ///
            sAry* fun_argv_before;
            sAry* fun_argv;
            if(strcmp(string_c_str(fun->arg_name), "") == 0) {
                // スタックを掘る
                sAry* argv2 = sAry_new(30, 1);
                int k;
                for(k=1; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    saphire_push_back_array_element(argv2, STRING_NEW(value));
                }
                dig_stack_frame(sname, sline, &fun_argv_before, argv2);
                fun_argv = NULL;
            }
            /// 引数名が指定されていなければスタックフレームは増やさない
            else {
                /// 前の引数を保存 ///
                fun_argv_before = hash_item(gArrays, string_c_str(fun->arg_name));

                /// 引数を渡す ///
                int k;
                fun_argv = sAry_new(30, 1);
                for(k=1; k<vector_size(argv); k++) {
                    char* value = string_c_str(vector_item(argv, k));
                    saphire_push_back_array_element(fun_argv, STRING_NEW(value));
                }
                
                hash_put(gArrays, string_c_str(fun->arg_name), fun_argv);

                /// スタックフレームの記録を付ける
                vector_add(gStackTraceFName, STRING_NEW(sname));
                vector_add(gStackTraceLineNum, (void*)sline);
            }

            /// 実行 ///
            int enable_return = runinfo->enable_return;
            runinfo->enable_return = TRUE;

            int enable_break = runinfo->enable_break;
            runinfo->enable_break = FALSE;

            sFunction* running_fun = runinfo->running_fun;
            runinfo->running_fun = fun;

            vector_obj* parent_blocks = runinfo->parent_blocks;
            runinfo->parent_blocks = blocks;

            *rcode = run(runinfo, fun->statments, title, nextout, nextin, nexterr, FALSE);
            *runinfo->break_ = FALSE;

            runinfo->enable_return = enable_return;
            runinfo->enable_break = enable_break;
            runinfo->running_fun = running_fun;
            runinfo->parent_blocks = parent_blocks;

            if(strcmp(string_c_str(fun->arg_name), "") == 0) {
                *runinfo->return_ = FALSE; // ユーザー関数が復帰したらreturn終わり
            }
            else {
                if(runinfo->enable_return && *runinfo->return_) {
                    if(nextout != pipeout) sWFd_delete(nextout);
                    if(nextin != pipein) sRFd_delete(nextin);
                    int i;
                    for(i=0; i<vector_size(argv); i++) {
                        string_delete(vector_item(argv, i));
                    }
                    vector_delete(argv);
                    for(i=0; i<vector_size(blocks); i++) {
                        sBlock_delete(vector_item(blocks, i));
                    }
                    vector_delete(blocks);
                    if(psub_security_umask && (home == NULL || *home == 0))
                    {
                        umask(umask_before);
                    }
                    return TRUE;
                }
            }

            string_obj* sname_strace;
            int line_strace;
            if(fun_argv == NULL) {
                burry_stack_frame(fun_argv_before, &sname_strace, &line_strace);
            }
            else {
                /// 引数を消す ///
                sAry_delete(hash_item(gArrays, string_c_str(fun->arg_name)));

                /// 前の引数を元に戻す ///
                if(fun_argv_before) {
                    hash_put(gArrays, string_c_str(fun->arg_name)
                                , fun_argv_before);
                }
                else {
                    hash_erase(gArrays, string_c_str(fun->arg_name));
                }

                /// スタックフレームの記録を消す
                sname_strace = vector_pop_back(gStackTraceFName);
                line_strace = (int)vector_pop_back(gStackTraceLineNum);
            }

            if(*rcode < 0) {
                string_push_back(gErrMsg, string_c_str(sname_strace));
                string_delete(sname_strace);
                char tmp[1024];
                snprintf(tmp, 1024, ":%d\n", line_strace);
                string_push_back(gErrMsg, tmp);
                
                if(nextin != pipein) sRFd_delete(nextin);
                if(nextout != pipeout) sWFd_delete(nextout);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
            else {
                string_delete(sname_strace);
            }

            /// 終了コード ///
            if(statment->mRCodeReverse) {
                if(*rcode == 0) 
                    *rcode = 1;
                else
                    *rcode = 0;
            }

            char buf2[256];
            snprintf(buf2, 256, "%d", *rcode);
            (void)saphire_set_global_var("RCODE", buf2, NULL);
        }

        /// 外部登録の内部コマンド ///
        else if(inner_fun) {
            if(!statment_tree_redirect(command, &nextin, &nextout, &nexterr, pipein, pipeout, pipeerr, sname, sline))
            {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
            if(*runinfo->return_) {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return TRUE;
            }

            *rcode = 1; // 初期値は1。コマンドが成功したら0を入れる

            if((gAppType == kATCursesApp 
                    || gAppType == kATConsoleApp))
            {
                if(tcsetpgrp(0, getpid()) < 0) {
                    perror("tcsetpgrp(inner command)");
                    exit(1);
                }
            }

            BOOL r = inner_fun->mFun(rcode, argv, blocks, runinfo->parent_blocks
                        , nextout, nextin, nexterr
                        , title, input, sname, sline);

            /// 終了コード ///
            if(statment->mRCodeReverse) {
                if(*rcode == 0) 
                    *rcode = 1;
                else
                    *rcode = 0;
            }

            char buf[256];
            snprintf(buf, 256, "%d", *rcode);
            (void)saphire_set_global_var("RCODE", buf, NULL);

            if(!r) {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
        }

        /// 内部コマンド ///
        else if(command->mKind != kCommand) {
//printf("command->mKind %d %s\n", command->mKind, gStatmentKindStrs[command->mKind]);
            if(!statment_tree_redirect(command, &nextin, &nextout, &nexterr, pipein, pipeout, pipeerr, sname, sline))
            {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
            if(*runinfo->return_) {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return TRUE;
            }

            *rcode = 1; // 初期値は1。コマンドが成功したら0を入れる

            if((gAppType == kATCursesApp 
                    || gAppType == kATConsoleApp))
            {
                if(tcsetpgrp(0, getpid()) < 0) {
                    perror("tcsetpgrp inner command");
                    exit(1);
                }
            }

            BOOL r = statment_tree_internal_commands(runinfo, command, rcode, argv, blocks, nextout, nextin, nexterr, j, title, input, sname, sline);

            /// 終了コード ///
            if(statment->mRCodeReverse) {
                if(*rcode == 0) 
                    *rcode = 1;
                else
                    *rcode = 0;
            }

            char buf[256];
            snprintf(buf, 256, "%d", *rcode);
            (void)saphire_set_global_var("RCODE", buf, NULL);
            if(!r) {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
        }

        /// 外部コマンド ///
        else {
//puts("外部コマンド");
            /// バッファーをフラッシュしておく
            if(nextout == pipeout) {
                if(!sWFd_flash(nextout)) {
                    if(nextout != pipeout) sWFd_delete(nextout);
                    if(nextin != pipein) sRFd_delete(nextin);
                    int i;
                    for(i=0; i<vector_size(argv); i++) {
                        string_delete(vector_item(argv, i));
                    }
                    vector_delete(argv);
                    for(i=0; i<vector_size(blocks); i++) {
                        sBlock_delete(vector_item(blocks, i));
                    }
                    vector_delete(blocks);
                    if(psub_security_umask && (home == NULL || *home == 0))
                    {
                        umask(umask_before);
                    }
                    return FALSE;
                }
            }
/*
            char fname[PATH_MAX];
            snprintf(fname, PATH_MAX, "%s/saphire_err", gTmpDir);
            statment_tree_make_argv_make_randomize_file_name(fname);
            vector_add(gTmpFiles, STRING_NEW(fname));
            
            int fd = open(fname, O_WRONLY|O_TRUNC|O_CREAT, 0600);
            if(fd < 0) {
                char buf[PATH_MAX];
                snprintf(buf, PATH_MAX, "open %s is err1\n", fname);
                err_msg(buf, sname, sline);
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
            int nexterr_before = nexterr;
            nexterr = fd;
*/

            /// リードバッファが空じゃないなら
            if(nextin->mFd != -1 && nextin->mBuffer[0] != 0)
            {
                int ret = sRFd_read_all_to_buffer(nextin);

                if(ret < 0) {
                    if(nextout != pipeout) sWFd_delete(nextout);
                    if(nextin != pipein) sRFd_delete(nextin);
                    int i;
                    for(i=0; i<vector_size(argv); i++) {
                        string_delete(vector_item(argv, i));
                    }
                    vector_delete(argv);
                    for(i=0; i<vector_size(blocks); i++) {
                        sBlock_delete(vector_item(blocks, i));
                    }
                    vector_delete(blocks);
                    if(psub_security_umask && (home == NULL || *home == 0))
                    {
                        umask(umask_before);
                    }
                    return FALSE;
                }

                nextin->mFd = -1;

                if(!statment_tree_external_command(runinfo, statment, argv, nextin, nextout, nexterr, pipeout, command, last_program, last_program_pid, sname, sline, blocks, pipein, psubs))
                {
                    if(nextout != pipeout) sWFd_delete(nextout);
                    if(nextin != pipein) sRFd_delete(nextin);
                    int i;
                    for(i=0; i<vector_size(argv); i++) {
                        string_delete(vector_item(argv, i));
                    }
                    vector_delete(argv);
                    for(i=0; i<vector_size(blocks); i++) {
                        sBlock_delete(vector_item(blocks, i));
                    }
                    vector_delete(blocks);
                    return FALSE;
                }
            }
            else {
                if(!statment_tree_external_command(runinfo, statment, argv, nextin, nextout, nexterr, pipeout, command, last_program, last_program_pid, sname, sline, blocks, pipein, psubs))
                {
                    if(nextout != pipeout) sWFd_delete(nextout);
                    if(nextin != pipein) sRFd_delete(nextin);
                    int i;
                    for(i=0; i<vector_size(argv); i++) {
                        string_delete(vector_item(argv, i));
                    }
                    vector_delete(argv);
                    for(i=0; i<vector_size(blocks); i++) {
                        sBlock_delete(vector_item(blocks, i));
                    }
                    vector_delete(blocks);
                    if(psub_security_umask && (home == NULL || *home == 0))
                    {
                        umask(umask_before);
                    }
                    return FALSE;
                }
            }
            //nexterr = nexterr_before;
            //(void)close(fd);

            if(*runinfo->return_) {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                int i;
                for(i=0; i<vector_size(argv); i++) {
                    string_delete(vector_item(argv, i));
                }
                vector_delete(argv);
                for(i=0; i<vector_size(blocks); i++) {
                    sBlock_delete(vector_item(blocks, i));
                }
                vector_delete(blocks);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return TRUE;
            }
        }

        int i;
        for(i=0; i<vector_size(argv); i++) {
            string_delete(vector_item(argv, i));
        }
        vector_delete(argv);
        for(i=0; i<vector_size(blocks); i++) {
            sBlock_delete(vector_item(blocks, i));
        }
        vector_delete(blocks);

        /// パイプを閉める ///
        if(nextin->mFd != -1 && nextin != pipein) {
            if(!sRFd_close(nextin)) {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
        }
        if(nextout->mFd != -1 && nextout != pipeout) {
            if(!sWFd_close(nextout)) {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
        }
        if(nexterr != -1 && nexterr != pipeerr && nexterr != 2 && nexterr != nextout->mFd) {
            if(close(nexterr) < 0) {
                if(nextout != pipeout) sWFd_delete(nextout);
                if(nextin != pipein) sRFd_delete(nextin);
                if(psub_security_umask && (home == NULL || *home == 0))
                {
                    umask(umask_before);
                }
                return FALSE;
            }
        }

        /// 出力処理
        if(last_program) {
            /// グローバルパイプへの出力 ///
            if(exist_globalout == 1) {
                if(exist_globalout_append) {
                    if(!string_push_back_cancelable(gGlobalPipe, nextout->mBuffer))
                    {
                        if(nextout != pipeout) sWFd_delete(nextout);
                        if(nextin != pipein) sRFd_delete(nextin);
                        if(psub_security_umask && (home == NULL || *home == 0))
                        {
                            umask(umask_before);
                        }
                        return FALSE;
                    }
                }
                else {
                    if(!string_put_cancelable(gGlobalPipe, nextout->mBuffer)) 
                    {
                        if(nextout != pipeout) sWFd_delete(nextout);
                        if(nextin != pipein) sRFd_delete(nextin);
                        if(psub_security_umask && (home == NULL || *home == 0))
                        {
                            umask(umask_before);
                        }
                        return FALSE;
                    }
                }
                *nextout->mBuffer = 0;
            }
            else if(exist_globalout == 4) {
                int num = statment->mGlobalPipeOutNum;
                if(exist_globalout_append) {
                    if(!string_push_back_cancelable(gGlobalPipeNum[num], nextout->mBuffer))
                    {
                        if(nextout != pipeout) sWFd_delete(nextout);
                        if(nextin != pipein) sRFd_delete(nextin);
                        if(psub_security_umask && (home == NULL || *home == 0))
                        {
                            umask(umask_before);
                        }
                        return FALSE;
                    }
                }
                else {
                    if(!string_put_cancelable(gGlobalPipeNum[num], nextout->mBuffer))
                    {
                        if(nextout != pipeout) sWFd_delete(nextout);
                        if(nextin != pipein) sRFd_delete(nextin);
                        if(psub_security_umask && (home == NULL || *home == 0))
                        {
                            umask(umask_before);
                        }
                        return FALSE;
                    }
                }
                *nextout->mBuffer = 0;
            }
        }
        else {
            /// 次の入力先を決める
            if(nextout->mFd == -1) {
                if(nextin != pipein) sRFd_delete(nextin);
                nextin = sRFd_new_from_wfd(-1, nextout);
                sWFd_delete(nextout);
                nextout = WFD_NEW(-1);
            }
            else {
                if(nextin != pipein) sRFd_delete(nextin);
                nextin = RFD_NEW(-1);
            }
        }

        /// 大域脱出
        if(runinfo->enable_return && *runinfo->return_) {
            if(nextout != pipeout) sWFd_delete(nextout);
            if(nextin != pipein) sRFd_delete(nextin);
            if(psub_security_umask && (home == NULL || *home == 0))
            {
                umask(umask_before);
            }
            return TRUE;
        }
        if(runinfo->enable_break && *runinfo->break_) {
            if(nextout != pipeout) sWFd_delete(nextout);
            if(nextin != pipein) sRFd_delete(nextin);
            if(psub_security_umask && (home == NULL || *home == 0))
            {
                umask(umask_before);
            }
            return TRUE;
        }

        if(nextout != pipeout) sWFd_delete(nextout);

        if(psub_security_umask && (home == NULL || *home == 0))
        {
            umask(umask_before);
        }


        if(gSigUser) {
            gSigUser = FALSE;
            err_msg("command not found", sname, sline);
            if(nextin != pipein) sRFd_delete(nextin);
            return FALSE;
        }
    }

    if(nextin != pipein) sRFd_delete(nextin);

    return TRUE;
}

// 消えた子供を回収したい関数
void run_wait_cprogs(sStatment* statment, int* rcode, sWFd* pipeout, int last_program_pid)
{
    /// - バックグラウンド - ///
    if((gAppType == kATCursesApp || gAppType == kATConsoleApp) 
        && statment->mBackground) 
    {
        sJob* job = sJob_new();

        job->mPGroup = last_program_pid;
        string_put(job->mName, string_c_str(statment->mTitle));

        vector_add(gJobs, job);
    }

    /// - フォアグラウンド - ///
    else {
        sigchld_block(0);       // ブロック解除
        int status = 0;

        if(gAppType == kATOptC) {
            while(1) {
                pid_t pid = waitpid(last_program_pid, &status, WUNTRACED);
                if(pid < 0 && errno == EINTR) {
                }
                else if(WIFSTOPPED(status)) {
                    kill(pid, SIGCONT);
                }
                else {
                    break;
                }
            }
        }
        else {
            pid_t pid = waitpid(last_program_pid, &status, WUNTRACED);

            /// シグナルの配送により終了した場合 ///
            if(WIFSIGNALED(status)) {
                *rcode = WTERMSIG(status) + 128;

                /// 終了コードの反転 ///
                if(statment->mRCodeReverse) {
                    if(*rcode == 0) {
                        *rcode = 1;
                    }
                    else {
                        *rcode = 0;
                    }
                }

                char buf[256];
                snprintf(buf, 256, "%d", *rcode);
                (void)saphire_set_global_var("RCODE", buf, NULL);
            }
            /// 終了した場合 ///
            else if(WIFEXITED(status)) {
                *rcode = WEXITSTATUS(status);

                /// 終了コードの反転 ///
                if(statment->mRCodeReverse) {
                    if(*rcode == 0) {
                        *rcode = 1;
                    }
                    else {
                        *rcode = 0;
                    }
                }

                char buf[256];
                snprintf(buf, 256, "%d", *rcode);
                (void)saphire_set_global_var("RCODE", buf, NULL);
            }
            /// シグナルの配送により停止した場合 ///
            else if(WIFSTOPPED(status)) {
                int sig = WSTOPSIG(status) + 128;

                *rcode = sig;

                /// 終了コードの反転 ///
                if(statment->mRCodeReverse) {
                    if(*rcode == 0) {
                        *rcode = 1;
                    }
                    else {
                        *rcode = 0;
                    }
                }

                char buf[256];
                snprintf(buf, 256, "%d", *rcode);
                (void)saphire_set_global_var("RCODE", buf, NULL);

                sJob* job = sJob_new();

                job->mPGroup = last_program_pid; 
                string_put(job->mName, string_c_str(statment->mTitle));
                vector_add(gJobs, job);

                if(gAppType == kATCursesApp 
                    || gAppType == kATConsoleApp)
                {
                    job->mTty = MALLOC(sizeof(struct termios));
                    tcgetattr(STDIN_FILENO, job->mTty);
                }
            }

            if((gAppType == kATCursesApp || gAppType == kATConsoleApp)
                && pipeout->mFd == STDOUT_FILENO) 
            {
                /// saphireを端末の前面に出す ///
                if(tcsetpgrp(0, getpid()) < 0) {
                    perror("tcsetpgrp(saphire)");
                    exit(1);
                }
            }
        }
    }
}

// 複文を実行したい関数。本体。
int run(sRunInfo* runinfo, sStatments* statments, char* title, sWFd* pipeout, sRFd* pipein, int pipeerr, BOOL try_)
{
    int rcode = 0;

    int i;
    for(i=0; i<vector_size(statments->mStatments); i++) {
        sStatment* statment = vector_item(statments->mStatments, i);

        int sline = statment->mLine;
        char* sname = string_c_str(statment->mFName);

        /// $によるコマンド展開
        if(statment->mNotEvaled) {
            string_obj* cmdline = STRING_NEW("");
            if(expand_env(string_c_str(statment->mNotEvaled), cmdline, pipein, sname, &sline, runinfo->object)) 
            {
                sStatments* estatments = STATMENTS_NEW();

                if(parse(string_c_str(cmdline), sname, &sline, estatments)) 
                {
                    string_delete(cmdline);
                    rcode = run(runinfo, estatments, title, pipeout, pipein, pipeerr, FALSE);
                    sStatments_delete(estatments);

                    if(rcode < 0) {
                    /*
                        /// saphireを前面に出す ///
                        if(tcsetpgrp(0, getpid()) < 0) {
                            perror("tcsetpgrp a");
                            return FALSE;
                        }
                    */
                        return rcode;
                    }
                }
                else {
                    sStatments_delete(estatments);
                    string_delete(cmdline);
                    /*
                    /// saphireを前面に出す ///
                    if(tcsetpgrp(0, getpid()) < 0) {
                        perror("tcsetpgrp a");
                        return FALSE;
                    }
                    */
                    return -1;
                }
            }
            else {
                string_delete(cmdline);
/*
                /// saphireを前面に出す ///
                if(tcsetpgrp(0, getpid()) < 0) {
                    perror("tcsetpgrp a");
                    return FALSE;
                }
*/
                return -1;
            }
        }
        /// パースされている文を実行
        else {
            sigchld_block(1);

            vector_obj* psubs = VECTOR_NEW(30);
            int last_program_pid = -1;
            BOOL r = statment_tree(runinfo, statment
                          , &last_program_pid
                          , pipeout, pipein, pipeerr
                          , title, &rcode
                          , sname, sline
                          , psubs);

            /// プロセス回収 ///
            if(last_program_pid != -1) {
                run_wait_cprogs(statment, &rcode, pipeout, last_program_pid);
            }

            /// ランタイムエラー ///
            if(r == FALSE) {
                int i;
                for(i=0; i<vector_size(psubs); i++) {
                    sPSub_delete(vector_item(psubs, i));
                }
                vector_delete(psubs);
/*
                /// saphireを前面に出す ///
                if(tcsetpgrp(0, getpid()) < 0) {
                    perror("tcsetpgrp a");
                    return FALSE;
                }
*/
                return -1;
            }
            // reutrn
            if(*runinfo->return_) {
                int i;
                for(i=0; i<vector_size(psubs); i++) {
                    sPSub_delete(vector_item(psubs, i));
                }
                vector_delete(psubs);
/*
                /// saphireを前面に出す ///
                if(tcsetpgrp(0, getpid()) < 0) {
                    perror("tcsetpgrp a");
                    return FALSE;
                }
*/
                return rcode;
            }

            /// プロセス置換を実行する ///
            int l;
            for(l=0; l<vector_size(psubs); l++) {
                sPSub* psub = vector_item(psubs, l);

                int f = open(string_c_str(psub->mFName), O_RDONLY);

                sRFd* nextin2 = RFD_NEW(f);
                int rcode = run(runinfo, psub->mStatments, "process substitution2", pipeout, nextin2, 2, FALSE);

                (void)close(f);
                sRFd_delete(nextin2);

                if(rcode < 0) {
                    int i;
                    for(i=l; i<vector_size(psubs); i++) {
                        sPSub_delete(vector_item(psubs, i));
                    }
                    vector_delete(psubs);
                    return rcode;
                }

                sPSub_delete(psub);
            }
            vector_delete(psubs);
        }

        /// 行末の処理 ///
        if(gSigUser) {
            gSigUser = FALSE;
            err_msg("command not found", sname, sline);
/*
            /// saphireを前面に出す ///
            if(tcsetpgrp(0, getpid()) < 0) {
                perror("tcsetpgrp a");
                return FALSE;
            }
*/
            return -1;
        }
        else if(rcode == 128+SIGINT || gKitutukiSigInt) {
            err_msg("signal interrupt!", sname, sline);
            gKitutukiSigInt = FALSE;
/*
            /// saphireを前面に出す ///
            if(tcsetpgrp(0, getpid()) < 0) {
                perror("tcsetpgrp a");
                return FALSE;
            }
*/
            return -1;
        }
        else if(statment->mTerminated == kTOrOr && rcode == 0) {
            while(i<vector_size(statments->mStatments) 
                    && statment->mTerminated != kTNormal) 
            {
                i++;
                statment = vector_item(statments->mStatments, i);
            }
        }
        else if(statment->mTerminated == kTAndAnd && rcode != 0) {
            while(i<vector_size(statments->mStatments) 
                    && statment->mTerminated != kTNormal) 
            {
                i++;
                statment = vector_item(statments->mStatments, i);
            }
        }
        else if(try_ && rcode != 0) {
/*
            /// saphireを前面に出す ///
            if(tcsetpgrp(0, getpid()) < 0) {
                perror("tcsetpgrp a");
                return FALSE;
            }
*/
            return -1;
        }

        /// 大域脱出
        if(runinfo->enable_return && *runinfo->return_) {
/*
            /// saphireを前面に出す ///
            if(tcsetpgrp(0, getpid()) < 0) {
                perror("tcsetpgrp a");
                return FALSE;
            }
*/
            return rcode;
        }
        if(runinfo->enable_break && *runinfo->break_) {
/*
            /// saphireを前面に出す ///
            if(tcsetpgrp(0, getpid()) < 0) {
                perror("tcsetpgrp a");
                return FALSE;
            }
*/
            return rcode;
        }
    }

    /// 空文の場合 ///
    if(vector_size(statments->mStatments) == 0) {
        /// CTRL-Cが押された ///
        if(gKitutukiSigInt) {
/*
            /// saphireを前面に出す ///
            if(tcsetpgrp(0, getpid()) < 0) {
                perror("tcsetpgrp a");
                return FALSE;
            }
*/
            gKitutukiSigInt = FALSE;
            return SIGINT + 128;
        }
    }
/*
    /// saphireを前面に出す ///
    if(tcsetpgrp(0, getpid()) < 0) {
        perror("tcsetpgrp a");
        return FALSE;
    }
*/

    return rcode;
}

///////////////////////////////////////////////////
// ファイル読み込み
///////////////////////////////////////////////////
// -1ならエラーメッセージがgErrMsgに入っているので出力してください

int saphire_load(char* fname, sWFd* pipeout, sRFd* pipein, int pipeerr)
{
    string_obj* str = STRING_NEW("");

    int f = open(fname, O_RDONLY);
    if(f < 0) {
        string_delete(str);
        return -1;
    }

    char buf[BUFSIZ];
    while(1) {
        int size = read(f, buf, BUFSIZ-1);
        if(size == 0) {
            break;
        }
        if(size < 0) {
            string_delete(str);
            (void)close(f);
            return -1;
        }

        buf[size] = 0;

        string_push_back(str, buf);
    }
    (void)close(f);
    int ret = saphire_shell(string_c_str(str), fname, pipeout, pipein, pipeerr);
    string_delete(str);

    return ret;

}

///////////////////////////////////////////////////
// シェル実行
///////////////////////////////////////////////////

// -1ならエラーメッセージがgErrMsgに入っているので出力してください
int saphire_shell(char* command, char* fname, sWFd* pipeout, sRFd* pipein, int pipeerr)
{
    gKitutukiSigInt = FALSE;

    string_obj* fname2;
    if(fname == NULL) {
        fname2 = STRING_NEW(command);
    }
    else {
        fname2 = STRING_NEW(fname);
    }

    int sline = 1;
    string_obj* str = STRING_NEW("");
    if(!delete_comments(command, str, " ", string_c_str(fname2), &sline)) {
        string_delete(fname2);
        string_delete(str);
        return -1;
    }

    sline = 1;

    sStatments* statments = STATMENTS_NEW();
    if(!parse(string_c_str(str), string_c_str(fname2), &sline, statments)) {
        string_delete(fname2);
        string_delete(str);
        sStatments_delete(statments);

        return -1;
    }
    else {
        string_delete(str);

        /// 実行 ///
        int return_ = FALSE;
        int break_ = FALSE;

        sRunInfo runinfo;
        runinfo.return_ = &return_;
        runinfo.enable_return = TRUE;
        runinfo.break_ = &break_;
        runinfo.enable_break = FALSE;
        runinfo.parent_blocks = NULL;
        runinfo.envs = VECTOR_NEW(10);
        runinfo.object = NULL;
        runinfo.running_fun = NULL;
        runinfo.running_class = NULL;

        int rcode = run(&runinfo, statments, string_c_str(fname2), pipeout, pipein, pipeerr, FALSE);

        string_delete(fname2);
        sStatments_delete(statments);

        int i;
        for(i=0; i<vector_size(runinfo.envs); i++) {
            string_delete(vector_item(runinfo.envs, i));
        }
        vector_delete(runinfo.envs);

        return rcode;
    }
}

// コマンド（文字列）を実行して結果をresultに入れたい関数
// -1ならエラーメッセージがgErrMsgに入っているので出力してください
int saphire_shell3(sWFd* pipeout, char* command, char* title, sRFd* pipein)
{
    int rcode = saphire_shell(command, title, pipeout, pipein, STDERR_FILENO);
    if(!sWFd_flash(pipeout)) {
        return -1;
    }

    if(rcode < 0) {
        return -1;
    }

    return rcode;
}

// -1ならエラーメッセージがgErrMsgに入っているので出力してください
int saphire_shell4(char* command, char* fname, sWFd* pipeout, sRFd* pipein, int pipeerr, sObject* object)
{
    gKitutukiSigInt = FALSE;

    string_obj* fname2;
    if(fname == NULL) {
        fname2 = STRING_NEW(command);
    }
    else {
        fname2 = STRING_NEW(fname);
    }

    int sline = 1;
    string_obj* str = STRING_NEW("");
    if(!delete_comments(command, str, " ", string_c_str(fname2), &sline)) {
        string_delete(fname2);
        string_delete(str);
        return -1;
    }

    sline = 1;

    sStatments* statments = STATMENTS_NEW();
    if(!parse(string_c_str(str), string_c_str(fname2), &sline, statments)) {
        string_delete(fname2);
        string_delete(str);
        sStatments_delete(statments);

        return -1;
    }
    else {
        string_delete(str);

        /// 実行 ///
        int return_ = FALSE;
        int break_ = FALSE;

        sRunInfo runinfo;
        runinfo.return_ = &return_;
        runinfo.break_ = &break_;
        runinfo.enable_return = TRUE;
        runinfo.enable_break = FALSE;
        runinfo.parent_blocks = NULL;
        runinfo.envs = VECTOR_NEW(10);
        runinfo.object = object;
        runinfo.running_fun = NULL;
        runinfo.running_class = NULL;

        int rcode = run(&runinfo, statments, string_c_str(fname2), pipeout, pipein, pipeerr, FALSE);

        /// 解放 ///
        string_delete(fname2);
        sStatments_delete(statments);
        int i;
        for(i=0; i<vector_size(runinfo.envs); i++) {
            string_delete(vector_item(runinfo.envs, i));
        }
        vector_delete(runinfo.envs);

        return rcode;
    }
}

// コマンド（文字列）を実行して結果をresultに入れたい関数
// -1ならエラーメッセージがgErrMsgに入っているので出力してください
int saphire_shell5(sWFd* pipeout, char* command, char* title, sRFd* pipein, sObject* object)
{
    int rcode = saphire_shell4(command, title, pipeout, pipein, STDERR_FILENO, object);
    if(!sWFd_flash(pipeout)) {
        return -1;
    }

    if(rcode < 0) {
        return -1;
    }

    return rcode;
}

///////////////////////////////////////////////////
// ジョブの数
///////////////////////////////////////////////////
int saphire_job_num()
{
    return vector_size(gJobs);
}

///////////////////////////////////////////////////
// ジョブのタイトルを返す
///////////////////////////////////////////////////
char* saphire_job_title(int num)
{
    if(num>=0 && num < vector_size(gJobs)) {
        sJob* job = (sJob*)vector_item(gJobs, num);
        return string_c_str(job->mName);
    }
    else {
        return NULL;
    }
}

///////////////////////////////////////////////////
// 全てのジョブをkillする
///////////////////////////////////////////////////
void saphire_kill_all_jobs()
{
    int i;
    for(i=0; i<vector_size(gJobs); i++) {
        sJob* job = (sJob*)vector_item(gJobs, i);

        int j;
        for(j=0; j<vector_size(job->mPIDs); j++) {
            int pid = (int)vector_item(job->mPIDs, j);
            kill(pid, SIGKILL);
        }

        sJob_delete(job);
    }

    vector_clear(gJobs);
}

///////////////////////////////////////////////////
// ジョブを消す
///////////////////////////////////////////////////
void saphire_kill_job(int num)
{
    if(num>=0 && num < vector_size(gJobs)) {
        sJob* job = (sJob*)vector_item(gJobs, num);

        int i;
        for(i=0; i<vector_size(job->mPIDs); i++) {
            int pid = (int)vector_item(job->mPIDs, i);
            kill(pid, SIGKILL);
        }

        sJob_delete(job);
        vector_erase(gJobs, num);
    }
}

///////////////////////////////////////////////////
// ジョブを前面に出す
///////////////////////////////////////////////////
void (*saphire_job_done)(int job_num, char* job_title) = NULL;

BOOL forground_job(int num)
{
    if(num>=0 && num < vector_size(gJobs)) {
        sJob* job = (sJob*)vector_item(gJobs, num);

        if(job->mTty) tcsetattr(STDIN_FILENO, TCSANOW, job->mTty);
        if(tcsetpgrp(0, job->mPGroup) < 0) {
            perror("tcsetpgrp(fg)");
            return FALSE;
        }
        
        if(kill(job->mPGroup, SIGCONT) < 0) {
            perror("kill(fg)");
            return FALSE;
        }

        /// wait ///
        int status = 0;
        pid_t pid = waitpid(job->mPGroup, &status, WUNTRACED);

        if(pid < 0) {
            perror("waitpid(run_wait_cprogs)");
            return FALSE;
        }

        /// 終了した場合 ///
        if(WIFEXITED(status)) {
            if(saphire_job_done) 
                saphire_job_done(num, string_c_str(job->mName));

            sJob_delete(vector_item(gJobs, num));
            vector_erase(gJobs, num);

            /// saphireを前面に出す ///
            //mreset_tty();
            if(tcsetpgrp(0, getpid()) < 0) {
                perror("tcsetpgrp");
                return FALSE;
            }

            int return_code = WEXITSTATUS(status);
            char buf[256];
            snprintf(buf, 256, "%d", return_code);
            (void)saphire_set_global_var("RCODE", buf, NULL);
        }
        /// シグナルの配送により終了した場合 ///
        else if(WIFSIGNALED(status)) {
            if(saphire_job_done) 
                saphire_job_done(num, string_c_str(job->mName));

            sJob_delete(vector_item(gJobs, num));
            vector_erase(gJobs, num);

            /// saphireを前面に出す ///
            if(tcsetpgrp(0, getpid()) < 0) {
                perror("tcsetpgrp");
                return FALSE;
            }

            int return_code = WTERMSIG(status) + 128;
            char buf[256];
            snprintf(buf, 256, "%d", return_code);
            (void)saphire_set_global_var("RCODE", buf, NULL);
        }
        /// シグナルの配送によりフォワグランドジョブが停止した場合 ///
        else if(WIFSTOPPED(status)) {
            if(!job->mTty) {
                job->mTty = MALLOC(sizeof(struct termios));
                tcgetattr(STDIN_FILENO, job->mTty);
            }

            /// saphire を前面に出す ///
            if(tcsetpgrp(0, getpid()) < 0) {
                perror("tcsetpgrp");
                return FALSE;
            }
        }

        return TRUE;
    }

    err_msg("invalid job number", "fg", 1);
    return FALSE;
}

///////////////////////////////////////////////////
// ジョブにSIGCONTする
///////////////////////////////////////////////////
void background_job(int num)
{
    if(num>=0 && num < vector_size(gJobs)) {
        sJob* job = (sJob*)vector_item(gJobs, num);
        
        if(kill(job->mPGroup, SIGCONT) < 0) {
            perror("kill(bg)");
            return;
        }
    }
}

///////////////////////////////////////////////////
// バックグラウンドジョブが終了していれば後処理をする
///////////////////////////////////////////////////
void saphire_wait_background_job()
{
    int status;
    pid_t pid = waitpid(-1, &status, WNOHANG);

    if(pid >= 0) {
        int i;
        for(i=0; i<vector_size(gJobs); i++) {
            sJob* job = (sJob*)vector_item(gJobs, i);

            if(pid == job->mPGroup) {
                if(saphire_job_done) 
                    saphire_job_done(i, string_c_str(job->mName));

                sJob_delete(vector_item(gJobs, i));
                vector_erase(gJobs, i);
                break;
            }
        }
    }
}

///////////////////////////////////////////////////////
// 内部コマンドの登録
///////////////////////////////////////////////////////
void saphire_add_inner_command(char* name, fInnerCommand fun)
{
    hash_put(gInnerCommands, name, sInnerCommand_new(name, fun));
}

// コマンド（文字列）をコンパイルしてstatmentsに格納したい関数
// エラー時はgErrMsgを出力する
BOOL saphire_compile2(char* cmdline, char* fname, sStatments* statments)
{
    int sline = 1;
    if(!parse(cmdline, fname, &sline, statments)) {
        return FALSE;
    }
    else {
        return TRUE;
    }
}

// compile2で格納されたstatmentsを実行したい関数
int saphire_run(sStatments* statments, char* title, sWFd* pipeout, sRFd* pipein, int pipeerr)
{
    sRunInfo runinfo;

    int return_ = FALSE;
    int break_ = FALSE;

    runinfo.return_ = &return_;
    runinfo.break_ = &break_;
    runinfo.envs = VECTOR_NEW(10);
    runinfo.enable_return = TRUE;
    runinfo.enable_break = FALSE;
    runinfo.parent_blocks = NULL;
    runinfo.object = NULL;
    runinfo.running_fun = NULL;
    runinfo.running_class = NULL;

    int ret = run(&runinfo, statments, title, pipeout, pipein, pipeerr, FALSE);

    int i;
    for(i=0; i<vector_size(runinfo.envs); i++) {
        string_delete(vector_item(runinfo.envs, i));
    }
    vector_delete(runinfo.envs);

    return ret;
}

void saphire_sweep()
{
    tmpfile_sweep();
}

/// 一時ファイルを消したい関数
void tmpfile_sweep()
{
    /// プロセス置換で使った一時ファイルを掃除 ///
    int k;
    for(k=0; k<vector_size(gTmpFiles); k++) {
        string_obj* fname = vector_item(gTmpFiles, k);

        (void)unlink(string_c_str(fname));
        string_delete(fname);
    }
    vector_clear(gTmpFiles);
}

/// 一時ファイルを全て消したい関数
void tmpfile_all_sweep()
{
    DIR* dir = opendir(gTmpDir);
    if(dir) {
        struct dirent* entry;
        while(entry = readdir(dir)) {
            if(strstr(entry->d_name, "saphire_") == entry->d_name) {
                char fname[PATH_MAX];
                snprintf(fname, PATH_MAX, "%s/%s", gTmpDir, entry->d_name);

                (void)unlink(fname);
            }
        }
    }
    closedir(dir);
}
