/*************************************************************************
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005-2008 by Kohei Yoshida.
 *    1039 Kingsway Dr., Apex, NC 27502, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "numeric/funcobj.hxx"
#include "numeric/exception.hxx"
#include <vector>
#include <string>
#include <sstream>

using ::std::vector;
using ::std::string;

namespace scsolver { namespace numeric {

const char* VarSizeException::what() const throw()
{
    return "variable size exception";
}

/** 
 * Wrapper class to use a mult-variable function object as a single variable
 * one by locking all variables but one.
 */
class BaseFuncSingleObjImpl : public SingleVarFuncObj
{
public:
    BaseFuncSingleObjImpl(BaseFuncObj& rParent, size_t varIndex) :
        m_rParent(rParent),
        m_varIndex(varIndex)
    {
        if (varIndex >= m_rParent.getVarCount())
            throw VarSizeException();
    }

    virtual ~BaseFuncSingleObjImpl()
    {
    }

    virtual void setVar(double var)
    {
        m_rParent.setVar(m_varIndex, var);
    }

    virtual double getVar() const
    {
        return m_rParent.getVar(m_varIndex);
    }

    virtual double eval() const
    {
        return m_rParent.eval();
    }

    virtual const string getFuncString() const
    {
        return m_rParent.getFuncString();
    }

private:
    BaseFuncObj& m_rParent;
    size_t m_varIndex;
};

//-----------------------------------------------------------------------------

/** 
 * This class wraps a multi-variable function object into a single-variable
 * one by locking the change ratio of the variables.  The variable in the 
 * wrapping class will always be the 1st variable of the original function. 
 * For instance, a 3-variable function can be turned into a single-variable 
 * function by specifying change ratios of (3.0, 2.0, 1.5).  In this case, 
 * these relationships - x2 = 2.0/3.0 * x1 and x3 = 1.5/3.0 * x1 - is 
 * established. 
 */
class BaseFuncRatioObjImpl : public SingleVarFuncObj
{
public:
    BaseFuncRatioObjImpl(BaseFuncObj& rParent, const vector<double>& ratios) :
        m_rParent(rParent),
        m_ratios(ratios),
        m_varIndex(0)
    {
        size_t varCount = m_rParent.getVarCount();
        if (ratios.size() != varCount || varCount == 0)
            throw VarSizeException();

        // By default, the 1st variable is used.  But if the ratio of the first
        // variable is zero, an alternative variable must be used.
        bool varIndexFound = false;
        for (size_t i = 0; i < varCount; ++i)
        {
            double absRatio = m_ratios[i] > 0 ? m_ratios[i] : -m_ratios[i];
            if (absRatio > 3.89e-15)
            {
                m_varIndex = i;
                varIndexFound = true;
                break;
            }
        }

        if (!varIndexFound)
            throw Exception("BaseFuncRatioObjImpl::BaseFuncratioObjImpl: variable index not found");

        m_oldVar = rParent.getVar(m_varIndex);
    }

    virtual ~BaseFuncRatioObjImpl()
    {
    }

    virtual void setVar(double var)
    {
        m_rParent.setVar(m_varIndex, var);
        double delta = var - m_oldVar;

        size_t n = m_ratios.size();
        double baseRatio = m_ratios.at(m_varIndex);
        for (size_t i = 0; i < n; ++i)
        {
            if (i == m_varIndex)
                continue;

            double ratio = m_ratios[i]/baseRatio;
            double origVal = m_rParent.getVar(i) + delta*ratio;
            m_rParent.setVar(i, origVal);
        }

        m_oldVar = var;
    }

    virtual double getVar() const
    {
        return m_rParent.getVar(0);
    }

    virtual double eval() const
    {
        return m_rParent.eval();
    }

    virtual const string getFuncString() const
    {
        string funcStr = m_rParent.getFuncString();
        if (m_ratios.empty())
            return funcStr;

        ::std::ostringstream os;
        os << funcStr << " : locked ratio (";
        size_t n = m_ratios.size();
        for (size_t i = 0; i < n; ++i)
        {
            if (i > 0)
                os << ", ";
            os << m_ratios.at(i);
        }
        os << ")";
        return os.str();
    }

private:
    BaseFuncObj&    m_rParent;
    vector<double>  m_ratios;
    size_t          m_varIndex;
    double          m_oldVar;
};

//-----------------------------------------------------------------------------
// BaseFuncObj

BaseFuncObj::BaseFuncObj() :
    m_pSVFuncObj(NULL)
{
}

BaseFuncObj::BaseFuncObj(const BaseFuncObj& r) :
    m_pSVFuncObj(NULL) // this member does NOT need to be copied since its 
                       // instance is always created on the spot.
{
}

BaseFuncObj::~BaseFuncObj()
{
}

SingleVarFuncObj& BaseFuncObj::getSingleVarFuncObj(size_t varIndex)
{
    m_pSVFuncObj.reset(new BaseFuncSingleObjImpl(*this, varIndex));
    return *m_pSVFuncObj;
}

SingleVarFuncObj& BaseFuncObj::getSingleVarFuncObjByRatio(const vector<double>& ratios)
{
    m_pSVFuncObj.reset(new BaseFuncRatioObjImpl(*this, ratios));
    return *m_pSVFuncObj;
}

double BaseFuncObj::operator ()(const vector<double>& vars) const
{
    setVars(vars);
    return eval();
}

// ----------------------------------------------------------------------------

SimpleFuncObj::SimpleFuncObj() :
    BaseFuncObj()
{
}

SimpleFuncObj::SimpleFuncObj(size_t varCount) :
    BaseFuncObj(),
    m_vars(varCount)
{
}

SimpleFuncObj::SimpleFuncObj(const SimpleFuncObj& r) :
    BaseFuncObj(r)
{
    m_vars.assign(r.m_vars.begin(), r.m_vars.end());
}
    
SimpleFuncObj::~SimpleFuncObj()
{
}

void SimpleFuncObj::getVars(vector<double>& rVars) const
{
    rVars.assign(m_vars.begin(), m_vars.end());
}

double SimpleFuncObj::getVar(size_t index) const
{
    return m_vars.at(index);
}

void SimpleFuncObj::setVar(size_t index, double var) const
{
    if (index >= m_vars.size())
        return;

    m_vars[index] = var;
}

void SimpleFuncObj::setVars(const vector<double> &vars) const
{
    vector<double> tmp(vars);
    m_vars.swap(tmp);
}

size_t SimpleFuncObj::getVarCount() const
{
    return m_vars.size();
}

// ----------------------------------------------------------------------------

SingleVarFuncObj::SingleVarFuncObj()
{
}

SingleVarFuncObj::~SingleVarFuncObj()
{
}

double SingleVarFuncObj::operator()(double var)
{
    setVar(var);
    return eval();
}

}}
