// The original version of this code (revision 2 of the Subversion
// archive) was released subject to the following license:

//   Copyright (c) 2006, Sun Microsystems, Inc.  All rights reserved.
//   Redistribution and use in source and binary forms, with or
//   without modification, are permitted provided that the following
//   conditions are met:

//   * Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above
//     copyright notice, this list of conditions and the following
//     disclaimer in the documentation and/or other materials provided
//     with the distribution.
//   * Neither the name of Sun Microsystems or the names of
//     contributors may be used to endorse or promote products derived
//     from this software without specific prior written permission.
 
//   THIS SOFTWARE IS PROVIDED BY SUN AND ITS LICENSORS ``AS IS'' AND
//   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
//   THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
//   PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SUN OR ITS
//   LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
//   USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
//   AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
//   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
//   ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
//   POSSIBILITY OF SUCH DAMAGE.

// Subsequent additions and modifications are Copyright (c) 2006 David
// Carlton, and may be used subject to the same conditions.

#ifndef UNITTEST_UNITTEST_H
#define UNITTEST_UNITTEST_H

#include <iosfwd>
#include <list>
#include <sstream>

namespace UnitTest {

  class Suite;
  class Test;
  class TestPtr;

  // A test should call these assertion functions (presumably via the
  // macros ASSERT and friends defined below) to check that the object
  // is behaving appropriately.  Call ASSERT if you want to check that
  // a boolean condition holds.  Call ASSERT_EQUALS if you want to
  // check that two objects are equal.  Call ASSERT_EXCEPTION if you
  // expect an expression to throw an exception.  These methods all
  // communicate failures by throwing an AssertionFailure exception.

  void assertTrue(bool val, const char *filename,
		  int line, const char *function);

  // ASSERT_EQUALS assumes that S and T are comparable via != and
  // support output via <<.  If they don't support output, just use
  // ASSERT instead.
  template<typename S, typename T>
  void assertEquals(const S &expected, const T &actual,
		    const char *filename,
		    int line,
		    const char *function);

  // We provide special cases to allow comparisons of std::strings with
  // string literals: otherwise, the compiler complains about assigning
  // to arrays.  (And probably output won't work right, either.)
  void assertEquals(const std::string &expected,
		    const char *actual,
		    const char *filename,
		    int line,
		    const char *function);
  void assertEquals(const char *expected,
		    const std::string &actual,
		    const char *filename,
		    int line,
		    const char *function);

  // A variant of assertTrue where an additional message is printed;
  // used by ASSERT_EXCEPTION.  This always triggers an assertion
  // failure.
  void assertMessage(const std::string &message,
		     const char *filename,
		     int line,
		     const char *function);

  // This is the approved way of signalling a failure during a
  // destructor or during run() in the context of another thread: it
  // marks the current test as having failed, but allows the
  // destructor to complete.
  void assertFailure(const std::string &message,
		     const char *filename,
		     int line,
		     const char *function);

  // Control whether assertion failures involving numbers are printed
  // in decimal (the default) or hex.
  enum Radix {
    DECIMAL,
    HEX
  };

  Radix radix();
  void radix(Radix radix);

  // Turn a subclass of UnitTest::Suite into a UnitTest::Test.
  template<typename TestSuite>
  TestPtr createSuite();

  // Turn a non-subclass of UnitTest::Suite into a UnitTest::Test.
  template<typename T>
  TestPtr createNonSuite();
  template<typename T, typename A>
  TestPtr createNonSuite(A a);
  template<typename T, typename A, typename B>
  TestPtr createNonSuite(A a, B b);
  template<typename T, typename A, typename B, typename C>
  TestPtr createNonSuite(A a, B b, C c);
  template<typename T, typename A, typename B, typename C, typename D>
  TestPtr createNonSuite(A a, B b, C c, D d);

  // See comments above assertTrue and assertEquals.
#define ASSERT(val)						        \
  UnitTest::assertTrue(val, __FILE__, __LINE__, __PRETTY_FUNCTION__)
#define ASSERT_EQUALS(expected, actual)                                 \
  UnitTest::assertEquals(expected, actual, __FILE__, __LINE__,	        \
			 __PRETTY_FUNCTION__)

  // Assert that EXPR throws an exception of type EXCEPTION.  For
  // example,

  // ASSERT_EXCEPTION(functionThatShouldThrowRangeException(),
  //                  std::runtime_error);
#define ASSERT_EXCEPTION(expr, exception)	        \
  do {							\
    try {						\
      expr;						\
      UnitTest::assertMessage			        \
	("Expected " #exception " to be thrown, "	\
	 "but none was.",				\
	 __FILE__, __LINE__, __PRETTY_FUNCTION__);	\
    } catch (const exception &expected) {		\
    }							\
  } while (false)					\
							\
    // Expect an exception with a specific message.
#define ASSERT_EXCEPTION_MESSAGE(expr, exception, message)	        \
  do {									\
    try {								\
      expr;								\
      UnitTest::assertMessage					        \
	("Expected " #exception " to be thrown, "			\
	 "but none was.",						\
	 __FILE__, __LINE__, __PRETTY_FUNCTION__);			\
    } catch (const exception &expected) {				\
      if (std::string(expected.what()) != message) {		        \
	std::ostringstream complaint;					\
	complaint << "Expected " << #exception				\
		  << " to be thrown, regarding " << message		\
		  << "; actual exception was "				\
		  << expected.what();					\
	UnitTest::assertMessage					        \
	  (complaint.str(),						\
	   __FILE__, __LINE__, __PRETTY_FUNCTION__);			\
      }									\
    }									\
  } while (false)


  // Use this to trigger a failure from within a destructor or during
  // run() within another thread.  (It doesn't work within the test's
  // constructor, for better or for worse; throw an exception there.)
#define FAIL(message)						        \
  UnitTest::assertFailure(message,					\
		          __FILE__, __LINE__, __PRETTY_FUNCTION__)	\
									\
									\
  // A class for aggregating a bunch of tests.  Its constructor should
  // add the tests to run by calling add.

  class Suite {
  public:
    Suite();
    ~Suite();

    // Add the test TEST to the suite.  Don't use this one.
    void add(TestPtr test);

    // For this version, T should be an arbitrary class with a public
    // function "void run()" and a constructor.  Its constructor will
    // be called right before running the test; its destructor will be
    // called right after running the test.  If you pass extra args
    // here, its constructor will be called with those args.
    template<typename T>
    void add() {
      add(createNonSuite<T>());
    }
    template<typename T, typename A>
    void add(A a) {
      add(createNonSuite<T, A>(a));
    }
    template<typename T, typename A, typename B>
    void add(A a, B b) {
      add(createNonSuite<T, A, B>(a, b));
    }
    template<typename T, typename A, typename B, typename C>
    void add(A a, B b, C c) {
      add(createNonSuite<T, A, B, C>(a, b, c));
    }
    template<typename T, typename A, typename B, typename C, typename D>
    void add(A a, B b, C c, D d) {
      add(createNonSuite<T, A, B, C, D>(a, b, c, d));
    }

    int passes() const;
    int failures() const;

    // Print status output (dots, assertion failures, etc.) to STREAM
    // instead of std::cout.  This is only here to help write unit
    // tests for UnitTest and friends.
    void stream(std::ostream &stream);

    void run();

  private:
    void invokeOne(TestPtr test);
    bool runTearDownOne(TestPtr test);
    bool setUpOne(TestPtr test);
    bool runOne(TestPtr test);
    bool tearDownOne(TestPtr test);
    bool checkForFailures(TestPtr test);

    bool doHandleExceptions(TestPtr test, void (Test::*op)(),
			    const std::string &description);

    const char *doHandleSignals(TestPtr test, void (Test::*op)());

    std::list<TestPtr> tests_;
    std::ostream *stream_;
    int passes_;
    int failures_;
  };

  // The basic test class.  You rarely want to use this: normally
  // tests are constructed either via Suite::add<> or
  // UnitTest::createSuite<>.
  class Test {
  public:
    // Redefine these to control what happens when operatingMode is
    // called.  The default implementations of setup and tearDown do
    // nothing.
    virtual void setUp();
    virtual void run() = 0;
    virtual void tearDown();

    // A UnitTest object can represent either a single test or a
    // collection of tests.  This means that a test has to report not
    // only its own errors but errors (or passes!) in its descendents.
    // It should use the assertion functions above to report its own
    // errors; it should, however, swallow up any exceptions (whether
    // caused by assertion failures or otherwise) from its children.
    // The methods passes()/failures() should return a count of the
    // total number of descendents who passed (i.e. didn't throw an
    // exception) and failed (i.e. did throw an exception).

    // The default implementations of passes()/failures() always
    // return 0: this is appropriate for single tests, since they
    // don't have any children.
    virtual int passes() const;
    virtual int failures() const;

    // Return the name of the test.  The default implementation
    // returns the name of the test class, using RTTI info.  (This is
    // used when printing assertion failure messages.)
    virtual std::string name() const;

    virtual ~Test() {}
  };

  class TestPtr {
  public:
    explicit TestPtr(Test *test);
    TestPtr(const TestPtr &other);
    TestPtr();

    ~TestPtr();

    TestPtr &operator=(const TestPtr &other);

    Test *operator->() const;
    Test &operator*() const;

  private:
    int *ref_;
    Test *test_;
  };

  // A helper class, used for communicating info via exceptions.
  class AssertionFailure {
  public:
    AssertionFailure();
    virtual ~AssertionFailure();
    virtual void printMe(std::ostream &stream, const std::string &testName)
      const = 0;
    
  private:
    // Use this for setting debug breakpoints: AssertionFailure's
    // constructor will call it.  (There's a bug in GDB preventing you
    // from setting a breakpoint on the constructor directly.)
    void triggered();
  };

}

#include "UnitTest.tpp"

#endif // UNITTEST_UNITTEST_H
