// The original version of this code (revision 2 of the Subversion
// archive) was released subject to the following license:

//   Copyright (c) 2006, Sun Microsystems, Inc.  All rights reserved.
//   Redistribution and use in source and binary forms, with or
//   without modification, are permitted provided that the following
//   conditions are met:

//   * Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above
//     copyright notice, this list of conditions and the following
//     disclaimer in the documentation and/or other materials provided
//     with the distribution.
//   * Neither the name of Sun Microsystems or the names of
//     contributors may be used to endorse or promote products derived
//     from this software without specific prior written permission.
 
//   THIS SOFTWARE IS PROVIDED BY SUN AND ITS LICENSORS ``AS IS'' AND
//   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
//   THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
//   PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SUN OR ITS
//   LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
//   USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
//   AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
//   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
//   ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
//   POSSIBILITY OF SUCH DAMAGE.

// Subsequent additions and modifications are Copyright (c) 2006 David
// Carlton, and may be used subject to the same conditions.

#include <UnitTest.hpp>

#include <cctype>
#include <list>
#include <iostream>

#include "Private.hpp"
#include "SigHandler.hpp"

namespace UnitTest {

  namespace Private {

    FailureList failureRequests_;

    class FailureListGuard {
    public:
      FailureListGuard() {
        failureRequests_.push_front(FailureRequest());
      }

      ~FailureListGuard() {
        failureRequests_.pop_front();
      }
    };

  }

  using namespace Private;

  Suite::Suite()
    : stream_(&std::cout),
      passes_(0), failures_(0) {
  }

  Suite::~Suite() {
  }

  void Suite::add(TestPtr test) {
    tests_.push_back(test);
  }

  int Suite::passes() const {
    return passes_;
  }

  int Suite::failures() const {
    return failures_;
  }

  void Suite::stream(std::ostream &stream) {
    stream_ = &stream;
  }

  // Run all tests.
  void Suite::run() {
    volatile bool signalled = false;
    std::string currentName = demangledName(typeid(this));

    // NOTE (2004-03-26, carlton): It's not clear to me if this
    // SigHandler should be in this function or in invokeOne.  For now,
    // I'm putting it here, on the theory that if we're running into
    // signals that aren't caught by one of the other handlers, then
    // something really bad has happened (probably serious memory
    // corruption), and there's not a lot of point in continuing.
    {
      SigHandler handler(stream_);
      sigjmp_buf jumpBuf;

      if (sigsetjmp(jumpBuf, 1) == 0) {
	handler.setUp(&jumpBuf);

	for (std::list<TestPtr>::const_iterator i =
	       tests_.begin();
	     i != tests_.end(); ++i) {
	  currentName = (*i)->name();
	  invokeOne(*i);
	}
      } else {
	signalled = true;
      }
    }

    if (signalled) {
      *stream_ << "Fatal error when running tests." << std::endl
	       << "Error probably occurred near test: "
	       << currentName << std::endl
	       << "Subsequent tests may not have run, and "
	       << "pass/failure count may be inaccurate." << std::endl;
      failures_ += 1;
    }
  }

  void Suite::invokeOne(TestPtr test) {
    FailureListGuard guard;

    bool passed = false;

    if (setUpOne(test))
      passed = runTearDownOne(test);

    passes_ += test->passes() + passed;
    failures_ += test->failures() + !passed;
  }

  bool Suite::runTearDownOne(TestPtr test) {
    bool runPassed = runOne(test);
    bool tearDownPassed = tearDownOne(test);

    bool passed = runPassed && tearDownPassed;

    if (passed)
      passed = checkForFailures(test);

    if (passed)
	*stream_ << "." << std::flush;

    return passed;
  }

  bool Suite::setUpOne(TestPtr test) {
    return doHandleExceptions(test, &Test::setUp, "setting up");
  }

  bool Suite::runOne(TestPtr test) {
    return doHandleExceptions(test, &Test::run, "running");
  }

  bool Suite::tearDownOne(TestPtr test) {
    return doHandleExceptions(test, &Test::tearDown, "tearing down");
  }

  bool Suite::checkForFailures(TestPtr test) {
    FailureRequest lastFailure = failureRequests_.front();

    if (lastFailure.shouldFail()) {
      printFailure(*stream_,
		   lastFailure.filename(),
		   lastFailure.line(),
		   "Failure",
		   test->name(),
		   lastFailure.function(),
		   ":");
      *stream_ << lastFailure.message() << std::endl;

      return false;
    } else {
      return true;
    }
  }

  bool Suite::doHandleExceptions(TestPtr test, void (Test::*op)(),
				 const std::string &description) {
    if (Debug::debug())
      *stream_ << static_cast<char>(std::toupper(description[0]))
	       << description.substr(1) << " test "
	       << test->name() << '.' << std::endl;

    try {
      const char *signalled = doHandleSignals(test, op);

      if (signalled) {
	*stream_ << std::endl
		 << signalled
		 << " when " << description << " test "
		 << test->name() << "." << std::endl;
      } else {
	return true;
      }
    } catch (const AssertionFailure &assert) {
      assert.printMe(*stream_, test->name());
    } catch (const std::exception &except) {
      *stream_ << std::endl
	       << "Unexpected exception when " << description << " test "
	       << test->name() << ":\n"
	       << except.what() << std::endl;
    } catch (...) {
      *stream_ << std::endl
	       << "Unexpected exception when " << description << " test "
	       << test->name() << "." << std::endl;
    }

    return false;
  }

  const char *Suite::doHandleSignals(TestPtr test, void (Test::*op)()) {
    SigHandler handler(stream_);
    sigjmp_buf jumpBuf;

    if (sigsetjmp(jumpBuf, 1) == 0) {
      handler.setUp(&jumpBuf);
      ((*test).*op)();
      return 0;
    } else {
      return handler.sigName();
    }
  }

  // NOTE (2006-07-06, carlton): We define this here instead of in
  // UnitTest.cpp so that the optimizer can't eliminate the call to
  // triggered().
  void AssertionFailure::triggered() {
    // Set breakpoints here.
  }

}
