// The original version of this code (revision 2 of the Subversion
// archive) was released subject to the following license:

//   Copyright (c) 2006, Sun Microsystems, Inc.  All rights reserved.
//   Redistribution and use in source and binary forms, with or
//   without modification, are permitted provided that the following
//   conditions are met:

//   * Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above
//     copyright notice, this list of conditions and the following
//     disclaimer in the documentation and/or other materials provided
//     with the distribution.
//   * Neither the name of Sun Microsystems or the names of
//     contributors may be used to endorse or promote products derived
//     from this software without specific prior written permission.
 
//   THIS SOFTWARE IS PROVIDED BY SUN AND ITS LICENSORS ``AS IS'' AND
//   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
//   THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
//   PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SUN OR ITS
//   LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
//   USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
//   AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
//   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
//   ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
//   POSSIBILITY OF SUCH DAMAGE.

// Subsequent additions and modifications are Copyright (c) 2006 David
// Carlton, and may be used subject to the same conditions.

#include <Registry.hpp>

#include <iostream>
#include <vector>

#include "unittesttest.hpp"
#include "Helper.hpp"

namespace RegistryTest {
  using namespace UnitTest;

  class Base : public HelperBase {
  protected:
    Base() {
      addArg("programName");
      registry_.stream(stream_);
    }

    Registry &registry() {
      return registry_;
    }

    void addTest(const char *name, bool pass = true) {
      Helper *test = new Helper();
      test->shouldFail(!pass);
      test->statusString(&statusString_, name);
      registry_.add(TestPtr(test), name);
    }

    void addArg(const char *arg) {
      args_.push_back(const_cast<char *>(arg));
    }

    std::string statusString() const {
      return statusString_;
    }

    int runTests() {
      return registry_.run(args_.size(), &args_.at(0));
    }

    void checkOutput(const std::string &expected) {
      bool success = (stream_.str().find(expected) !=
		      std::string::npos);

      if (!success)
	std::cout << "Actual output: "
		  << stream_.str() << std::endl;

      ASSERT(success);
    }

  private:
    Registry registry_;
    std::string statusString_;
    std::vector<char *> args_;
    std::ostringstream stream_;
  };

  class Zero : public Base {
  public:
    void run() {
      addTest("1");
      addTest("2");

      runTests();

      ASSERT_EQUALS("1:setup;1:run;1:teardown;2:setup;2:run;2:teardown;",
		    statusString());
    }
  };

  class One : public Base {
  public:
    void run() {
      addTest("1");
      addTest("2");

      addArg("1");
      runTests();

      ASSERT_EQUALS("1:setup;1:run;1:teardown;", statusString());
    }
  };

  class Two : public Base {
  public:
    void run() {
      addTest("1");
      addTest("2");
      addTest("3");

      addArg("3");
      addArg("2");
      runTests();

      ASSERT_EQUALS("3:setup;3:run;3:teardown;2:setup;2:run;2:teardown;",
		    statusString());
    }
  };

  class Missing : public Base {
  public:
    void run() {
      addTest("1");

      addArg("oops");

      ASSERT_EQUALS(1, runTests());
      checkOutput("Bad test name: oops.  Use -list for valid names.\n");
    }
  };

  class ArgsFirst : public Base {
  public:
    void run() {
      ASSERT_EQUALS(1, runTests());
      checkOutput("No tests available.\n");
    }
  };

  class Stream : public Base {
  public:
    void run() {
      std::ostringstream stream;

      addTest("1", false);

      registry().stream(stream);

      runTests();

      ASSERT(stream.str().find("Helper\n") != std::string::npos);
    }
  };

  class PassFail : public Base {
  public:
    void run() {
      addTest("1", false);
      addTest("2", true);
      addTest("3", true);

      runTests();

      checkOutput("\nTests finished with 2 passes and 1 failures.\n");
    }
  };

  class StatusBad : public Base {
  public:
    void run() {
      addTest("1", false);
      addTest("2", true);

      ASSERT_EQUALS(1, runTests());
    }
  };

  class StatusGood : public Base {
  public:
    void run() {
      addTest("1", true);
      addTest("2", true);

      ASSERT_EQUALS(0, runTests());
    }
  };

  class Debug : public Base {
  public:
    void run() {
      addTest("1", true);
      addTest("2", true);

      addArg("-debug");

      runTests();

      checkOutput("Setting up test Helper.\n"
		  "Running test Helper.\n"
		  "Tearing down test Helper.\n"
		  "."
		  "Setting up test Helper.\n"
		  "Running test Helper.\n"
		  "Tearing down test Helper.\n"
		  "."
		  );
    }
  };

  class DebugArg : public Base {
  public:
    void run() {
      addTest("1", true);
      addTest("2", true);
      addTest("3", true);

      addArg("-debug");
      addArg("2");

      runTests();

      checkOutput("Setting up test Helper.\n"
		  "Running test Helper.\n"
		  "Tearing down test Helper.\n"
		  "."
		  );

      ASSERT_EQUALS("2:setup;2:run;2:teardown;", statusString());
    }
  };

  class List : public Base {
  public:
    void run() {
      addTest("1", true);
      addTest("2", true);

      addArg("-list");

      ASSERT_EQUALS(0, runTests());

      checkOutput("1\n2\n");
    }
  };

  class All : public Suite {
  public:
    All() {
      add<Zero>();
      add<One>();
      add<Two>();
      add<Missing>();
      add<ArgsFirst>();
      add<PassFail>();
      add<Stream>();
      add<StatusBad>();
      add<StatusGood>();
      add<Debug>();
      add<DebugArg>();
      add<List>();
    }
  };

}

UnitTest::TestPtr registryTests() {
  return UnitTest::createSuite<RegistryTest::All>();
}
