/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: TableManager.hxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: obo $ $Date: 2008/01/10 11:33:24 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef INCLUDED_TABLE_MANAGER_HXX
#define INCLUDED_TABLE_MANAGER_HXX

#ifndef INCLUDED_TABLE_DATA_HXX
#include <resourcemodel/TableData.hxx>
#endif

#ifndef INCLUDED_WW8_RESOURCE_MODEL_HXX
#include <resourcemodel/WW8ResourceModel.hxx>
#endif

#ifndef INCLUDED_SPRMIDS_HXX
#include <doctok/sprmids.hxx>
#endif

#include <boost/shared_ptr.hpp>
#include <stack>

namespace writerfilter
{

using namespace ::std;

template <typename T, typename PropertiesPointer>
/**
   Class to handle events generated by TableManager::resolveCurrentTable
 */
class WRITERFILTER_DLLPUBLIC TableDataHandler
{
public:
    typedef boost::shared_ptr<TableDataHandler> Pointer_t;

    /**
       Handle start of table.

       @param nRows   number of rows in the table
       @param nDepth  depth of the table in surrounding table hierarchy
       @param pProps  properties of the table
     */
    virtual void startTable(unsigned int nRows, unsigned int nDepth,
                            PropertiesPointer pProps) = 0;

    /**
       Handle end of table.
     */
    virtual void endTable() = 0;

    /**
       Handle start of row.

       @param nCols    number of columns in the table
       @param pProps   properties of the row
     */
    virtual void startRow(unsigned int nCols,
                          PropertiesPointer pProps) = 0;

    /**
       Handle end of row.
    */
    virtual void endRow() = 0;

    /**
       Handle start of cell.

       @param rT     start handle of the cell
       @param pProps properties of the cell
    */
    virtual void startCell(const T & rT, PropertiesPointer pProps) = 0;

    /**
        Handle end of cell.

        @param rT    end handle of cell
    */
    virtual void endCell(const T & rT) = 0;
};

template <typename T, typename PropertiesPointer>
/**
   The table manager.

   This class gets forwarded events from the tokenizer. It gathers the
   table data and after ending the table generates events for the
   table structure. The events have to be handles by a TableDataHandler.

 */
class TableManager
{
    typedef boost::shared_ptr<T> T_p;

    /**
       true if at the end of a row
     */
    bool mbRowEnd;

    /**
       true when in a cell
    */
    bool mbInCell;

    /**
       true when at the end of a cell
    */
    bool mbCellEnd;

    /**
       depth of the current cell
    */
    sal_uInt32 mnTableDepthNew;

    /**
        depth of the previous cell
    */
    sal_uInt32 mnTableDepth;

    /**
       properties at the current point in document
    */
    PropertiesPointer mpProps;

    /**
       properties of the current cell
    */
    PropertiesPointer mpCellProps;

    /**
        properties of the current row
    */
    PropertiesPointer mpRowProps;

    /**
       properties of the current table
    */
    PropertiesPointer mpTableProps;

    /**
       handle for the current position in document
    */
    T mCurHandle;

    /**
       stack of table data

       for each level of nested tables there is one frame in the stack
     */
    stack<typename TableData<T, PropertiesPointer>::Pointer_t > mTableDataStack;

    typedef typename TableDataHandler<T, PropertiesPointer>::Pointer_t TableDataHandlerPointer_t;

    /**
       handler for resolveCurrentTable
     */
    TableDataHandlerPointer_t mpTableDataHandler;

    /**
       Set flag which indicates the current handle is in a cell.
     */
    void inCell();

    /**
       Set flag which indicate the current handle is at the end of a cell.
    */
    void endCell();

    /**
       Set the table depth of the current cell.

       @param nDepth     the cell depth
     */
    void cellDepth(sal_uInt32 nDepth);

    /**
       Set flag indication the current handle is at the end of a row.
    */
    void endRow();

    /**
       Resolve the current table to the TableDataHandler.
     */
    void resolveCurrentTable();

protected:

    /**
       Return current table depth.
     */
    sal_uInt32 getTableDepthNew() { return mnTableDepthNew; }

    /**
       Action to be carried out at the end of the last paragraph of a
       cell.
     */
    virtual void endOfCellAction();

    /**
       Action to be carried out at the end of the "table row"
       paragraph.
     */
    virtual void endOfRowAction();
    /** let the derived class clear their table related data
     */
    virtual void clearData();


public:
    TableManager();
    virtual ~TableManager(){}

    /**
       Set handler for resolveCurrentTable.

       @param pTableDataHandler     the handler
     */
    void setHandler(TableDataHandlerPointer_t pTableDataHandler);

    /**
       Set the current handle.

       @param rHandle     the handle
     */
    virtual void handle(const T & rHandle);

    /**
       Start a new table level.

       A new context is pushed onto the table data stack,
     */
    virtual void startLevel();

    /**
       End a table level.

       The current table is resolved and the context is popped from
       the stack.
     */
    virtual void endLevel();

    /**
       Handle the start of a paragraph group.
     */
    virtual void startParagraphGroup();

    /**
       Handle the end of a paragraph group.
    */
    virtual void endParagraphGroup();

    /**
       Handle an SPRM at curent handle.

       @param rSprm   the SPRM
     */
    virtual bool sprm(Sprm & rSprm);

    /**
       Handle properties at current handle.

       @param pProps   the properites
     */
    virtual void props(PropertiesPointer pProps);

    /**
       Handle occurance of character 0x7.
     */
    virtual void handle0x7();

    /**
       Handle 8 bit text at current handle.

       @param data    array of characters
       @param len     number of characters to handle
     */
    virtual void text(const sal_uInt8 * data, size_t len);

    /**
       Handle 16 bit text at current handle.

       @param data    array of characters
       @param len     number of characters to handle
     */
    virtual void utext(const sal_uInt8 * data, size_t len);

    /**
       Handle properties of the current cell.

       @param pProps   the properties
     */
    virtual void cellProps(PropertiesPointer pProps);

    /**
       Handle properties of a certain cell in the current row.

       @paran i        index of the cell in the current row
       @param pProps   the properties
     */
    virtual void cellPropsByCell(unsigned int i, PropertiesPointer pProps);

    /**
       Handle properties of the current row.

       @param pProps   the properties
     */
    virtual void insertRowProps(PropertiesPointer pProps);

    /**
       Handle properties of the current table.

       @param pProps   the properties
     */
    virtual void insertTableProps(PropertiesPointer pProps);

    /**
       Return if table manager has detected paragraph to ignore.

       If this function returns true the current paragraph contains
       only control information, e.g. end of row.
     */
    virtual bool isIgnore() const;
};

template <typename T, typename PropertiesPointer>
TableManager<T, PropertiesPointer>::TableManager()
: mbRowEnd(false), mbInCell(false), mbCellEnd(false), mnTableDepthNew(0),
  mnTableDepth(0)
{
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::cellDepth(sal_uInt32 nDepth)
{
    mnTableDepthNew = nDepth;
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::inCell()
{
    mbInCell = true;

    if (mnTableDepthNew < 1)
        mnTableDepthNew = 1;
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::endCell()
{
    mbCellEnd = true;
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::endRow()
{
    mbRowEnd = true;
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::setHandler
(typename TableDataHandler<T, PropertiesPointer>::Pointer_t pTableDataHandler)
{
    mpTableDataHandler = pTableDataHandler;
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::handle(const T & rHandle)
{
    mCurHandle = rHandle;
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::startLevel()
{
    typename TableData<T, PropertiesPointer>::Pointer_t pTableData
        (new TableData<T, PropertiesPointer>(mTableDataStack.size()));

    mTableDataStack.push(pTableData);
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::endLevel()
{
    if (mpTableDataHandler.get() != NULL)
        resolveCurrentTable();

    mTableDataStack.pop();
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::startParagraphGroup()
{
    mbRowEnd = false;
    mbInCell = false;
    mbCellEnd = false;
    mnTableDepthNew = 0;
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::endParagraphGroup()
{
    sal_Int32 nTableDepthDifference = mnTableDepthNew - mnTableDepth;
    while (nTableDepthDifference > 0)
    {
        startLevel();

        --nTableDepthDifference;
    }
    while (nTableDepthDifference < 0)
    {
        endLevel();

        ++nTableDepthDifference;
    }

    mnTableDepth = mnTableDepthNew;

    typename TableData<T, PropertiesPointer>::Pointer_t pTableData =
        mTableDataStack.top();

    if (mbRowEnd)
    {
        endOfRowAction();
        pTableData->endRow(mpRowProps);
        mpRowProps.reset();
    }

    else if (mbInCell)
    {
        if (! pTableData->isCellOpen())
            pTableData->addCell(mCurHandle, mpCellProps);

        if (mbCellEnd)
        {
            endOfCellAction();
            pTableData->endCell(mCurHandle);
        }
    }
    mpCellProps.reset();
}

template <typename T, typename PropertiesPointer>
bool TableManager<T, PropertiesPointer>::sprm(Sprm & rSprm)
{
    bool bRet = true;
    switch (rSprm.getId())
    {
    case NS_sprm::LN_PTableDepth:
        {
            Value::Pointer_t pValue = rSprm.getValue();

            cellDepth(pValue->getInt());
        }
        break;
    case NS_sprm::LN_PFInTable:
        inCell();
        break;
    case NS_sprm::LN_PCell:
        endCell();
        break;
    case NS_sprm::LN_PFTtp:
    case NS_sprm::LN_PRow:
        endRow();
        break;
    default:
        bRet = false;
    }
    return bRet;
}
template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::props(PropertiesPointer pProps)
{
    mpProps = pProps;
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::handle0x7()
{
    if (mnTableDepthNew < 1)
        mnTableDepthNew = 1;

    if (mbInCell)
        endCell();
    else
        endRow();
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::text(const sal_uInt8 * data, size_t len)
{
    // optimization: cell/row end characters are the last characters in a run
    if (len > 0)
    {
        if (data[len - 1] == 0x7)
            handle0x7();
    }
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::utext(const sal_uInt8 * data, size_t len)
{
    // optimization: cell/row end characters are the last characters in a run

    if (len > 0)
    {
        sal_Unicode nChar = data[(len - 1) * 2] + (data[(len - 1) * 2 + 1] << 8);
        if (nChar == 0x7)
            handle0x7();
    }
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::cellProps(PropertiesPointer pProps)
{
    if(mpCellProps.get())
        mpCellProps->insert( pProps );
    else
        mpCellProps = pProps;
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::cellPropsByCell
(unsigned int i, PropertiesPointer pProps)
{
    mTableDataStack.top()->insertCellProperties(i, pProps);
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::insertRowProps(PropertiesPointer pProps)
{
    if( mpRowProps.get() )
        mpRowProps->insert( pProps );
    else
        mpRowProps = pProps;
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::insertTableProps(PropertiesPointer pProps)
{
    if( mpTableProps.get() )
        mpTableProps->insert( pProps );
    else
        mpTableProps = pProps;
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::resolveCurrentTable()
{
    if (mpTableDataHandler.get() != NULL)
    {
        typename TableData<T, PropertiesPointer>::Pointer_t
            pTableData = mTableDataStack.top();

        unsigned int nRows = pTableData->getRowCount();

        mpTableDataHandler->startTable(nRows, pTableData->getDepth(), mpTableProps);

        for (unsigned int nRow = 0; nRow < nRows; ++nRow)
        {
            typename RowData<T, PropertiesPointer>::Pointer_t pRowData = pTableData->getRow(nRow);

            unsigned int nCells = pRowData->getCellCount();

            mpTableDataHandler->startRow(nCells, pRowData->getProperties());

            for (unsigned int nCell = 0; nCell < nCells; ++nCell)
            {
                mpTableDataHandler->startCell
                    (pRowData->getCellStart(nCell),
                     pRowData->getCellProperties(nCell));

                mpTableDataHandler->endCell(pRowData->getCellEnd(nCell));
            }

            mpTableDataHandler->endRow();
        }

        mpTableDataHandler->endTable();
    }
    mpTableProps.reset();
    clearData();
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::endOfCellAction()
{
}

template <typename T, typename PropertiesPointer>
void TableManager<T, PropertiesPointer>::endOfRowAction()
{
}

template <typename T, typename PropertiesPointer>
bool TableManager<T, PropertiesPointer>::isIgnore() const
{
    return mbRowEnd;
}
template <typename T, typename PropertiesPointer>
void  TableManager<T, PropertiesPointer>::clearData() 
{
}

}

#endif // INCLUDED_TABLE_MANAGER_HXX
