/*
 *  Copyright (C) 2005 UCHINO Satoshi.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include "windowmonitor.h"
#include "serverdata.h"
#include <iostream>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>

ServerData theServerData;

/* C interface */
void launch(unsigned long id)
{
  theServerData.launch(id);
}

/* C interface */
void sendWindowIcon(WnckWindow *window)
{
  theServerData.sendWindowIcon(window);
}

/* C interface */
void deleteWindowIcon(WnckWindow *window)
{
  theServerData.deleteWindowIcon(window);
}

/* C interface */
void processServerDataReqMsg(rfbServerDataReqMsg *msg)
{
  char *data = NULL;
  unsigned int  dataType   = Swap16IfLE(msg->dataType);
  unsigned long id         = Swap32IfLE(msg->id);
  unsigned long dataLength = Swap32IfLE(msg->dataLength);
  unsigned long cl         = msg->clientPtr;

  if (dataLength > 0) {
    data = (char *)malloc(dataLength+1);
    if (!data || ReadFromRFBServer(data, dataLength) < 0) {
      fprintf(stderr, "processServerDataReqMsg: read");
      free(data);
      return;
    }
  }
  switch (dataType) {
  case rfbServerDataMenuPref:
    if (dataLength == 0) {
      fprintf(stderr,"processServerDataReqMsg: data too short for rfbServerDataMenuPref\n");
      break;
    }
    data[dataLength] = '\0';
    theServerData.processServerDataMenuPref(cl, (rfbServerDataMenuPrefReq *)data);
    break;
  case rfbServerDataIconPref:
    if (dataLength < sizeof(rfbServerDataIconPrefReq)) {
      fprintf(stderr,"processServerDataReqMsg: data too short for rfbServerDataIconPref\n");
      break;
    }
    theServerData.processServerDataIconPref(cl, (rfbServerDataIconPrefReq *)data);
    break;
  case rfbServerDataClientGone:
    theServerData.clientGone(cl);
    break;

  default:
    fprintf(stderr,"processServerDataReqMsg: unknown data type: %d\n", dataType);
    break;
  }
  if (data)
    free(data);
}

ServerData::ServerData()
  : clientMgr(NULL), iconMgr(NULL), launcher(NULL), tmpFileMgr(NULL)
{
DEBUG_PRINTF(("%s\n", __FUNCTION__));
}

ServerData::~ServerData()
{
DEBUG_PRINTF(("%s\n", __FUNCTION__));
  if (clientMgr)
    delete clientMgr;
  if (iconMgr)
    delete iconMgr;
  if (launcher)
    delete launcher;
  if (tmpFileMgr)
    delete tmpFileMgr;
}

void ServerData::processServerDataMenuPref(unsigned long cl, rfbServerDataMenuPrefReq *menuPrefReq)
{
  if (!clientMgr)
    clientMgr = new ClientMgr;
  if (!launcher) {
    if (!iconMgr)
      iconMgr = new IconMgr(*clientMgr);
    if (!tmpFileMgr)
      tmpFileMgr = new TmpFileMgr;
    launcher = new Launcher(*clientMgr, *iconMgr, *tmpFileMgr);
  }
  clientMgr->setMenuPref(cl, menuPrefReq);
  launcher->sendAllMenuItems(cl);
}

void ServerData::processServerDataIconPref(unsigned long cl, rfbServerDataIconPrefReq *iconpref)
{
  if (!clientMgr)
    clientMgr = new ClientMgr;
  if (!launcher) {
    if (!iconMgr)
      iconMgr = new IconMgr(*clientMgr);
    if (!tmpFileMgr)
      tmpFileMgr = new TmpFileMgr;
    launcher = new Launcher(*clientMgr, *iconMgr, *tmpFileMgr);
  }
  clientMgr->setIconPref(cl, iconpref);
  launcher->sendMenuIcons(cl);
  sendAllWindowIcons(cl);
}

void ServerData::clientGone(unsigned long cl)
{
  if (clientMgr) {
    clientMgr->clientGone(cl);
    if (clientMgr->allGone()) {
      if (clientMgr)
	delete clientMgr;
      clientMgr = NULL;
      if (iconMgr)
	delete iconMgr;
      iconMgr = NULL;
      if (launcher)
	delete launcher;
      launcher = NULL;
      if (tmpFileMgr)
	delete tmpFileMgr;
      tmpFileMgr = NULL;
    }
  }
}

bool ServerData::launch(unsigned long id)
{
  return launcher && launcher->launch(id);
}

bool ServerData::createTmpIconFileKey(string &key,
				      WnckWindow *window, int iconSize)
{
  char tmp[256];
  if (snprintf(tmp, 256, "%08X-%d", wnck_window_get_xid(window), iconSize) <= 0)
    return false;

  key = tmp;
  return true;
}

bool ServerData::sendWindowIcon(unsigned long clientPtr, const ClientMgr::IconPref& iconPref, WnckWindow *window)
{
  string tmpfilekey;
  string realpath;
  int tmpfile_fd;	// fd of tmpfile
  bool result;
  int iconSize;

  if (iconPref.size < 24)
    iconSize = 16;
  else
    iconSize = 32;

  if (!createTmpIconFileKey(tmpfilekey, window, iconSize)) {
    fprintf(stderr, "%s: failed to creat tmpfilekey\n", __FUNCTION__);
    return false;
  }

  // first try to create tmpfile
  tmpfile_fd = tmpFileMgr->createTmpFile(tmpfilekey, realpath);
  if (tmpfile_fd == -1) {
    // check if it already exists.
    tmpfile_fd = tmpFileMgr->lookupTmpFile(tmpfilekey, realpath);
    if (tmpfile_fd != -1) { // found
      /* send item */
      result = SendServerIconPath(wnck_window_get_xid (window),
				  rfbServerDataWindowIconPath, clientPtr,
				  0, iconSize, rfbIconFormatTypePng,
				  realpath.length(), realpath.c_str());
      close(tmpfile_fd);
      return result;
    }
    fprintf(stderr, "%s: failed to lookup %s\n", __FUNCTION__, tmpfilekey);
    // error
    return false;
  }

  GdkPixbuf *pixbuf;
  GError *error = NULL;
  if (iconSize == 16) {
    pixbuf = wnck_window_get_mini_icon(window);
  } else {
    pixbuf = wnck_window_get_icon(window);
  }
  if (!gdk_pixbuf_save(pixbuf, realpath.c_str(), "png", &error, NULL)) {
    fprintf(stderr, "%s: gdk_pixbuf_save() failed\n", __FUNCTION__);
    return false;
  }

  /* send item */
  result =SendServerIconPath(wnck_window_get_xid (window),
			     rfbServerDataWindowIconPath, clientPtr,
			     0, iconSize, rfbIconFormatTypePng,
			     realpath.length(), realpath.c_str());
  close(tmpfile_fd);
  return result;
}

bool ServerData::sendWindowIconCallback(unsigned long clientPtr, const ClientMgr::IconPref& iconPref, void *_this, void *param)
{
  ServerData *serverData = (ServerData *)_this;
  return serverData->sendWindowIcon(clientPtr, iconPref, (WnckWindow *)param);
}

bool ServerData::sendWindowIcon(WnckWindow *window)
{
  return clientMgr &&
    clientMgr->foreachIconPref(sendWindowIconCallback, (void *)this, (void *)window);
}

/* WindowMonitor class may be needed */
bool ServerData::sendAllWindowIcons(unsigned long clientPtr)
{
  if (!clientMgr)
    return false;

  const ClientMgr::IconPref *iconPref = clientMgr->getIconPref(clientPtr);
  if (iconPref == NULL)
    return false;

  bool result = true;
  for (GList *glist = wnck_screen_get_windows(thisScreen); glist; glist = glist->next) {
    WnckWindow *window = (WnckWindow *)glist->data;
    DEBUG_PRINTF(("window '%s'(xid=0x%X)\n",
             wnck_window_get_name(window),
             wnck_window_get_xid(window)));
    if (!wnck_window_is_skip_tasklist(window))   /* ignore if skip_tasklist */
      if (!sendWindowIcon(clientPtr, *iconPref, window))
	result = false;
  }
  return result;
}

bool ServerData::deleteWindowIcon(WnckWindow *window)
{
  if (!tmpFileMgr)
    return false;

  string tmpfilekey;
  string realpath;
  int tmpfile_fd;	// fd of tmpfile
  bool result = false;
  if (createTmpIconFileKey(tmpfilekey, window, 16) &&
      (tmpfile_fd = tmpFileMgr->lookupTmpFile(tmpfilekey, realpath)) != -1) {
    if (unlink(realpath.c_str()) == 0)
      result = true;
    close(tmpfile_fd);
  }
  if (createTmpIconFileKey(tmpfilekey, window, 32) &&
      (tmpfile_fd = tmpFileMgr->lookupTmpFile(tmpfilekey, realpath)) != -1) {
    if (unlink(realpath.c_str()) == 0)
      result = true;
    close(tmpfile_fd);
  }
  return result;	// true if at least one file has been deleted
}
