/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#ifndef KITAFILELOADER_H
#define KITAFILELOADER_H

#include "event.h"

#include <kio/job.h>
#include <qmutex.h>

class QFile;

namespace Kita
{
    class FileLoader;


    /* user option */
    struct LoaderOption
    {
        KURL usrurl;
        QString usrstr;
        QStringList userstrlist;
        int usrnum;
    };


    /* data */
    struct LoaderData
    {
        /* URL and path of target */
        KURL url;
        QString path;

        /*  path of tmpfile */
        QString tmppath;

        /* redirection */
        bool redirection;
        KURL redirectURL; /* = new URL */

        /* file info */
        int code;
        QString header;
        KIO::filesize_t size;
        KIO::filesize_t totalsize;

        /* user option */
        LoaderOption option;
    };


    /*-----------------------------------------*/


    class DownloadManager : public QObject
    {
        Q_OBJECT

        static DownloadManager* instance;
        static QMutex m_mutex;
        QPtrList< FileLoader > m_loaderList;

    public:
        DownloadManager();
        ~DownloadManager();

        static FileLoader* download( const KURL& url, const QString& path,
                                     const LoaderOption& option = LoaderOption() );
        static void stopLoading( const KURL& url );
        static bool isLoadingNow( const KURL& url );

    private:
        FileLoader* downloadPrivate( const KURL& url, const QString& path, const LoaderOption& option );
        void stopLoadingPrivate( const KURL& url );
        bool isLoadingNowPrivate( const KURL& url );

        FileLoader* getLoader( const KURL& url );
        void deleteLoader( Kita::FileLoader* );

    private slots:
        void slotResult( const Kita::LoaderData& );

    protected:
        virtual void customEvent( QCustomEvent * e );
    };


    /*-----------------------------------------*/

    /* loader */
    class FileLoader : public QObject
    {
        Q_OBJECT

        LoaderData m_data;

        KIO::Job* m_currentJob;
        QFile *m_file;
        QDataStream *m_ds;

    public:
        FileLoader( const KURL& url, const QString& path, const LoaderOption& option );
        ~FileLoader();

        void killJob();
        void stopJob();
        bool get();

        const bool isRunning() const;
        const LoaderData& getData() const;

    private:
        void closeFile();
        int responseCode();

    private slots:
        void slotData( KIO::Job* , const QByteArray& );
        void slotTotalSize( KIO::Job * , KIO::filesize_t );
        void slotRedirection( KIO::Job* , const KURL& );
        void slotResult( KIO::Job* );

    signals:
        void data( const Kita::LoaderData&, const QByteArray& );
        void result( const Kita::LoaderData& );
    };


    /*-----------------------------------------*/

    /* delete loader event */
    class DeleteLoaderEvent : public QCustomEvent
    {
        Kita::FileLoader* m_loader;

    public:
        DeleteLoaderEvent( Kita::FileLoader* loader ) : QCustomEvent( EVENT_DeleteLoader )
        , m_loader( loader ) {}

        Kita::FileLoader* getLoader() const { return m_loader; }
    };
}


#endif
