# Copyright (c) 2008 Hideki Ikemoto
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

class BoardListView < KDE::ListView
  slots 'load_board_list()'
  slots 'show_board(QListViewItem*)'
  slots 'context_menu(QListViewItem*,const QPoint&,int)'

  POPUP_OPEN_WITH_WEB_BROWSER = 0
  POPUP_COPY_URL = 1
  POPUP_COPY_TITLE_AND_URL = 2
  POPUP_ADD_TO_FAVORITE = 3
  POPUP_DELETE_FROM_FAVORITE = 4

  COLUMN_TITLE = 0
  COLUMN_URL = 1

  def initialize(parent)
    super(parent)

    setRootIsDecorated(true)
    setSorting(-1)
    addColumn("board name")

    connect(self, SIGNAL("clicked(QListViewItem*)"),
            self, SLOT("show_board(QListViewItem*)"))
    connect(self, SIGNAL("contextMenuRequested(QListViewItem*,const QPoint&,int)"),
            self, SLOT("context_menu(QListViewItem*,const QPoint&,int)"))

    str = nil
    begin
      str = Downloader.load_menu_list
    rescue
      str = ""
    end
    parse_board_list(str)
  end

  BOARD_REGEXP = /<A HREF=([^ ]*).*>(.*)<\/A>/
  GROUP_REGEXP = /<BR><BR><B>(.*)<\/B><BR>/

  def load_board_list
    str = Downloader.download_menu_list
    parse_board_list(str)
  end

  def parse_board_list(str)
    group_list = []
    current_group = nil
    str.each {|line|
      if GROUP_REGEXP.match(line)
        group_name = Regexp.last_match[1]

        current_group = BoardGroup.new(group_name)
        group_list.push(current_group)
      elsif BOARD_REGEXP.match(line)
        board_url = Regexp.last_match[1]
        board_name = Regexp.last_match[2]

        next unless Util.board_url?(board_url)
        board_id = Util.board_url_to_board_id(board_url)
        board_type = Util.board_url_to_board_type_name(board_url)

        board = Board.new(board_url, board_name)
        BoardManager.instance.register_name(board_url, board_name)
        BoardManager.instance.register_url(board_type, board_id, board_url)
        current_group.push(board) if current_group
      end
    }
    group_list.delete_if {|group|
      group.list.empty?
    }

    clear

    group_list.reverse.each {|group|
      item = KDE::ListViewItem.new(self, group.name)
      group.list.reverse.each {|board|
        KDE::ListViewItem.new(item, board.title, board.url)
      }
    }

    kitabbs_url = "http://jbbs.livedoor.jp/computer/18420/"
    BoardManager.instance.register_name(kitabbs_url, "Kita板")
    BoardManager.instance.register_url("jbbs.livedoor.jp", "computer_18420", kitabbs_url)
    KDE::ListViewItem.new(self, "Kita板", "http://jbbs.livedoor.jp/computer/18420/")

    @favorite_item = KDE::ListViewItem.new(self, "お気に入り")
    load_favorites
  end

  def show_board(item)
    return if item == nil

    board_url = item.text(COLUMN_URL)
    return if board_url == nil

    ViewMediator.show_board(board_url)
  end

  def context_menu(item, pos, col)
    return unless item

    board_url = item.text(COLUMN_URL)
    board_title = item.text(COLUMN_TITLE)

    popup = KDE::PopupMenu.new
    popup.insertItem("ウェブブラウザで開く", POPUP_OPEN_WITH_WEB_BROWSER)
    popup.insertItem("Copy URL", POPUP_COPY_URL)
    popup.insertItem("Copy title and URL", POPUP_COPY_TITLE_AND_URL)
    if item.parent == @favorite_item then
      popup.insertItem("お気に入りから削除", POPUP_DELETE_FROM_FAVORITE)
    else
      popup.insertItem("お気に入りに追加", POPUP_ADD_TO_FAVORITE)
    end

    clipboard = Qt::Application.clipboard
    case popup.exec(pos)
    when POPUP_OPEN_WITH_WEB_BROWSER
      $app.invokeBrowser(board_url)
    when POPUP_COPY_URL
      clipboard.setText(board_url, Qt::Clipboard::Clipboard)
      clipboard.setText(board_url, Qt::Clipboard::Selection)
    when POPUP_COPY_TITLE_AND_URL
      cliptext = board_title + "\n" + board_url
      clipboard.setText(cliptext, Qt::Clipboard::Clipboard)
      clipboard.setText(cliptext, Qt::Clipboard::Selection)
    when POPUP_ADD_TO_FAVORITE
      add_to_favorite(board_url)
    when POPUP_DELETE_FROM_FAVORITE
      delete_from_favorite(item, board_url)
    end
  end

  def add_to_favorite(board_url)
    board_type = Util.board_url_to_board_type_name(board_url)
    board_id = Util.board_url_to_board_id(board_url)
    registered = FavoriteBoards.instance.register(board_type, board_id)

    if registered then
      board_name = BoardManager.instance.get_name(board_url)
      KDE::ListViewItem.new(@favorite_item, board_name, board_url)
    end
  end

  def delete_from_favorite(item, board_url)
    board_type = Util.board_url_to_board_type_name(board_url)
    board_id = Util.board_url_to_board_id(board_url)
    FavoriteBoards.instance.delete(board_type, board_id)

    @favorite_item.takeItem(item)
  end

  def load_favorites
    FavoriteBoards.instance.each {|type, id|
      board_url = BoardManager.instance.get_url(type, id)
      board_name = BoardManager.instance.get_name(board_url)
      KDE::ListViewItem.new(@favorite_item, board_name, board_url)
    }
  end
end
