/**
 * ===========================================
 * LibBase : a free Java utility library
 * ===========================================
 *
 * Project Info:  http://reporting.pentaho.org/libbase
 *
 * (C) Copyright 2007,2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 * ------------
 * StackableRuntimeException
 * ------------
 * (C) Copyright 2006, by Pentaho Corporation.
 */



package org.pentaho.reporting.libraries.base.util;

import java.io.PrintStream;
import java.io.PrintWriter;

/**
 * A baseclass for RuntimeExceptions, which could have parent exceptions. These parent exceptions
 * are raised in a subclass and are now wrapped into a subclass of this Exception.
 * <p/>
 * The parents are printed when this exception is printed. This class exists mainly for
 * debugging reasons, as with them it is easier to detect the root cause of an error.
 *
 * @author Thomas Morgner
 */
public class StackableRuntimeException extends RuntimeException
{

  /**
   * The parent exception.
   */
  private Exception parent;
  private static final long serialVersionUID = -4378774171699885841L;

  /**
   * Creates a StackableRuntimeException with no message and no parent.
   */
  public StackableRuntimeException()
  {
    super();
  }

  /**
   * Creates an exception.
   *
   * @param message the exception message.
   * @param ex      the parent exception.
   */
  public StackableRuntimeException(final String message, final Exception ex)
  {
    super(message);
    this.parent = ex;
  }

  /**
   * Creates an exception.
   *
   * @param message the exception message.
   */
  public StackableRuntimeException(final String message)
  {
    super(message);
  }

  /**
   * Returns the parent exception (possibly null).
   *
   * @return the parent exception.
   */
  public Exception getParent()
  {
    return this.parent;
  }

  /**
   * Prints the stack trace to the specified stream.
   *
   * @param stream the output stream.
   */
  public void printStackTrace(final PrintStream stream)
  {
    super.printStackTrace(stream);
    if (getParent() != null)
    {
      stream.println("ParentException: ");
      getParent().printStackTrace(stream);
    }
  }

  /**
   * Prints the stack trace to the specified writer.
   *
   * @param writer the writer.
   */
  public void printStackTrace(final PrintWriter writer)
  {
    super.printStackTrace(writer);
    if (getParent() != null)
    {
      writer.println("ParentException: ");
      getParent().printStackTrace(writer);
    }
  }

}