/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a query creator for skrooge
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgquerycreator.h"
#include "skgquerydelegate.h"
#include "skgservices.h"
#include "skgdocument.h"
#include "skgpredicatcreator.h"
#include "skgruleobject.h"

#include <QDomDocument>
#include <QHeaderView>

SKGQueryCreator::SKGQueryCreator(QWidget *parent)
                : QWidget(parent), document(NULL), updateMode(false)
{
        ui.setupUi(this);

        ui.kRemove->setIcon ( KIcon ( "edit-delete" ) );

        ui.kList->horizontalHeader()->setResizeMode ( QHeaderView::Interactive );
        ui.kList->setWordWrap ( false );


        QHeaderView* verticalHeader=ui.kList->verticalHeader();
        if (verticalHeader) {
                verticalHeader->setResizeMode(QHeaderView::Fixed);
                verticalHeader->setDefaultSectionSize(verticalHeader->minimumSectionSize());
        }

        addNewLine();
        onSelectionChanged();
}


SKGQueryCreator::~SKGQueryCreator()
{
        if (document) {
                document=NULL;
        }
}

void SKGQueryCreator::setParameters(const SKGDocument* iDocument, const QString& iTable, const QStringList& iListAttribute, bool iModeUpdate)
{
        document=(SKGDocument*) iDocument;
        table=iTable;
        updateMode=iModeUpdate;
        ui.kRemove->setVisible(!updateMode);

        //Build list of attributes
        if (document) {
                SKGQueryDelegate* delegate=new SKGQueryDelegate (ui.kList, document, updateMode, iListAttribute);
                connect ( delegate, SIGNAL ( closeEditor(QWidget*, QAbstractItemDelegate::EndEditHint)   ), this, SLOT ( onCloseEditor() ) );

                ui.kList->setItemDelegate(delegate);

                SKGServices::SKGAttributesList listAtts;
                SKGServices::SKGAttributesList attributes;
                SKGServices::getAttributesDescription(document, table, attributes);
                foreach(const SKGServices::SKGAttributeInfo& att, attributes) {
                        if (iListAttribute.count()==0 || iListAttribute.contains(att.name)) listAtts.push_back(att);
                }

                ui.kList->setRowCount(0);

                int nbCol=listAtts.count();
                ui.kList->setColumnCount(nbCol);
                for (int i=0; i<nbCol; ++i) {
                        QTableWidgetItem* item=new QTableWidgetItem(listAtts.at(i).icon, listAtts.at(i).display);
                        item->setData(Qt::UserRole, listAtts.at(i).name);
                        ui.kList->setHorizontalHeaderItem ( i, item );
                }

                addNewLine();
        }
}

int SKGQueryCreator::getIndexQueryColumn(const QString& iAttribute)
{
        int output=-1;
        int nbCol=ui.kList->columnCount();
        for (int i=0; i<nbCol; ++i) {
                QTableWidgetItem* it_h=ui.kList->horizontalHeaderItem(i);
                if (iAttribute==it_h->data(Qt::UserRole).toString()) {
                        output=i;
                        break;
                }
        }
        return output;
}

void SKGQueryCreator::clearContents()
{
        ui.kList->clearContents();
        ui.kList->setRowCount(0);

        addNewLine();
}

void SKGQueryCreator::setXMLCondition(const QString& iXML)
{
        QDomDocument doc("SKGML");
        doc.setContent(iXML);
        QDomElement root = doc.documentElement();

        ui.kList->clearContents();
        ui.kList->setRowCount(0);

        int row=-1;
        QDomNode l = root.firstChild();
        while (!l.isNull()) {
                QDomElement elementl = l.toElement();
                if (!elementl.isNull()) {
                        //Add new line
                        addNewLine();
                        ++row;

                        QDomNode n = elementl.firstChild();
                        while (!n.isNull()) {
                                QDomElement element = n.toElement();
                                if (!element.isNull()) {
                                        QString attribute=element.attribute ( "attribute");
                                        int idx=getIndexQueryColumn(attribute);
                                        if (idx>=0) {

                                                QDomDocument doc2 ( "SKGML" );
                                                QDomElement root2 = doc2.createElement ( "element" );
                                                doc2.appendChild ( root2 );
                                                root2.setAttribute ( "operator", element.attribute ( "operator"));
                                                root2.setAttribute ( "value", element.attribute ( "value") );
                                                root2.setAttribute ( "value2", element.attribute ( "value2") );
                                                root2.setAttribute ( "att2", element.attribute ( "att2") );
                                                root2.setAttribute ( "att2s", element.attribute ( "att2s") );
                                                QTableWidgetItem* it=ui.kList->item (row, idx);
                                                if (it) {
                                                        QString xml=doc2.toString();

                                                        it->setText(SKGPredicatCreator::getTextFromXml(xml));
                                                        it->setData(Qt::UserRole, xml);
                                                }
                                        }
                                }
                                n = n.nextSibling();
                        }
                }
                l = l.nextSibling();
        }

        addNewLine();
}

QString SKGQueryCreator::getXMLCondition()
{
        QString output;

        QDomDocument doc("SKGML");
        QDomElement element = doc.createElement ( "element" );
        doc.appendChild ( element );

        QDomComment com=doc.createComment ("OR");
        element.appendChild ( com );

        bool empty=true;
        int nbRow=ui.kList->rowCount();
        int nbCol=ui.kList->columnCount();
        for (int j=0; j<nbRow; ++j) {
                QDomElement elementLine = doc.createElement ( "element" );
                element.appendChild ( elementLine );

                QDomComment com=doc.createComment ("AND");
                elementLine.appendChild ( com );

                bool atLeastOne=false;
                for (int i=0; i<nbCol; ++i) {
                        QTableWidgetItem* it=ui.kList->item (j, i);
                        if (it) {
                                QString co=it->data(Qt::UserRole).toString();
                                if (!co.isEmpty()) {
                                        QDomElement elementElement = doc.createElement ( "element" );
                                        elementLine.appendChild ( elementElement );

                                        QTableWidgetItem* it_h=ui.kList->horizontalHeaderItem(i);
                                        QString attname=it_h->data(Qt::UserRole).toString();

                                        QDomDocument doc2("SKGML");
                                        doc2.setContent(co);
                                        QDomElement root2 = doc2.documentElement();

                                        elementElement.setAttribute ( "attribute", attname);
                                        elementElement.setAttribute ( "operator", root2.attribute ( "operator"));
                                        if (root2.hasAttribute("value")) elementElement.setAttribute ( "value", root2.attribute ( "value") );
                                        if (root2.hasAttribute("value2")) elementElement.setAttribute ( "value2", root2.attribute ( "value2") );
                                        if (root2.hasAttribute("att2")) elementElement.setAttribute ( "att2", root2.attribute ( "att2") );
                                        if (root2.hasAttribute("att2s")) elementElement.setAttribute ( "att2s", root2.attribute ( "att2s") );

                                        atLeastOne=true;
                                        empty=false;
                                }
                        }
                }

                if (!atLeastOne) element.removeChild ( elementLine );
        }
        if (!empty) output=doc.toString();

        return output;
}

void SKGQueryCreator::onCloseEditor()
{
        //If all lignes have at least one value the add a new line
        bool lineEmpty=false;
        int nbCol=ui.kList->columnCount();
        int nbRow=ui.kList->rowCount();
        for (int j=0; !lineEmpty && j<nbRow; ++j) {
                lineEmpty=true;
                for (int i=0; lineEmpty && i<nbCol; ++i) {
                        QTableWidgetItem* it=ui.kList->item(j, i);
                        if (!it->text().isEmpty()) lineEmpty=false;
                }
        }

        if (!lineEmpty) addNewLine();
}

void SKGQueryCreator::addNewLine()
{
        //add line is only for
        if (!updateMode || ui.kList->rowCount()<1) {
                bool previous=ui.kList->blockSignals ( true );

                int nbCol=ui.kList->columnCount();
                int row=ui.kList->rowCount();
                ui.kList->insertRow ( row );
                for (int i=0; i<nbCol; ++i) {
                        QTableWidgetItem* item = new QTableWidgetItem();
                        ui.kList->setItem(row, i, item );
                }
                ui.kList->blockSignals ( previous );

                ui.kList->resizeColumnsToContents();
        }
}

void SKGQueryCreator::onRemove()
{
        QList<int> rowsToRemove;
        QList<QTableWidgetItem*> selectedItems = ui.kList->selectedItems();
        int nb=selectedItems.count();
        for ( int i = 0; i < nb; ++i ) {
                QTableWidgetItem* item = selectedItems.at ( i );
                int row = item->row();
                if ( !rowsToRemove.contains ( row ) ) {
                        rowsToRemove.append ( row );
                }
        }
        for ( int j = rowsToRemove.count()-1; j >=0 ; --j ) {
                ui.kList->removeRow ( rowsToRemove.at ( j ) );
        }

        // If all rows removed, add an empty line
        if ( ui.kList->rowCount() == 0 ) {
                addNewLine();
        }
}

void SKGQueryCreator::onSelectionChanged()
{
        QList<QTableWidgetItem*> selectedItems = ui.kList->selectedItems();
        ui.kRemove->setEnabled ( selectedItems.count() >0 );
}

#include "skgquerycreator.moc"

