/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGDOCUMENTBANK_H
#define SKGDOCUMENTBANK_H
/** @file
 * This file is part of Skrooge and defines classes SKGDocumentBank.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "skgdocument.h"
#include "skgdefinebank.h"
#include "skgbankmodeler_export.h"

class SKGUnitObject;
class SKGUnitValueObject;

/**
 * This class manages skg bank documents
 */
class SKGBANKMODELER_EXPORT SKGDocumentBank : public SKGDocument
{
        Q_OBJECT;

public:
        /**
        * Constructor
        * @param iMode the mode of the database
        */
        SKGDocumentBank(DatabaseMode iMode=SKGDocument::CopiedInMemory);

        /**
        * Destructor
        */
        ~SKGDocumentBank();

        /**
         * dump the document in the std output.
         * It's useful for debug.
         * @param iMode to select what you want to dump.
         * @code
         * document->dump (DUMPUNIT|DUMPPARAMETERS);
         * @endcode
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError dump(int iMode = DUMPBANKOBJECT) const;

        /**
         * Create or modify an account
         * @param iName account name
         * @param iNumber account number
         * @param iBankName name of the bank
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError addOrModifyAccount(const QString& iName, const QString& iNumber, const QString& iBankName) const;

        /**
         * Rename an ccount
         * @param iName account name
         * @param inewName new account name
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError RenameAccount(const QString& iName, const QString& inewName) const;

        /**
         * delete an account
         * @param iName account name
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError deleteAccount(const QString& iName) const;

        /**
         * Create or modify the value of an unit
         * @param iUnitName unit name
         * @param iDate date
         * @param iValue unit value for the date @p iDate
         * @param oValue this output unit value
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError addOrModifyUnitValue(const QString& iUnitName, const QDate& iDate, double iValue, SKGUnitValueObject* oValue=NULL) const;

        /**
         * Update unit's attributes
         * @param iUnitName unit name
         * @param iInternetCode new internet code
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError ModifyUnit(const QString& iUnitName, const QString& iInternetCode) const;

        /**
         * Rename a unit
         * @param iUnitName old name of the unit
         * @param inewName new name of the unit
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError RenameUnit(const QString& iUnitName, const QString& inewName) const;

        /**
         * Delete a unit
         * @param iUnitName unitname
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError deleteUnit(const QString& iUnitName) const;

        /**
         * Get Primary unit. WARNING: This value can be not uptodated in a transaction.
         * @return Primary unit.
         */
        virtual QString getPrimaryUnit();

        /**
         * Get Secondary unit. WARNING: This value can be not uptodated in a transaction.
         * @return Secondary unit.
         */
        virtual QString getSecondaryUnit();

        /**
         * Get Secondary unit value. WARNING: This value can be not uptodated in a transaction.
         * @return Secondary unit value.
         */
        virtual double getSecondaryUnitValue();

        /**
         * Refresh all views and indexes in the database
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError refreshViewsIndexesAndTriggers() const;

        /**
         * Return the number version of views, indexes and triggers
         * @return
         */
        virtual QString getViewsIndexesAndTriggersVersion() const;

        /**
         * Get the display string for any modeler object (table, attribute)
         * @param iString the name of the object (example: v_operation, v_unit.t_name)
         * @return the display string
         */
        virtual QString getDisplay(const QString& iString) const;

        /**
         * Get the icon for attribute
         * @param iString the name of the attribute
         * @return the icon
         */
        virtual QIcon getIcon(const QString& iString) const;

        /**
         * Get the attribute type
         * @param iAttributeName the name of an attribute
         * @return the type
         */
        virtual SKGServices::AttibuteType getAttibuteType(const QString& iAttributeName) const;

protected:
        /**
         * Get impacted tables if one object of @p iTable is modifier.
         * @param iTable name of a table
         * @return impacted tables
         */
        virtual QStringList getImpactedTable(const QString& iTable) const;

        /**
         * Migrate the current SKGDocument to the latest version of the data model.
         * WARNING: This method must be used in a transaction.
         * @see beginTransaction
         * @param oMigrationDone to know if a migration has been done or not.
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError migrate(bool& oMigrationDone);

private slots:
        virtual void refreshCache(const QString& iTable);

private:
        Q_DISABLE_COPY(SKGDocumentBank);
};


#endif
