/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for import and export operation.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgimportexportplugin.h"
#include "skgtraces.h"
#include "skgerror.h"
#include "skgbankincludes.h"
#include "skgmainpanel.h"
#include "skgoperationobject.h"
#include "skgimportexport_settings.h"

#include <kactioncollection.h>
#include <kaction.h>
#include <kactionmenu.h>
#include <kfiledialog.h>
#include <kencodingfiledialog.h>

#include <QTextCodec>
#include <QDomDocument>

K_PLUGIN_FACTORY(SKGImportExportPluginFactory, registerPlugin<SKGImportExportPlugin>();)
K_EXPORT_PLUGIN(SKGImportExportPluginFactory("skrooge_importexport", "skrooge_importexport"))

SKGImportExportPlugin::SKGImportExportPlugin(QObject* iParent, const QVariantList& /*iArg*/) : SKGInterfacePlugin(iParent)
{
        SKGTRACEIN(10, "SKGImportExportPlugin::SKGImportExportPlugin");
}

SKGImportExportPlugin::~SKGImportExportPlugin()
{
        SKGTRACEIN(10, "SKGImportExportPlugin::~SKGImportExportPlugin");
        currentBankDocument=NULL;

        importAction=NULL;
        importCsvUnitAction=NULL;
        exportCsvAction=NULL;
        exportQifAction=NULL;
        processingFoundTransfert=NULL;
        processingBankPerfect=NULL;
        processingBank=NULL;
        validateImportedOperationsAction=NULL;
        openNotValidatedAction=NULL;
        mergeImportedOperationAction=NULL;
}

void SKGImportExportPlugin::setupActions(SKGDocument* iDocument, const QStringList& iArgument)
{
        SKGTRACEIN(10, "SKGImportExportPlugin::setupActions");
        Q_UNUSED(iArgument);

        currentBankDocument=iDocument;

        // Tell the host application to load my GUI component
        setComponentData( SKGImportExportPluginFactory::componentData() );
        setXMLFile("skrooge_importexport.rc");

        //Imports
        KActionMenu* imports=new  KActionMenu(KIcon("document-import"),i18nc("Verb, action to import items from another format","Import"), this);
        actionCollection()->addAction( QLatin1String("import"), imports );

        //Import
        QStringList overlayofx;
        overlayofx.push_back("skrooge");

        importAction = new KAction(KIcon("document-import", NULL, overlayofx), i18nc("Verb, action to import items from another format","&Import..."), this);
        importAction->setShortcut ( Qt::CTRL+Qt::ALT+Qt::Key_I );
        connect(importAction, SIGNAL(triggered(bool)), SLOT(import()));
        actionCollection()->addAction( QLatin1String("import_operation"), importAction );
        imports->addAction(importAction);

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("import_operation", importAction);

        //Import CSV Unit
        QStringList overlaycsv;
        overlaycsv.push_back("text-csv");
        importCsvUnitAction = new KAction(KIcon("document-import", NULL, overlaycsv), i18n("Import CSV &Unit..."), this);
        connect(importCsvUnitAction, SIGNAL(triggered(bool)), SLOT(import()));
        actionCollection()->addAction( QLatin1String("import_csv_unit"), importCsvUnitAction );
        imports->addAction(importCsvUnitAction);

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("import_csv_unit", importCsvUnitAction);

        //Exports
        KActionMenu* exports=new  KActionMenu(KIcon("document-export"),i18nc("Verb, action to export items in another format","Export"), this);
        actionCollection()->addAction( QLatin1String("export"), exports );

        //Export QIF
        exportQifAction = new KAction(KIcon("document-export"), i18n("Export &QIF..."), this);
        connect(exportQifAction, SIGNAL(triggered(bool)), SLOT(exportQif()));
        actionCollection()->addAction( QLatin1String("export_qif"), exportQifAction );
        exports->addAction(exportQifAction);

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("export_qif", exportQifAction);

        //Export CSV
        exportCsvAction = new KAction(KIcon("document-export", NULL, overlaycsv), i18n("Export &CSV..."), this);
        connect(exportCsvAction, SIGNAL(triggered(bool)), SLOT(exportCsv()));
        actionCollection()->addAction( QLatin1String("export_csv"), exportCsvAction );
        exports->addAction(exportCsvAction);

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("export_csv", exportCsvAction);

        //Processing
        KActionMenu* processing=new  KActionMenu(KIcon("tools-wizard"),i18nc("Noun, apply some kind of transformation on an item","Processing"), this);
        actionCollection()->addAction( QLatin1String("processing"), processing );

        //Processing found and group
        QStringList overlaytransfers;
        overlaytransfers.push_back("skrooge_transfer");

        processingFoundTransfert = new KAction(KIcon("tools-wizard", NULL, overlaytransfers), i18n("Find and group transfers"), this);
        connect(processingFoundTransfert, SIGNAL(triggered(bool)), SLOT(foundTransfer()));
        actionCollection()->addAction( QLatin1String("process_foundtransfer"), processingFoundTransfert );
        processing->addAction(processingFoundTransfert);

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("process_foundtransfer", processingFoundTransfert);

        //Processing bankperfect
        processingBankPerfect = new KAction(KIcon("tools-wizard"), i18n("Clean BankPerfect's imports"), this);
        connect(processingBankPerfect, SIGNAL(triggered(bool)), SLOT(cleanBankPerfect()));
        actionCollection()->addAction( QLatin1String("process_bankperfect"), processingBankPerfect );
        processing->addAction(processingBankPerfect);

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("process_bankperfect", processingBankPerfect);

        //Processing banks
        processingBank = new KAction(KIcon("tools-wizard"), i18n("Clean bank's imports"), this);
        connect(processingBank, SIGNAL(triggered(bool)), SLOT(cleanBanks()));
        actionCollection()->addAction( QLatin1String("process_banks"), processingBank );
        processing->addAction(processingBank);

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("process_banks", processingBank);

        //Processing banks
        QStringList overlayValidate;
        overlayValidate.push_back("dialog-ok-apply");
        validateImportedOperationsAction = new KAction(KIcon("document-import", NULL, overlayValidate), i18n("Validate imported operations"), this);
        connect(validateImportedOperationsAction, SIGNAL(triggered(bool)), SLOT(validateImportedOperations()));
        actionCollection()->addAction( QLatin1String("validate_imported_operation"), validateImportedOperationsAction );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("validate_imported_operation", validateImportedOperationsAction);

        openNotValidatedAction = new KAction ( KIcon("skrooge_open_imported"), i18n ( "Open imported operations not yet validated" ), this);
        connect ( openNotValidatedAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( openNotValidated() ) );
        actionCollection()->addAction ( QLatin1String ( "view_open_not_validated" ), openNotValidatedAction );
        openNotValidatedAction->setShortcut ( Qt::CTRL+Qt::SHIFT+Qt::Key_V );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "view_open_not_validated", openNotValidatedAction );

        mergeImportedOperationAction = new KAction(KIcon("skrooge_merge_import"), i18n("Merge imported operations"), this);
        connect(mergeImportedOperationAction, SIGNAL(triggered(bool)), SLOT(mergeImportedOperation()));
        actionCollection()->addAction( QLatin1String("merge_imported_operation"), mergeImportedOperationAction );
        mergeImportedOperationAction->setShortcut ( Qt::CTRL+Qt::Key_M );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("merge_imported_operation", mergeImportedOperationAction);
}

void SKGImportExportPlugin::processArguments(const QStringList& iArgument)
{
        int nbArg=iArgument.count();
        if (nbArg) {
                QString filename=iArgument.at(nbArg-1);
                QString extension=QFileInfo(filename).suffix().toUpper();
                if (QFile(filename).exists() && (extension=="QIF" || extension=="CSV" || extension=="OFX" || extension=="QFX")) {
                        import(filename);
                }
        }
}

QWidget* SKGImportExportPlugin::getPreferenceWidget()
{
        SKGTRACEIN(10, "SKGImportExportPlugin::getPreferenceWidget");
        QWidget* widget=new QWidget();
        ui.setupUi(widget);

        ui.kHeaderPositionFrm->hide();
        ui.kColumnsPositionsFrm->hide();

        return widget;
}

KConfigSkeleton* SKGImportExportPlugin::getPreferenceSkeleton()
{
        return skgimportexport_settings::self();
}

QString SKGImportExportPlugin::title() const
{
        return i18n("Import / Export");
}

QString SKGImportExportPlugin::icon() const
{
        return "utilities-file-archiver";
}

QString SKGImportExportPlugin::statusTip () const
{
        return i18n("Import / Export management (CSV, QIF ...)");
}

QString SKGImportExportPlugin::toolTip () const
{
        return i18n("Import / Export management");
}

QStringList SKGImportExportPlugin::tips() const
{
        QStringList output;
        output.push_back(i18n("<p>... skrooge is able to detect automatically transfers after an import.</p>"));
        output.push_back(i18n("<p>... you can import many files in one shot.</p>"));
        output.push_back(i18n("<p>... unit amounts can be imported through a CSV file.</p>"));
        return output;
}

int SKGImportExportPlugin::getOrder() const
{
        return 70;
}

bool SKGImportExportPlugin::isInContext() const
{
        return false;
}

void SKGImportExportPlugin::refresh()
{
        SKGTRACEIN(10, "SKGImportExportPlugin::refresh");

        if (currentBankDocument && SKGMainPanel::getMainPanel()) {
                bool test=(currentBankDocument->getDatabase()!=NULL);
                if (openNotValidatedAction) openNotValidatedAction->setEnabled (test);
                if (importAction) importAction->setEnabled(test);
                if (exportQifAction) exportQifAction->setEnabled(test);
                if (exportCsvAction) exportCsvAction->setEnabled(test);
                if (importCsvUnitAction) importCsvUnitAction->setEnabled(test);
                if (processingFoundTransfert) processingFoundTransfert->setEnabled(test);
                if (processingBankPerfect) processingBankPerfect->setEnabled(test);
                if (processingBank) processingBank->setEnabled(test);

                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                if ( selection.count() >0 && test) {
                        bool onOperation= ( selection.at ( 0 ).getRealTable() =="operation" );
                        if ( validateImportedOperationsAction ) validateImportedOperationsAction->setEnabled ( onOperation );
                        if ( mergeImportedOperationAction ) mergeImportedOperationAction->setEnabled ( onOperation );
                } else {
                        if ( validateImportedOperationsAction ) validateImportedOperationsAction->setEnabled ( false );
                        if ( mergeImportedOperationAction ) mergeImportedOperationAction->setEnabled ( false );
                }
        }
}

void SKGImportExportPlugin::close()
{
        SKGTRACEIN(10, "SKGImportExportPlugin::close");
}

void SKGImportExportPlugin::import(const QString& iFile)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::import",err);
        if (currentBankDocument) {
                QSet<QString> fileNames;
                QString lastCodecUsed=currentBankDocument->getParameter("SKG_LAST_CODEC_USED_FOR_IMPORT");
                if (lastCodecUsed.length()==0) lastCodecUsed=QTextCodec::codecForLocale()->name();
                QString codec;

                if (iFile.isEmpty()) {
                        //Panel to ask files
                        KEncodingFileDialog::Result result=KEncodingFileDialog::getOpenFileNamesAndEncoding(lastCodecUsed, "kfiledialog:///IMPEXP",
                                                           sender()==importCsvUnitAction ? "*.csv|"+i18n("CSV Files") :
                                                           "*.ofx *.qfx *.qif *.csv|"+i18n("All supported formats")+
                                                           "\n*.ofx *.qfx|"+i18n("OFX Files")+
                                                           "\n*.qif|"+i18n("QIF Files")+
                                                           "\n*.csv|"+i18n("CSV Files") ,
                                                           SKGMainPanel::getMainPanel());
                        fileNames=result.fileNames.toSet();
                        codec=result.encoding;
                } else {
                        fileNames.insert(iFile);
                        QString codec=lastCodecUsed;
                }

                int nbFiles=fileNames.count();
                if (nbFiles) {

                        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                        {
                                SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18n("Import with codec %1", codec), err, nbFiles);

                                //Read Setting
                                KSharedConfigPtr config=KSharedConfig::openConfig("skrooge_importexportrc");
                                KConfigGroup pref=config->group("skrooge_importexport");
                                bool automatic_validation = pref.readEntry("automatic_validation",false);

                                if (err.isSucceeded()) err=currentBankDocument->setParameter("SKG_LAST_CODEC_USED_FOR_IMPORT", codec);

                                QSetIterator<QString> f(fileNames);
                                int i=1;
                                while (err.isSucceeded() && f.hasNext()) {
                                        //Get Filename
                                        QString fileName=f.next();

                                        //Compute mode
                                        SKGImportExportManager::ImportExportMode mode=SKGImportExportManager::CSV;
                                        if (sender()==importCsvUnitAction) mode=SKGImportExportManager::CSVUNIT;
                                        QString extension=QFileInfo(fileName).suffix().toUpper();
                                        if (extension=="QIF") mode=SKGImportExportManager::QIF;
                                        else if (extension=="OFX" || extension=="QFX") mode=SKGImportExportManager::OFX;

                                        //Import
                                        SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, mode, fileName);
                                        imp1.setAutomaticValidation(automatic_validation);
                                        imp1.setCodec(codec);

                                        if (mode==SKGImportExportManager::CSV || mode==SKGImportExportManager::CSVUNIT) {
                                                //Set header line
                                                bool automatic_search_header = pref.readEntry("automatic_search_header", true);
                                                if (!automatic_search_header) {
                                                        int header_position = pref.readEntry("header_position", 1);
                                                        if (err.isSucceeded()) err=imp1.setCSVHeaderIndex(header_position-1);

                                                }

                                                bool automatic_search_columns = pref.readEntry("automatic_search_columns", true);
                                                if (!automatic_search_columns) {
                                                        QStringList columns_positions = pref.readEntry("columns_positions", "").split('|');
                                                        if (err.isSucceeded()) err=imp1.setCSVMapping(&columns_positions);
                                                }
                                        }

                                        if (err.isSucceeded()) err=imp1.importFile();
                                        if (err.isFailed()) {
                                                err.addError(ERR_FAIL, i18n("Import of %1 file named [%2] failed", extension, fileName));
                                        }

                                        if (err.isSucceeded()) err=currentBankDocument->stepForward(i);
                                        i++;
                                }
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if (err.isSucceeded()) {
                        err=SKGError(0, i18np("%1 file successfully imported.", "%1 files successfully imported.", nbFiles));
                }

                //Display error
                if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
        }
}

void SKGImportExportPlugin::exportCsv()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::exportCsv",err);
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "*.csv|"+i18n("CSV Files") , SKGMainPanel::getMainPanel());
        if (fileName.isEmpty() || !currentBankDocument) return;

        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                SKGBEGINTRANSACTION(*currentBankDocument, i18n("Export CSV"), err);
                if (err.isSucceeded()) {
                        SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::CSV, fileName);
                        err=imp1.exportFile();
                }
        }
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("File [%1] successfully exported.", fileName));
        else err.addError(ERR_FAIL, i18n("Export of [%1] failed", fileName));

        //Display error
        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGImportExportPlugin::exportQif()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::exportQif",err);
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "*.qif|"+i18n("QIF Files") , SKGMainPanel::getMainPanel());
        if (fileName.isEmpty() || !currentBankDocument) return;

        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                SKGBEGINTRANSACTION(*currentBankDocument, i18n("Export QIF"), err);
                if (err.isSucceeded()) {
                        SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::QIF, fileName);
                        err=imp1.exportFile();
                }
        }
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("File [%1] successfully exported.", fileName));
        else err.addError(ERR_FAIL, i18n("Export of [%1] failed", fileName));

        //Display error
        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGImportExportPlugin::foundTransfer()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::foundTransfer",err);
        if (currentBankDocument) {
                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                int NbOperationsMerged=0;
                {
                        SKGBEGINTRANSACTION(*currentBankDocument, i18n("Find and group transfers"), err);
                        if (err.isSucceeded()) {
                                SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::QIF, "");
                                err=imp1.findAndGroupTransfers(NbOperationsMerged);
                        }
                }
                QApplication::restoreOverrideCursor();

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18np("Document successfully processed. %1 operation merged.",
                                                            "Document successfully processed. %1 operations merged.", NbOperationsMerged));
                else err.addError(ERR_FAIL, i18n("Processing failed."));

                //Display error
                if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
        }
}

void SKGImportExportPlugin::cleanBankPerfect()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::cleanBankPerfect",err);
        if (currentBankDocument) {
                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                {
                        SKGBEGINTRANSACTION(*currentBankDocument, i18n("Clean BankPerfect's imports"), err);
                        if (err.isSucceeded()) {
                                SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::QIF, "");
                                err=imp1.cleanBankPerfectImport();
                        }
                }
                QApplication::restoreOverrideCursor();

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18n("Document successfully cleaned."));
                else err.addError(ERR_FAIL, i18n("Clean failed."));

                //Display error
                if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
        }
}

void SKGImportExportPlugin::cleanBanks()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::cleanBanks",err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                SKGBEGINTRANSACTION(*currentBankDocument, i18n("Clean bank's imports"), err);
                if (err.isSucceeded()) {
                        SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::QIF, "");
                        err=imp1.cleanBankImport();
                }
        }
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("Document successfully cleaned."));
        else err.addError(ERR_FAIL, i18n("Clean failed."));

        //Display error
        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGImportExportPlugin::validateImportedOperations()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::validateImportedOperations",err);

        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        if (SKGMainPanel::getMainPanel() && currentBankDocument) {
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18n("Validate imported operations"), err, nb);
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        SKGOperationObject op=selection[i];
                        if (op.getAttribute("t_imported")=="P") {
                                err=op.setImported(true);
                                if (err.isSucceeded()) err=op.save();
                        }
                        if (err.isSucceeded()) err=currentBankDocument->stepForward(i+1);
                }

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18n("Imported operations validated."));
                else err.addError(ERR_FAIL, i18n("Validation failed"));
        }
        QApplication::restoreOverrideCursor();

        //Display error
        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGImportExportPlugin::mergeImportedOperation()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::mergeImportedOperation",err);

        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        if (SKGMainPanel::getMainPanel() && currentBankDocument) {
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                err=SKGError(ERR_INVALIDARG, i18n("Invalid selection, you must select one imported operation and one manual operation with same amounts"));
                if (nb==2) {
                        SKGOperationObject opImported=selection.at(0);
                        SKGOperationObject opManual=selection.at(1);
                        if (opImported.isImported()!=opManual.isImported() && opImported.getCurrentAmount()==opManual.getCurrentAmount()) {
                                if (!opImported.isImported()) {
                                        qSwap(opImported, opManual);
                                }

                                SKGBEGINTRANSACTION(*currentBankDocument, i18n("Merge imported operations"), err);
                                err=opManual.setDate(opImported.getDate());
                                if (err.isSucceeded()) err=opManual.setImported(true);
                                if (err.isSucceeded()) err=opManual.setImportID(opImported.getImportID());
                                if (err.isSucceeded() && opManual.getComment().isEmpty()) err=opManual.setComment(opImported.getComment());
                                if (err.isSucceeded() && opManual.getPayee().isEmpty()) err=opManual.setPayee(opImported.getPayee());
                                if (err.isSucceeded() && opManual.getMode().isEmpty()) err=opManual.setMode(opImported.getMode());
                                if (err.isSucceeded() && !opManual.isBookmarked()) err=opManual.bookmark(opImported.isBookmarked());
                                if (err.isSucceeded() && !opManual.getNumber()) err=opManual.setNumber(opImported.getNumber());
                                if (err.isSucceeded()) err=opManual.save();
                                if (err.isSucceeded()) err=opImported.remove();

                                if (err.isFailed()) err.addError(ERR_FAIL, i18n("Merge failed"));
                        }
                }

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18n("Imported operations merged."));
        }
        QApplication::restoreOverrideCursor();

        //Display error
        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGImportExportPlugin::openNotValidated()
{
        SKGTRACEIN ( 10, "SKGImportExportPlugin::openNotValidated");
        QString wc="t_imported='P'";
        QString title=i18n ( "Operations imported and not yet validated" );

        //Call operation plugin
        QDomDocument doc("SKGML");
        doc.setContent ( currentBankDocument->getParameter("SKGOPERATION_DEFAULT_PARAMETERS") );
        QDomElement root = doc.documentElement();
        if (root.isNull()) {
                root=doc.createElement("parameters");
                doc.appendChild(root);
        }

        root.setAttribute ( "operationTable", "v_operation_display" );
        root.setAttribute ( "operationWhereClause", wc );
        root.setAttribute ( "title", title );
        root.setAttribute ( "title_icon", icon() );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
}

#include "skgimportexportplugin.moc"
