/* Copyright (c) 2002-2009 Dovecot Sieve authors, see the included COPYING file
 */

#include "lib.h"
#include "mempool.h"
#include "imem.h"
#include "array.h"
#include "unlink-directory.h"
#include "env-util.h"
#include "mail-namespace.h"
#include "mail-storage.h"

#include "sieve-common.h" 
#include "sieve-error.h"
#include "sieve-interpreter.h"
 
#include "testsuite-message.h"
#include "testsuite-common.h"
#include "testsuite-smtp.h"

#include "testsuite-mailstore.h"

#include <sys/stat.h>
#include <sys/types.h>

/*
 * Forward declarations
 */

static void testsuite_mailstore_close(void);

/*
 * State
 */

static char *testsuite_mailstore_tmp = NULL;
static struct mail_user *testsuite_mailstore_user = NULL;

static char *testsuite_mailstore_folder = NULL;
static struct mailbox *testsuite_mailstore_box = NULL;
static struct mailbox_transaction_context *testsuite_mailstore_trans = NULL;
static struct mail *testsuite_mailstore_mail = NULL;

/*
 * Initialization
 */

void testsuite_mailstore_init(const char *user, const char *home)
{	
	testsuite_mailstore_tmp = i_strconcat
		(testsuite_tmp_dir_get(), "/mailstore", NULL);

	if ( mkdir(testsuite_mailstore_tmp, 0700) < 0 ) {
		i_fatal("failed to create temporary directory '%s': %m.", 
			testsuite_mailstore_tmp);		
	}

	env_put(t_strdup_printf("NAMESPACE_1=maildir:%s", testsuite_mailstore_tmp));
	env_put("NAMESPACE_1_INBOX=1");
	env_put("NAMESPACE_1_LIST=1");
	env_put("NAMESPACE_1_SEP=.");
	env_put("NAMESPACE_1_SUBSCRIPTIONS=1");

	testsuite_mailstore_user = mail_user_init(user);
	mail_user_set_home(testsuite_mailstore_user, home);
	if (mail_namespaces_init(testsuite_mailstore_user) < 0)
		i_fatal("Namespace initialization failed");	
}

void testsuite_mailstore_deinit(void)
{
	testsuite_mailstore_close();

	/* De-initialize mail user object */
	if ( testsuite_mailstore_user != NULL )
		mail_user_unref(&testsuite_mailstore_user);

	if ( unlink_directory(testsuite_mailstore_tmp, TRUE) < 0 ) {
		i_warning("failed to remove temporary directory '%s': %m.",
			testsuite_mailstore_tmp);
	}
	
	i_free(testsuite_mailstore_tmp);		
}

void testsuite_mailstore_reset(void)
{
}

/*
 * Mailbox Access
 */

struct mail_namespace *testsuite_mailstore_get_namespace(void)
{
	return testsuite_mailstore_user->namespaces;
}

bool testsuite_mailstore_mailbox_create
(const struct sieve_runtime_env *renv ATTR_UNUSED, const char *folder)
{
	struct mail_storage *storage = testsuite_mailstore_user->namespaces->storage;

	/* Try creating it */
	if ( mail_storage_mailbox_create(storage, folder, FALSE) < 0 )
		return FALSE;

	return TRUE;
}

static void testsuite_mailstore_close(void)
{
	if ( testsuite_mailstore_mail != NULL )
		mail_free(&testsuite_mailstore_mail);

	if ( testsuite_mailstore_trans != NULL )
		mailbox_transaction_rollback(&testsuite_mailstore_trans);
		
	if ( testsuite_mailstore_box != NULL )
		mailbox_close(&testsuite_mailstore_box);

	if ( testsuite_mailstore_folder != NULL )
		i_free(testsuite_mailstore_folder);
}

static struct mail *testsuite_mailstore_open(const char *folder)
{
	enum mailbox_open_flags open_flags = 
		MAILBOX_OPEN_FAST | MAILBOX_OPEN_KEEP_RECENT | 
		MAILBOX_OPEN_SAVEONLY | MAILBOX_OPEN_POST_SESSION;
	struct mail_storage *storage = testsuite_mailstore_user->namespaces->storage;
	struct mailbox *box;
	struct mailbox_transaction_context *t;

	if ( testsuite_mailstore_mail == NULL ) {
		testsuite_mailstore_close();
	} else if ( testsuite_mailstore_folder != NULL 
		&& strcmp(folder, testsuite_mailstore_folder) != 0  ) {
		testsuite_mailstore_close();	
	} else {
		return testsuite_mailstore_mail;
	}

	box = mailbox_open(&storage, folder, NULL, open_flags);
	if ( box == NULL ) {
		sieve_sys_error("testsuite: failed to open mailbox '%s'", folder);
		return NULL;	
	}
	
	/* Sync mailbox */

	if ( mailbox_sync(box, MAILBOX_SYNC_FLAG_FULL_READ, 0, NULL) < 0 ) {
		sieve_sys_error("testsuite: failed to sync mailbox '%s'", folder);
		return NULL;
	}

	/* Start transaction */

	t = mailbox_transaction_begin(box, 0);

	testsuite_mailstore_folder = i_strdup(folder);
	testsuite_mailstore_box = box;
	testsuite_mailstore_trans = t;
	testsuite_mailstore_mail = mail_alloc(t, 0, NULL);

	return testsuite_mailstore_mail;
}

bool testsuite_mailstore_mail_index
(const struct sieve_runtime_env *renv, const char *folder, unsigned int index)
{
	struct mail *mail = testsuite_mailstore_open(folder);

	if ( mail == NULL )
		return FALSE;

	mail_set_seq(mail, index+1);
	testsuite_message_set_mail(renv, mail);

	return TRUE;
}
