/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: sharedstringsbuffer.hxx,v $
 *
 *  $Revision: 1.1.2.9 $
 *
 *  last change: $Author: dr $ $Date: 2007/08/24 09:06:55 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef OOX_XLS_SHAREDSTRINGSBUFFER_HXX
#define OOX_XLS_SHAREDSTRINGSBUFFER_HXX

#include <vector>
#include "oox/core/containerhelper.hxx"
#include "oox/xls/globaldatahelper.hxx"
#include "oox/xls/stylesbuffer.hxx"

namespace com { namespace sun { namespace star {
    namespace text { class XText; }
} } }

namespace oox { namespace core {
    class AttributeList;
} }

namespace oox {
namespace xls {

// ============================================================================

/** Represents a position in a rich-string containing current font identifier.

    This object stores the position of a formatted character in a rich-string
    and the identifier of a font from the global font list used to format this
    and the following characters.
 */
struct BiffRichStringFontId
{
    sal_Int32           mnPos;          /// First character in the string.
    sal_Int32           mnFontId;       /// Font identifier for the next characters.

    explicit inline     BiffRichStringFontId() : mnPos( 0 ), mnFontId( -1 ) {}
    explicit inline     BiffRichStringFontId( sal_Int32 nPos, sal_Int32 nFontId ) :
                            mnPos( nPos ), mnFontId( nFontId ) {}
};

bool operator==( const BiffRichStringFontId& rLeft, const BiffRichStringFontId& rRight );
bool operator<( const BiffRichStringFontId& rLeft, const BiffRichStringFontId& rRight );

// ----------------------------------------------------------------------------

/** A vector with all font identifiers in a rich-string. */
typedef ::std::vector< BiffRichStringFontId > BiffRichStringFontIdVec;

// ============================================================================

/** Contains text data and font attributes for a part of a rich formatted string. */
class RichStringPortion : public GlobalDataHelper
{
public:
    explicit            RichStringPortion( const GlobalDataHelper& rGlobalData );

    /** Sets text data for this portion. */
    void                setText( const ::rtl::OUString& rText );
    /** Creates and returns a new font formatting object. */
    FontRef             importFont( const ::oox::core::AttributeList& rAttribs );
    /** Links this portion to a font object from the global font list. */
    void                setFontId( sal_Int32 nFontId );

    /** Final processing after import of all strings. */
    void                finalizeImport();

    /** Converts the portion and appends it to the passed XText. */
    void                convert(
                            const ::com::sun::star::uno::Reference< ::com::sun::star::text::XText >& rxText,
                            sal_Int32 nXfId );

private:
    ::rtl::OUString     maText;         /// Portion text.
    FontRef             mxFont;         /// Embedded portion font, may be empty.
    sal_Int32           mnFontId;       /// Link to global font list.
};

typedef ::boost::shared_ptr< RichStringPortion > RichStringPortionRef;

// ============================================================================

/** Contains text data and positioning information for a phonetic text portion. */
class RichStringPhoneticPortion : public GlobalDataHelper
{
public:
    explicit            RichStringPhoneticPortion( const GlobalDataHelper& rGlobalData );

    /** Sets text data for this phonetic portion. */
    void                setText( const ::rtl::OUString& rText );

private:
    ::rtl::OUString     maText;         /// Portion text.
};

typedef ::boost::shared_ptr< RichStringPhoneticPortion > RichStringPhoneticPortionRef;

// ============================================================================

/** Contains string data and a list of formatting runs for a rich formatted string. */
class RichString : public GlobalDataHelper
{
public:
    explicit            RichString( const GlobalDataHelper& rGlobalData );

    /** Appends and returns a portion object for a plain string (t element). */
    RichStringPortionRef importText( const ::oox::core::AttributeList& rAttribs );
    /** Appends and returns a portion object for a new formatting run (r element). */
    RichStringPortionRef importPortion( const ::oox::core::AttributeList& rAttribs );

    /** Insert a font identifier to the passed vector. */
    static void         appendFontId( BiffRichStringFontIdVec& orFontIds, sal_Int32 nPos, sal_Int32 nFontId );
    /** Reads nCount font identifiers from stream and inserts them into the passed vector. */
    static void         importFontIds( BiffRichStringFontIdVec& orFontIds, BiffInputStream& rStrm, sal_uInt16 nCount, bool b16Bit );
    /** Reads count and font identifiers from stream and inserts them into the passed vector. */
    static void         importFontIds( BiffRichStringFontIdVec& orFontIds, BiffInputStream& rStrm, bool b16Bit );

    /** Imports a byte string from the passed BIFF stream. */
    void                importByteString( BiffInputStream& rStrm, rtl_TextEncoding eDefaultTextEnc, BiffStringFlags nFlags = BIFF_STR_DEFAULT );
    /** Imports a Unicode rich-string from the passed BIFF stream. */
    void                importUniString( BiffInputStream& rStrm, BiffStringFlags nFlags = BIFF_STR_DEFAULT );

    /** Final processing after import of all strings. */
    void                finalizeImport();

    /** Converts the string and writes it into the passed XText. */
    void                convert(
                            const ::com::sun::star::uno::Reference< ::com::sun::star::text::XText >& rxText,
                            sal_Int32 nXfId ) const;

private:
    /** Creates, appends, and returns a new empty string portion. */
    RichStringPortionRef createPortion();

private:
    typedef ::oox::core::RefVector< RichStringPortion > PortionVec;
    PortionVec          maPortions;     /// String portions with font data.
};

typedef ::boost::shared_ptr< RichString > RichStringRef;

// ============================================================================

/** Collects all strings from the shared strings substream. */
class SharedStringsBuffer : public GlobalDataHelper
{
public:
    explicit            SharedStringsBuffer( const GlobalDataHelper& rGlobalData );

    /** Creates and returns a new string entry. */
    RichStringRef       importString( const ::oox::core::AttributeList& rAttribs );
    /** Imports the complete shared string table from a BIFF file. */
    void                importSst( BiffInputStream& rStrm );

    /** Final processing after import of all strings. */
    void                finalizeImport();

    /** Converts the specified string table entry. */
    void                convertString(
                            const ::com::sun::star::uno::Reference< ::com::sun::star::text::XText >& rxText,
                            sal_Int32 nIndex,
                            sal_Int32 nXfId ) const;

private:
    typedef ::oox::core::RefVector< RichString > StringVec;
    StringVec           maStrings;
};

// ============================================================================

} // namespace xls
} // namespace oox

#endif

