/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to search and process operations
 *
 * @author Stephane MANKOWSKI
 */
#include "skgsearchplugin.h"
#include "skgsearchpluginwidget.h"
#include "skgsearch_settings.h"
#include "skgtraces.h"
#include "skgerror.h"
#include "skgruleobject.h"
#include "skgmainpanel.h"
#include "skgtransactionmng.h"
#include "skgalarmboardwidget.h"

#include <kactioncollection.h>
#include <kstandardaction.h>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY(SKGSearchPluginFactory, registerPlugin<SKGSearchPlugin>();)
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN(SKGSearchPluginFactory("skrooge_search", "skrooge_search"))

SKGSearchPlugin::SKGSearchPlugin(QObject* iParent, const QVariantList& /*iArg*/) : SKGInterfacePlugin(iParent)
{
        SKGTRACEIN(10, "SKGSearchPlugin::SKGSearchPlugin");
        timer = new QTimer( this );
        timer->setSingleShot( true );
        connect( timer, SIGNAL( timeout() ), this, SLOT( raiseAlarms() ) );
}

SKGSearchPlugin::~SKGSearchPlugin()
{
        SKGTRACEIN(10, "SKGSearchPlugin::~SKGSearchPlugin");
        currentBankDocument=NULL;
        timer=NULL;
}

int SKGSearchPlugin::getNbDashboardWidgets()
{
        return 1;
}

QString SKGSearchPlugin::getDashboardWidgetTitle(int iIndex)
{
        if (iIndex==0) return i18nc("Noun, alarms","Alarms");
        return "";
}

SKGWidget* SKGSearchPlugin::getDashboardWidget(int iIndex)
{
        if (iIndex==0) return new SKGAlarmBoardWidget(currentBankDocument);
        return NULL;
}

void SKGSearchPlugin::setupActions(SKGDocument* iDocument, const QStringList& iArgument)
{
        SKGTRACEIN(10, "SKGSearchPlugin::setupActions");
        Q_UNUSED(iArgument);

        currentBankDocument=iDocument;


        setComponentData( SKGSearchPluginFactory::componentData() );
        setXMLFile("skrooge_search.rc");

        //Create yours actions here
}

void SKGSearchPlugin::refresh()
{
        SKGTRACEIN(10, "SKGSearchPlugin::refresh");
        if (SKGMainPanel::getMainPanel()) {
                //Start alarm
                if ( currentBankDocument && currentBankDocument->getDatabase() !=NULL ) {
                        QString doc_id=currentBankDocument->getUniqueIdentifier();
                        if (docUniqueIdentifier!=doc_id) {
                                docUniqueIdentifier=doc_id;

                                raiseAlarms();
                        }
                }
        }
}

void SKGSearchPlugin::raiseAlarms()
{
        SKGObjectBase::SKGListSKGObjectBase rules;
        SKGError err=SKGObjectBase::getObjects(currentBankDocument, "v_rule", "t_action_type='A' ORDER BY i_ORDER", rules);
        int nb=rules.count();
        if (err.isSucceeded() && nb) {
                SKGBEGINTRANSACTION ( *currentBankDocument, "", err );
                QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        SKGRuleObject rule=rules.at(i);
                        rule.execute();
                }
                QApplication::restoreOverrideCursor();
        }

        //Display error
        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage ( err );

        if (timer) timer->start(skgsearch_settings::alarm_frequency()*60*1000 );
}

void SKGSearchPlugin::close()
{
        SKGTRACEIN(10, "SKGSearchPlugin::close");
}

SKGTabPage* SKGSearchPlugin::getWidget()
{
        SKGTRACEIN(10, "SKGSearchPlugin::getWidget");
        return new SKGSearchPluginWidget(currentBankDocument);
}

QWidget* SKGSearchPlugin::getPreferenceWidget()
{
        SKGTRACEIN(10, "SKGSearchPlugin::getPreferenceWidget");
        QWidget* widget=new QWidget();
        ui.setupUi(widget);
        return widget;
}

KConfigSkeleton* SKGSearchPlugin::getPreferenceSkeleton()
{
        return skgsearch_settings::self();
}

QString SKGSearchPlugin::title() const
{
        return i18n("Search and process");
}

QString SKGSearchPlugin::icon() const
{
        return "edit-find";
}

QString SKGSearchPlugin::toolTip () const
{
        return i18n("Search and process management");
}

int SKGSearchPlugin::getOrder() const
{
        return 35;
}

QStringList SKGSearchPlugin::tips() const
{
        QStringList output;
        output.push_back(i18n("<p>... skrooge can search and automatically process some operations.</p>"));
        output.push_back(i18n("<p>... you can create alarms based on searches.</p>"));
        return output;
}

bool SKGSearchPlugin::isInContext() const
{
        return true;
}

#include "skgsearchplugin.moc"
