# Copyright (c) 2010 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.
"""A generator for initializing_coclass.h, which contains a bunch of
repeated code that can't be produced through the preprocessor."""

import sys
from string import Template

HEADER_ = """\
// This file is autogenerated by initializing_coclass.py, do not edit.
#ifndef CEEE_COMMON_INITIALIZING_COCLASS_GEN_INL_
#define CEEE_COMMON_INITIALIZING_COCLASS_GEN_INL_

"""

FOOTER_ = """\
#endif // CEEE_COMMON_INITIALIZING_COCLASS_GEN_INL_
"""

# This template is used to generate a pair of template functions for N
# initialization parameters.
TEMPLATE_ = Template('''\
  // Creates a new instance of ImplClass
  // @returns S_OK on success, an appropriate error on failure.
  // @param new_instance on success returns the newly created and initialized
  //     ImplClass instance.
  // @note *new_instance has zero references, and as result the use of this
  //     function is highly discouraged as the existence of *new_instance
  //     on return of this function is quite brittle.
  template <${templ_params}>
  static HRESULT CreateInstance(${params}, T** new_instance) {
    TImpl* instance;
    HRESULT hr = TImpl::CreateInstance(&instance);
    if (FAILED(hr))
      return hr;

    instance->InternalFinalConstructAddRef();
    hr = instance->Initialize(${args});
    instance->InternalFinalConstructRelease();
    if (FAILED(hr)) {
      delete instance;
      instance = NULL;
    }
    *new_instance = instance;

    return hr;
  }

  template <class I, ${templ_params}>
  static HRESULT CreateInitialized(${params}, I** instance) {
    T* new_instance;
    HRESULT hr = CreateInstance(${args}, &new_instance);
    if (FAILED(hr))
      return hr;

    hr = new_instance->QueryInterface(__uuidof(I),
                                      reinterpret_cast<void**>(instance));
    if (FAILED(hr))
      delete new_instance;

    return hr;
  }

  template <class I, ${templ_params}>
  static HRESULT CreateInitializedIID(${params}, REFIID iid, I** instance) {
    T* new_instance;
    HRESULT hr = CreateInstance(${args}, &new_instance);
    if (FAILED(hr))
      return hr;

    hr = new_instance->QueryInterface(iid, reinterpret_cast<void**>(instance));
    if (FAILED(hr))
      delete new_instance;

    return hr;
  }

''')


def Emit(outfile, templ, num_args):
  """Emits a template function for num_args arguments"""
  mapping = {
    'templ_params':
      ", ".join(["class A%d" % d for d in xrange(1, num_args + 1)]),
    'params':
      ", ".join(["const A%d &a%d" % (d, d) for d in xrange(1, num_args + 1)]),
    'args': ", ".join(["a%d" % d for d in xrange(1, num_args + 1)]),
  }
  outfile.write(templ.substitute(mapping))


NUM_ARGUMENTS = 10


def Main(outfile):
  """Emits the contents of initializing_coclass-inl.h."""
  outfile.write(HEADER_)
  # do one to NUM_ARGUMENTS arguments
  for num_args in xrange(1, 1 + NUM_ARGUMENTS):
    Emit(outfile, TEMPLATE_, num_args)
  outfile.write(FOOTER_)

if (__name__ == "__main__"):
  outfile = open(sys.argv[1], 'w')
  Main(outfile)
