# Copyright (c) 2010 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.
#!python
'''Autogenerates mock interface implementations for MSHTML interfaces.'''
import os
import string
import sys

import com_mock

# Adjust our module path so we can import com_mock.
script_dir = os.path.abspath(os.path.dirname(__file__))
client_root = os.path.normpath(os.path.join(script_dir, '../../..'))


# The interfaces we want mocked. These have to be declared in MsHTML.h,
# and this will only work for IDispatch-derived interfaces.
# Any interface "IFoo" named here will have a corresponding IFooMockImpl
# mock implementation class in the generated output file.
_INTERFACES = [
  'IHTMLDocument2',
  'IHTMLDocument3',
  'IHTMLDOMNode',
  'IHTMLElement',
  'IHTMLElementCollection',
  'IHTMLStyleElement',
  'IHTMLStyleSheet',
  'IHTMLWindow2',
]


# Find the path to the MSHTML.h include file.
_MSHTML_PATH = None
include_dirs = os.environ['INCLUDE'].split(';')
for include_dir in include_dirs:
  candidate_path = os.path.join(include_dir, 'MsHTML.h')
  if os.path.exists(candidate_path):
    _MSHTML_PATH = candidate_path
if not _MSHTML_PATH:
  print >> sys.stderr, "Could not find MsHTML.h in any INCLUDE path."
  sys.exit(2)


# Template string for output file header.
_HEADER_TEMPLATE = '''\
// This file is autogenerated by ${file} ** DO NOT EDIT **

'''

# Template string for output file footer.
_FOOTER_TEMPLATE = ''

# Template string for mock interface definition.
_INTERFACE_TEMPLATE = '''\
class ${interface}MockImpl
    : public IDispatchImpl<${interface},
                           &IID_${interface},
                           &LIBID_MSHTML,
                           4, 0> {  // Version 4.0 of the typelib.
 public:
${mocks}
};

'''


def Main():
  mocker = com_mock.Mocker()
  mocker.AddHeaders([_MSHTML_PATH])

  header_template = string.Template(_HEADER_TEMPLATE)
  print header_template.substitute(file = __file__)

  interface_template = string.Template(_INTERFACE_TEMPLATE)
  for interface in _INTERFACES:
    mocks = '\n'.join(mocker.MockInterface(interface))
    print interface_template.substitute(interface = interface, mocks = mocks)

  footer_template = string.Template(_FOOTER_TEMPLATE)
  print footer_template.substitute(file = os.path.abspath(__file__))


if __name__ == '__main__':
  Main()
