// Copyright (c) 2010 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_SAFE_BROWSING_MALWARE_DETAILS_H_
#define CHROME_BROWSER_SAFE_BROWSING_MALWARE_DETAILS_H_
#pragma once

// A class that encapsulates the detailed malware reports sent when
// users opt-in to do so from the malware warning page.

// An instance of this class is generated when a malware warning page
// is shown (SafeBrowsingBlockingPage). It is passed on to the
// SafeBrowsing service when the warning goes away.

#include <string>
#include <vector>

#include "base/hash_tables.h"
#include "base/linked_ptr.h"
#include "base/scoped_ptr.h"
#include "chrome/browser/safe_browsing/report.pb.h"
#include "chrome/browser/safe_browsing/safe_browsing_service.h"

class TabContents;

class MalwareDetails : public base::RefCountedThreadSafe<MalwareDetails> {
 public:
  MalwareDetails(TabContents* tab_contents,
                 const SafeBrowsingService::UnsafeResource resource);

  // The SafeBrowsingService calls this from the IO thread, to get the
  // serialized report as a string and send it over.
  const std::string* GetSerializedReport();

 private:
  friend class base::RefCountedThreadSafe<MalwareDetails>;

  // Maps a URL to its Resource.
  typedef base::hash_map<
    std::string,
    linked_ptr<safe_browsing::ClientMalwareReportRequest::Resource> >
  ResourceMap;

  // Starts the collection of the report.
  void StartCollection();

  // Whether the url is "public" so we can add it to the report.
  bool IsPublicUrl(const GURL& url) const;

  // Finds an existing Resource for the given url, or creates a new
  // one if not found, and adds it to |resources_|. Returns the
  // found/created resource.
  safe_browsing::ClientMalwareReportRequest::Resource* FindOrCreateResource(
      const std::string& url);

  // Adds a Resource to resources_ with the given parent
  // relationship. |parent| can be empty.
  void AddUrl(const std::string& url, const std::string& parent);

  ~MalwareDetails();

  TabContents* tab_contents_;
  const SafeBrowsingService::UnsafeResource resource_;

  // For every Url we collect we create a Resource message. We keep
  // them in a map so we can avoid duplicates.
  ResourceMap resources_;

  // The report protocol buffer.
  scoped_ptr<safe_browsing::ClientMalwareReportRequest> report_;

  DISALLOW_COPY_AND_ASSIGN(MalwareDetails);
};

#endif  // CHROME_BROWSER_SAFE_BROWSING_MALWARE_DETAILS_H_
