// Copyright (c) 2010 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include <algorithm>
#include "chrome/browser/renderer_host/test/test_render_view_host.h"

#include "chrome/browser/browser_thread.h"
#include "chrome/browser/safe_browsing/malware_details.h"
#include "chrome/browser/safe_browsing/report.pb.h"
#include "chrome/browser/tab_contents/navigation_entry.h"
#include "chrome/browser/tab_contents/test_tab_contents.h"
#include "chrome/common/render_messages.h"
#include "chrome/common/render_messages_params.h"

static const char* kOriginalLandingURL = "http://www.originallandingpage.com/";
static const char* kLandingURL = "http://www.landingpage.com/";
static const char* kMalwareURL = "http://www.malware.com/";
static const char* kHttpsURL = "https://www.url.com/";

using safe_browsing::ClientMalwareReportRequest;

class MalwareDetailsTest : public RenderViewHostTestHarness {
 public:
  MalwareDetailsTest()
      : ui_thread_(BrowserThread::UI, MessageLoop::current()),
        io_thread_(BrowserThread::IO, MessageLoop::current()) {
  }

  virtual void SetUp() {
    RenderViewHostTestHarness::SetUp();
  }

  static bool ResourceLessThan(
      const ClientMalwareReportRequest::Resource* lhs,
      const ClientMalwareReportRequest::Resource* rhs) {
    return lhs->id() < rhs->id();
  }

 protected:
  void InitResource(SafeBrowsingService::UnsafeResource* resource,
                    ResourceType::Type resource_type,
                    const GURL& url) {
    resource->client = NULL;
    resource->url = url;
    resource->resource_type = resource_type;
    resource->threat_type = SafeBrowsingService::URL_MALWARE;
    resource->render_process_host_id = contents_->GetRenderProcessHost()->id();
    resource->render_view_id = contents_->render_view_host()->routing_id();
  }

  void VerifyResults(const ClientMalwareReportRequest& report_pb,
                     const ClientMalwareReportRequest& expected_pb) {
    EXPECT_EQ(expected_pb.malware_url(), report_pb.malware_url());
    EXPECT_EQ(expected_pb.page_url(), report_pb.page_url());
    EXPECT_EQ(expected_pb.referrer_url(), report_pb.referrer_url());

    ASSERT_EQ(expected_pb.resources_size(), report_pb.resources_size());
    // Sort the resources, to make the test deterministic
    std::vector<const ClientMalwareReportRequest::Resource*> resources;
    for (int i = 0; i < report_pb.resources_size(); ++i) {
      const ClientMalwareReportRequest::Resource& resource =
          report_pb.resources(i);
      resources.push_back(&resource);
    }
    std::sort(resources.begin(), resources.end(),
              &MalwareDetailsTest::ResourceLessThan);

    std::vector<const ClientMalwareReportRequest::Resource*> expected;
    for (int i = 0; i < report_pb.resources_size(); ++i) {
      const ClientMalwareReportRequest::Resource& resource =
          expected_pb.resources(i);
      expected.push_back(&resource);
    }
    std::sort(expected.begin(), expected.end(),
              &MalwareDetailsTest::ResourceLessThan);

    for (uint32 i = 0; i < expected.size(); ++i) {
      EXPECT_EQ(expected[i]->id(), resources[i]->id());
      EXPECT_EQ(expected[i]->url(), resources[i]->url());
      EXPECT_EQ(expected[i]->parent_id(), resources[i]->parent_id());
    }
  }

  BrowserThread ui_thread_;
  BrowserThread io_thread_;
};

// Tests creating a simple malware report.
TEST_F(MalwareDetailsTest, MalwareSubResource) {
  // Start a load.
  controller().LoadURL(GURL(kLandingURL), GURL(), PageTransition::TYPED);

  SafeBrowsingService::UnsafeResource resource;
  InitResource(&resource, ResourceType::SUB_RESOURCE, GURL(kMalwareURL));

  scoped_refptr<MalwareDetails> report = new MalwareDetails(
      contents(), resource);

  scoped_ptr<const std::string> serialized(report->GetSerializedReport());
  ClientMalwareReportRequest actual;
  actual.ParseFromString(*serialized);

  ClientMalwareReportRequest expected;
  expected.set_malware_url(kMalwareURL);
  expected.set_page_url(kLandingURL);
  expected.set_referrer_url("");

  ClientMalwareReportRequest::Resource* pb_resource = expected.add_resources();
  pb_resource->set_id(0);
  pb_resource->set_url(kLandingURL);
  pb_resource = expected.add_resources();
  pb_resource->set_id(1);
  pb_resource->set_url(kMalwareURL);

  VerifyResults(actual, expected);
}

// Tests creating a simple malware report where the subresource has a
// different original_url.
TEST_F(MalwareDetailsTest, MalwareSubResourceWithOriginalUrl) {
  controller().LoadURL(GURL(kLandingURL), GURL(), PageTransition::TYPED);

  SafeBrowsingService::UnsafeResource resource;
  InitResource(&resource, ResourceType::SUB_RESOURCE, GURL(kMalwareURL));
  resource.original_url = GURL(kOriginalLandingURL);

  scoped_refptr<MalwareDetails> report = new MalwareDetails(
      contents(), resource);

  scoped_ptr<const std::string> serialized(report->GetSerializedReport());
  ClientMalwareReportRequest actual;
  actual.ParseFromString(*serialized);

  ClientMalwareReportRequest expected;
  expected.set_malware_url(kMalwareURL);
  expected.set_page_url(kLandingURL);
  expected.set_referrer_url("");

  ClientMalwareReportRequest::Resource* pb_resource = expected.add_resources();
  pb_resource->set_id(0);
  pb_resource->set_url(kLandingURL);

  pb_resource = expected.add_resources();
  pb_resource->set_id(1);
  pb_resource->set_url(kOriginalLandingURL);

  pb_resource = expected.add_resources();
  pb_resource->set_id(2);
  pb_resource->set_url(kMalwareURL);
  // The Resource for kMmalwareUrl should have the Resource for
  // kOriginalLandingURL (with id 1) as parent.
  pb_resource->set_parent_id(1);

  VerifyResults(actual, expected);
}

// Verify that https:// urls are dropped.
TEST_F(MalwareDetailsTest, NotPublicUrl) {
  controller().LoadURL(GURL(kHttpsURL), GURL(), PageTransition::TYPED);
  SafeBrowsingService::UnsafeResource resource;
  InitResource(&resource, ResourceType::SUB_RESOURCE, GURL(kMalwareURL));
  scoped_refptr<MalwareDetails> report = new MalwareDetails(
      contents(), resource);

  scoped_ptr<const std::string> serialized(report->GetSerializedReport());
  ClientMalwareReportRequest actual;
  actual.ParseFromString(*serialized);

  ClientMalwareReportRequest expected;
  expected.set_malware_url(kMalwareURL);  // No page_url
  expected.set_referrer_url("");

  ClientMalwareReportRequest::Resource* pb_resource = expected.add_resources();
  pb_resource->set_url(kMalwareURL);  // Only one resource

  VerifyResults(actual, expected);
}
