#!/usr/bin/python
# Copyright (c) 2010 The Chromium OS Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""This program generates a C++ header file containing the mapping from
input method ID to keyboard overlay ID from
chromeos-assets/input_methods/whitelist.txt

It will produce output that looks like:

// This file is automatically generated by gen_keyboard_overlay_map.py
#ifndef CHROMEOS_KEYBOARD_OVERLAY_MAP_H_
#define CHROMEOS_KEYBOARD_OVERLAY_MAP_H_

namespace {
const struct KeyboardOverlayMap {
  std::string input_method_id;
  std::string keyboard_overlay_id;
} kKeyboardOverlayMap[] = {
  { "xkb:nl::nld", "nl" },
  { "xkb:be::nld", "nl" },
  { "xkb:fr::fra", "fr" },
  { "xkb:be::fra", "fr" },
  ...
};
}  // namespace

#endif  // CHROMEOS_KEYBOARD_OVERLAY_MAP_H_
"""

import fileinput
import re
import sys


OUTPUT_HEADER = """
// This file is automatically generated by gen_keyboard_overlay_map.py
#ifndef CHROMEOS_KEYBOARD_OVERLAY_MAP_H_
#define CHROMEOS_KEYBOARD_OVERLAY_MAP_H_

namespace {
const struct KeyboardOverlayMap {
  std::string input_method_id;
  std::string keyboard_overlay_id;
} kKeyboardOverlayMap[] = {
"""

OUTPUT_FOOTER = """
};
}  // namespace

#endif  // CHROMEOS_KEYBOARD_OVERLAY_MAP_H_
"""

def main():
  if len(sys.argv) != 2:
    print >> sys.stderr, 'Usage: gen_input_method_whitelist.py [file]'
    sys.exit(1)
  print OUTPUT_HEADER
  for line in fileinput.input(sys.argv[1]):
    values = re.sub(r'#.*', '', line)    # Remove comments.
    values = values.split()              # Split the values using a space.
    if len(values) == 2:
      print '  { "%s", "%s" }, ' % (values[0], values[1])
    elif len(values) == 0:
      pass                               # Skip a comment line.
    else:
      print >> sys.stderr, 'Incorrect format: ' + line
      sys.exit(2)
  print OUTPUT_FOOTER


if __name__ == '__main__':
  main()
