#!python
# Copyright 2010 Google Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import optparse
import os
import pywintypes
import win32com.client

LICENSE_HEADER = """\
#!python
# Copyright 2010 Google Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License."""

DO_NOT_EDIT_HEADER = """\
# DO NOT EDIT. This is an ETW event descriptor file generated by
# sawbuck/py/generate_descriptor.py. It contains event descriptions for
# MOF GUID %s."""

# Generate symbols for the WbemScripting module so that we can have symbols
# for debugging and use constants throughout the file.
win32com.client.gencache.EnsureModule('{565783C6-CB41-11D1-8B02-00600806D9B6}',
                                      0, 1, 1)

class DescriptorGenerator(object):
  def __init__(self):
    self._locator = win32com.client.Dispatch('WbemScripting.SWbemLocator')
    self._services = self._locator.ConnectServer('localhost', r'root\wmi')

  def _GetCategories(self, guid):
    """Returns the event classes (categories) that match the given GUID."""
    categories = self._services.SubclassesOf('EventTrace',
        win32com.client.constants.wbemFlagUseAmendedQualifiers)

    ret = []
    for category in categories:
      try:
        if category.Qualifiers_.Item('Guid').Value == guid:
          ret.append(category)
      except pywintypes.com_error:
        # Ignore categories that don't have a GUID.
        pass

    return ret

  def _GetEvents(self, category):
    """Returns the event subclasses of the given category class."""
    return self._services.SubclassesOf(category.Path_.Class,
        win32com.client.constants.wbemFlagUseAmendedQualifiers)

  def _GetQualifier(self, obj, name):
    """Helper function to get a qualifer that may not exist."""
    ret = None
    try:
      ret = obj.Qualifiers_.Item(name).Value
    except pywintypes.com_error:
      pass
    return ret

  def Generate(self, guid, output_dir):
    """Generates a descriptor file for guid and stores it in output_dir."""
    guid = guid.lower()

    lines = []
    lines.append(LICENSE_HEADER)
    lines.append('#')
    lines.append(DO_NOT_EDIT_HEADER % guid)
    lines.append('')
    lines.append('from etw.descriptors import event, field')

    categories = self._GetCategories(guid)
    lines.append('')
    lines.append('')
    lines.append(self._GenerateEventTypeClass(guid, categories))

    for category in categories:
      lines.append('')
      lines.append('')
      lines.append(self._GenerateCategoryClass(category))
    lines.append('')

    # All categories of the same GUID have the same display name.
    name = categories[0].Qualifiers_.Item('DisplayName').Value.lower()
    f = open(os.path.join(output_dir, '%s.py' % name), 'wb')
    f.write('\n'.join(lines))
    f.close()

  def _GenerateEventTypeClass(self, guid, categories):
    """Generates the event type class definition for the given categories."""
    lines = []
    lines.append('class Event(object):')
    lines.append('  GUID = \'%s\'' % guid)

    eventTypes = dict()
    for category in categories:
      for event in self._GetEvents(category):
        eventTypes.update(self._GetEventTypes(event))

    keys = eventTypes.keys()
    keys.sort()
    for key in keys:
      lines.append('  %s = (GUID, %d)' % (eventTypes[key], key))

    return '\n'.join(lines)

  def _GetEventTypes(self, event):
    """Gets a dict of event type to name for the given event."""
    eventTypes = event.Qualifiers_.Item('EventType').Value
    eventTypeNames = event.Qualifiers_.Item('EventTypeName').Value

    # The above qualifiers can be int or list, so we convert ints to list.
    if type(eventTypes) is int:
      eventTypes = [eventTypes]
      eventTypeNames = [eventTypeNames]

    return dict(zip(eventTypes, eventTypeNames))

  def _GenerateCategoryClass(self, category):
    """Generates the category class definition for the given category."""
    lines = []
    lines.append('class %s(event.EventCategory):' % category.Path_.Class)
    lines.append('  GUID = Event.GUID')
    try:
      lines.append('  VERSION = %d' %
                   category.Qualifiers_.Item('EventVersion').Value)
    except pywintypes.com_error:
      # Assume the version is zero if none is specified.
      lines.append('  VERSION = 0')

    for event in self._GetEvents(category):
      lines.append('')
      lines.append(self._GenerateEventClass(event))

    return '\n'.join(lines)

  def _GenerateEventClass(self, event):
    """Generates the event class definition for the given event."""
    lines = []
    lines.append('  class %s(event.EventClass):' % event.Path_.Class)
    lines.append('    _event_types_ = [%s]' % self._GenerateEventTypes(event))
    lines.append('    _fields_ = [%s]' % self._GenerateFields(event))

    return '\n'.join(lines)

  def _GenerateEventTypes(self, event):
    """Generates the event types as a comma delimited string for event."""
    eventTypes = self._GetEventTypes(event).values()
    eventTypes.sort()
    eventTypes = ['Event.%s' % str(eventType) for eventType in eventTypes]
    return ',\n                     '.join(eventTypes)

  def _GenerateFields(self, event):
    """Generates the fields as comma delimited tuples for event."""
    lines = []
    fields = self._GetFields(event)
    keys = fields.keys()
    keys.sort()
    for key in keys:
      field = fields[key]
      lines.append('(\'%s\', field.%s)' %
                   (field.Name, self._GetFieldType(field)))

    return ',\n                '.join(lines)

  def _GetFields(self, event):
    """Gets the fields of the given event.

    Gets the fields of the given event and returns them in a dictionary using
    the fields' WmiDataId qualifier (which is the index of the filed in
    the MOF data structure) as a key so that they are sorted properly.
    """
    fields = dict()
    for prop in event.Properties_:
      id = self._GetQualifier(prop, 'WmiDataId')
      if id:
        fields[id] = prop
    return fields

  def _GetFieldType(self, field):
    """Gets the buffer reader function to call for the given property."""
    const = win32com.client.constants
    if field.CIMType == const.wbemCimtypeBoolean:
      return 'Boolean'
    elif field.CIMType == const.wbemCimtypeSint8:
      return 'Int8'
    elif field.CIMType == const.wbemCimtypeUint8:
      return 'UInt8'
    elif field.CIMType == const.wbemCimtypeSint16:
      return 'Int16'
    elif field.CIMType == const.wbemCimtypeUint16:
      return 'UInt16'
    elif field.CIMType == const.wbemCimtypeSint32:
      return 'Int32'
    elif field.CIMType == const.wbemCimtypeUint32:
      pointer = self._GetQualifier(field, 'pointer')
      if pointer:
        return 'Pointer'
      else:
        return 'UInt32'
    elif field.CIMType == const.wbemCimtypeSint64:
      return 'Int64'
    elif field.CIMType == const.wbemCimtypeUint64:
      return 'UInt64'
    elif field.CIMType == const.wbemCimtypeString:
      format = self._GetQualifier(field, 'format')
      if format == 'w':
        return 'WString'
      else:
        return 'String'
    elif field.CIMType == const.wbemCimtypeObject:
      ext = field.Qualifiers_.Item('extension').Value
      if ext == 'Sid':
        return 'Sid'
      elif ext == 'SizeT':
        return 'Int32'
      elif ext == 'WmiTime':
        return 'WmiTime'
      raise ValueError('Field %s is of unhandled object type: %s' %
                       (field.Name, ext))

    raise ValueError('Field %s is of unhandled type: %s' %
                     (field.Name, field.CIMType))


def main():
  parser = optparse.OptionParser()
  parser.add_option('-g', '--guid', dest='guid',
                    help='GUID of the event category to generate a paser for.')
  parser.add_option('-o', '--output_dir', dest='output_dir', default='.',
                    help='Output directory for the generated file.')
  (options, args) = parser.parse_args()

  if options.guid:
    generator = DescriptorGenerator()
    generator.Generate(options.guid, options.output_dir)
  else:
    print 'No guid specified.'

if __name__ == '__main__':
  main()
