// Copyright 2010 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Declaration of a filter class that wraps the information stored in a list
// filter.

#ifndef SAWBUCK_VIEWER_FILTER_H_
#define SAWBUCK_VIEWER_FILTER_H_

#include <string>
#include <vector>
#include "sawbuck/viewer/log_list_view.h"
#include "pcrecpp.h"  // NOLINT

// Represents a single filter entered by the user. Also provides a matching
// function to see if the filter matches a certain element in an ILogView as
// well as serialization / deserialization.
//
// Does not yet perform validation on column data types. For example,
// if a PID if provided as a filter value, we don't check to see if the value is
// numeric.
class Filter {
 public:
  // Make this synonymous to the logview formatters enum.
  enum Column {
    SEVERITY = LogViewFormatter::SEVERITY,
    PROCESS_ID = LogViewFormatter::PROCESS_ID,
    THREAD_ID = LogViewFormatter::THREAD_ID,
    TIME = LogViewFormatter::TIME,
    FILE = LogViewFormatter::FILE,
    LINE = LogViewFormatter::LINE,
    MESSAGE = LogViewFormatter::MESSAGE,
    NUM_COLUMNS
  };

  enum Relation {
    IS,
    CONTAINS,
    NUM_RELATIONS
  };

  enum Action {
    INCLUDE,
    EXCLUDE,
    NUM_ACTIONS
  };

  Filter(Column column,
         Relation relation,
         Action action,
         const wchar_t* value);
  explicit Filter(const std::wstring& serialized);

  // Returns true if the Filter is correctly constructed, false otherwise.
  bool IsValid() { return is_valid_; }

  // Returns true if this filter matches the log entry in log_view on row_index.
  bool Matches(ILogView* log_view, int row_index) const;

  // Returns a string representation of this filter. This representation
  // can be used in the constructor that takes a serialized representation.
  std::wstring Serialize() const;

  // Sets this filter using the serialized representation generated by
  // Serialize(). Returns true if successful, false otherwise.
  bool Deserialize(const std::wstring& serialized);

  Column column() const { return column_; }
  Relation relation() const { return relation_; }
  Action action() const { return action_; }
  std::wstring value() const;

  // Returns a wstring representation of a list of Filters. This wstring can
  // be passed to DeserializeFilters below.
  static std::wstring SerializeFilters(const std::vector<Filter>& filters);

  // Returns a list of Filters from a serialized representation generated by
  // SerializeFilters(). Returns an empty list on failure.
  static std::vector<Filter> DeserializeFilters(const std::wstring& stored);


  bool operator==(const Filter& other) const;

 private:

  bool ValueMatchesInt(int check_value) const;
  bool ValueMatchesString(const std::string& check_string) const;

  // Sets up match_re_ if needed.
  void BuildRegExp();

  // As an optimization, we compile a matching reg exp at construction.
  pcrecpp::RE match_re_;

  Column column_;
  Relation relation_;
  Action action_;

  // We store a UTF8 string internally, but use wide strings on the class
  // interface.
  std::string value_;

  bool is_valid_;
};


#endif  // SAWBUCK_VIEWER_FILTER_H_
