// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ui/base/keycodes/keyboard_code_conversion.h"

#include <string.h>

#include "ui/base/keycodes/keyboard_codes.h"

namespace ui {

namespace {

typedef struct KeyIdentifier {
  const char* key_name;
  const char character;
  const char* unicode_codepoint;
  const KeyboardCode key_code;
} KeyIdentifier;

// Taken from Section 6.3.3 here:
// http://www.w3.org/TR/DOM-Level-3-Events/#keyset-keyidentifiers
// TODO(bryeung): keycodes could be wrong: I took the keydown code only
static const KeyIdentifier kKeyIdentifiers[] = {
  { "Accept", '\0', "", VKEY_ACCEPT },
  { "Add", '\0', "", VKEY_ADD },
  { "Again", '\0', "", VKEY_UNKNOWN },
  { "AllCandidates", '\0', "", VKEY_UNKNOWN },
  { "Alphanumeric", '\0', "", VKEY_UNKNOWN },
  { "Alt", '\0', "", VKEY_MENU },
  { "AltGraph", '\0', "", VKEY_UNKNOWN },
  { "Apps", '\0', "", VKEY_APPS },
  { "Attn", '\0', "", VKEY_ATTN },
  { "BrowserBack", '\0', "", VKEY_BROWSER_BACK },
  { "BrowserFavorites", '\0', "", VKEY_BROWSER_FAVORITES },
  { "BrowserForward", '\0', "", VKEY_BROWSER_FAVORITES },
  { "BrowserHome", '\0', "", VKEY_BROWSER_HOME },
  { "BrowserRefresh", '\0', "", VKEY_BROWSER_REFRESH },
  { "BrowserSearch", '\0', "", VKEY_BROWSER_SEARCH },
  { "BrowserStop", '\0', "", VKEY_BROWSER_STOP },
  { "CapsLock", '\0', "", VKEY_CAPITAL },
  { "Clear", '\0', "", VKEY_OEM_CLEAR },
  { "CodeInput", '\0', "", VKEY_UNKNOWN },
  { "Compose", '\0', "", VKEY_UNKNOWN },
  { "Control", '\0', "", VKEY_CONTROL },
  { "Crsel", '\0', "", VKEY_CRSEL },
  { "Convert", '\0', "", VKEY_CONVERT },
  { "Copy", '\0', "", VKEY_UNKNOWN },
  { "Cut", '\0', "", VKEY_UNKNOWN },
  { "Decimal", '\0', "", VKEY_DECIMAL },
  { "Divide", '\0', "", VKEY_DIVIDE },
  { "Down", '\0', "", VKEY_DOWN },
  { "DownLeft", '\0', "", VKEY_UNKNOWN },
  { "DownRight", '\0', "", VKEY_UNKNOWN },
  { "End", '\0', "", VKEY_END },
  { "Enter", '\0', "", VKEY_RETURN },
  { "EraseEof", '\0', "", VKEY_EREOF },
  { "Execute", '\0', "", VKEY_EXECUTE },
  { "Exsel", '\0', "", VKEY_EXSEL },
  { "Fn", '\0', "", VKEY_UNKNOWN },
  { "F1", '\0', "", VKEY_F1 },
  { "F2", '\0', "", VKEY_F2 },
  { "F3", '\0', "", VKEY_F3 },
  { "F4", '\0', "", VKEY_F4 },
  { "F5", '\0', "", VKEY_F5 },
  { "F6", '\0', "", VKEY_F6 },
  { "F7", '\0', "", VKEY_F7 },
  { "F8", '\0', "", VKEY_F8 },
  { "F9", '\0', "", VKEY_F9 },
  { "F10", '\0', "", VKEY_F10 },
  { "F11", '\0', "", VKEY_F11 },
  { "F12", '\0', "", VKEY_F12 },
  { "F13", '\0', "", VKEY_F13 },
  { "F14", '\0', "", VKEY_F14 },
  { "F15", '\0', "", VKEY_F15 },
  { "F16", '\0', "", VKEY_F16 },
  { "F17", '\0', "", VKEY_F17 },
  { "F18", '\0', "", VKEY_F18 },
  { "F19", '\0', "", VKEY_F19 },
  { "F20", '\0', "", VKEY_F20 },
  { "F21", '\0', "", VKEY_F21 },
  { "F22", '\0', "", VKEY_F22 },
  { "F23", '\0', "", VKEY_F23 },
  { "F24", '\0', "", VKEY_F24 },
  { "FinalMode", '\0', "", VKEY_FINAL },
  { "Find", '\0', "", VKEY_UNKNOWN },
  { "FullWidth", '\0', "", VKEY_UNKNOWN },
  { "HalfWidth", '\0', "", VKEY_UNKNOWN },
  { "HangulMode", '\0', "", VKEY_HANGUL },
  { "HanjaMode", '\0', "", VKEY_HANJA },
  { "Help", '\0', "", VKEY_HELP },
  { "Hiragana", '\0', "", VKEY_UNKNOWN },
  { "Home", '\0', "", VKEY_HOME },
  { "Insert", '\0', "", VKEY_INSERT },
  { "JapaneseHiragana", '\0', "", VKEY_UNKNOWN },
  { "JapaneseKatakana", '\0', "", VKEY_UNKNOWN },
  { "JapaneseRomaji", '\0', "", VKEY_UNKNOWN },
  { "JunjaMode", '\0', "", VKEY_JUNJA },
  { "KanaMode", '\0', "", VKEY_KANA },
  { "KanjiMode", '\0', "", VKEY_KANJI },
  { "Katakana", '\0', "", VKEY_UNKNOWN },
  { "LaunchApplication1", '\0', "", VKEY_MEDIA_LAUNCH_APP1 },
  { "LaunchApplication2", '\0', "", VKEY_MEDIA_LAUNCH_APP2 },
  { "LaunchMail", '\0', "", VKEY_MEDIA_LAUNCH_MAIL },
  { "Left", '\0', "", VKEY_LEFT },
  { "Menu", '\0', "", VKEY_MENU },
  { "Meta", '\0', "", VKEY_UNKNOWN },
  { "MediaNextTrack", '\0', "", VKEY_MEDIA_NEXT_TRACK },
  { "MediaPlayPause", '\0', "", VKEY_MEDIA_PLAY_PAUSE },
  { "MediaPreviousTrack", '\0', "", VKEY_MEDIA_PREV_TRACK },
  { "MediaStop", '\0', "", VKEY_MEDIA_STOP },
  { "ModeChange", '\0', "", VKEY_MODECHANGE },
  { "Multiply", '\0', "", VKEY_MULTIPLY },
  { "NextCandidate", '\0', "", VKEY_UNKNOWN },
  { "Nonconvert", '\0', "", VKEY_NONCONVERT },
  { "NumLock", '\0', "", VKEY_NUMLOCK },
  { "PageDown", '\0', "", VKEY_NEXT },
  { "PageUp", '\0', "", VKEY_PRIOR },
  { "Paste", '\0', "", VKEY_UNKNOWN },
  { "Pause", '\0', "", VKEY_PAUSE },
  { "Play", '\0', "", VKEY_PLAY },
  { "PreviousCandidate", '\0', "", VKEY_UNKNOWN },
  { "PrintScreen", '\0', "", VKEY_SNAPSHOT },
  { "Process", '\0', "", VKEY_PROCESSKEY },
  { "Props", '\0', "", VKEY_UNKNOWN },
  { "Right", '\0', "", VKEY_RIGHT },
  { "RomanCharacters", '\0', "", VKEY_UNKNOWN },
  { "Scroll", '\0', "", VKEY_SCROLL },
  { "Select", '\0', "", VKEY_SELECT },
  { "SelectMedia", '\0', "", VKEY_MEDIA_LAUNCH_MEDIA_SELECT },
  { "Separator", '\0', "", VKEY_SEPARATOR },
  { "Shift", '\0', "", VKEY_SHIFT },
  { "Soft1", '\0', "", VKEY_UNKNOWN },
  { "Soft2", '\0', "", VKEY_UNKNOWN },
  { "Soft3", '\0', "", VKEY_UNKNOWN },
  { "Soft4", '\0', "", VKEY_UNKNOWN },
  { "Stop", '\0', "", VKEY_UNKNOWN },
  { "Subtract", '\0', "", VKEY_SUBTRACT },
  { "Up", '\0', "", VKEY_UP },
  { "UpLeft", '\0', "", VKEY_UNKNOWN },
  { "UpRight", '\0', "", VKEY_UNKNOWN },
  { "Undo", '\0', "", VKEY_UNKNOWN },
  { "VolumeDown", '\0', "", VKEY_VOLUME_DOWN },
  { "VolumeMute", '\0', "", VKEY_VOLUME_MUTE },
  { "VolumeUp", '\0', "", VKEY_VOLUME_UP },
  { "Win", '\0', "", VKEY_LWIN },
  { "Zoom", '\0', "", VKEY_ZOOM },
  { "Backspace", '\0', "U+0008", VKEY_BACK },
  { "Tab", '\0', "U+0009", VKEY_TAB },
  { "Cancel", '\0', "U+0018", VKEY_UNKNOWN },
  { "Esc", '\0', "U+001B", VKEY_ESCAPE },
  { "Spacebar", ' ', "U+0020", VKEY_SPACE },
  { "Exclamation", '!', "U+0021", VKEY_1 },
  { "DoubleQuote", '\"', "U+0022", VKEY_OEM_7 },
  { "Hash", '\0', "U+0023", VKEY_3 },
  { "Dollar", '$', "U+0024", VKEY_4 },
  { "Ampersand", '&', "U+0026", VKEY_5 },
  { "Apostrophe", '\'', "U+0027", VKEY_OEM_7 },
  { "LeftParen", '(', "U+0028", VKEY_9 },
  { "RightParen", ')', "U+0029", VKEY_0 },
  { "Asterisk", '*', "U+002A", VKEY_8 },
  { "Plus", '+', "U+002B", VKEY_OEM_PLUS },
  { "Percent", '+', "U+0025", VKEY_5 },
  { "Comma", ',', "U+002C", VKEY_OEM_COMMA },
  { "HyphenMinus", '-', "U+002D", VKEY_OEM_MINUS },
  { "Period", '.', "U+002E", VKEY_OEM_PERIOD },
  { "Solidus", '/', "U+002F", VKEY_OEM_2 },
  { "", '0', "U+0030", VKEY_0 },
  { "", '1', "U+0031", VKEY_1 },
  { "", '2', "U+0032", VKEY_2 },
  { "", '3', "U+0033", VKEY_3 },
  { "", '4', "U+0034", VKEY_4 },
  { "", '5', "U+0035", VKEY_5 },
  { "", '6', "U+0036", VKEY_6 },
  { "", '7', "U+0037", VKEY_7 },
  { "", '8', "U+0038", VKEY_8 },
  { "", '9', "U+0039", VKEY_9 },
  { "Colon", ':', "U+003A", VKEY_OEM_1 },
  { "Semicolon", ';', "U+003B", VKEY_OEM_1 },
  { "LessThan", '<', "U+003C", VKEY_OEM_COMMA },
  { "Equals", '=', "U+003D", VKEY_OEM_PLUS },
  { "GreaterThan", '>', "U+003E", VKEY_OEM_PERIOD },
  { "QuestionMark", '?', "U+003F", VKEY_OEM_2 },
  { "At", '@', "U+0040", VKEY_2 },
  { "", 'A', "U+0041", VKEY_A },
  { "", 'B', "U+0042", VKEY_B },
  { "", 'C', "U+0043", VKEY_C },
  { "", 'D', "U+0044", VKEY_D },
  { "", 'E', "U+0045", VKEY_E },
  { "", 'F', "U+0046", VKEY_F },
  { "", 'G', "U+0047", VKEY_G },
  { "", 'H', "U+0048", VKEY_H },
  { "", 'I', "U+0049", VKEY_I },
  { "", 'J', "U+004A", VKEY_J },
  { "", 'K', "U+004B", VKEY_K },
  { "", 'L', "U+004C", VKEY_L },
  { "", 'M', "U+004D", VKEY_M },
  { "", 'N', "U+004E", VKEY_N },
  { "", 'O', "U+004F", VKEY_O },
  { "", 'P', "U+0050", VKEY_P },
  { "", 'Q', "U+0051", VKEY_Q },
  { "", 'R', "U+0052", VKEY_R },
  { "", 'S', "U+0053", VKEY_S },
  { "", 'T', "U+0054", VKEY_T },
  { "", 'U', "U+0055", VKEY_U },
  { "", 'V', "U+0056", VKEY_V },
  { "", 'W', "U+0057", VKEY_W },
  { "", 'X', "U+0058", VKEY_X },
  { "", 'Y', "U+0059", VKEY_Y },
  { "", 'Z', "U+005A", VKEY_Z },
  { "", 'a', "U+0061", VKEY_A },
  { "", 'b', "U+0062", VKEY_B },
  { "", 'c', "U+0063", VKEY_C },
  { "", 'd', "U+0064", VKEY_D },
  { "", 'e', "U+0065", VKEY_E },
  { "", 'f', "U+0066", VKEY_F },
  { "", 'g', "U+0067", VKEY_G },
  { "", 'h', "U+0068", VKEY_H },
  { "", 'i', "U+0069", VKEY_I },
  { "", 'j', "U+006A", VKEY_J },
  { "", 'k', "U+006B", VKEY_K },
  { "", 'l', "U+006C", VKEY_L },
  { "", 'm', "U+006D", VKEY_M },
  { "", 'n', "U+006E", VKEY_N },
  { "", 'o', "U+006F", VKEY_O },
  { "", 'p', "U+0070", VKEY_P },
  { "", 'q', "U+0071", VKEY_Q },
  { "", 'r', "U+0072", VKEY_R },
  { "", 's', "U+0073", VKEY_S },
  { "", 't', "U+0074", VKEY_T },
  { "", 'u', "U+0075", VKEY_U },
  { "", 'v', "U+0076", VKEY_V },
  { "", 'w', "U+0077", VKEY_W },
  { "", 'x', "U+0078", VKEY_X },
  { "", 'y', "U+0079", VKEY_Y },
  { "", 'z', "U+007A", VKEY_Z },
  { "LeftSquareBracket", '[', "U+005B", VKEY_OEM_4 },
  { "Backslash", '\\', "U+005C", VKEY_OEM_5 },
  { "RightSquareBracket", ']', "U+005D", VKEY_OEM_6 },
  { "Circumflex", '^', "U+005E", VKEY_OEM_6 },
  { "Underscore", '_', "U+005F", VKEY_OEM_MINUS },
  { "Grave", '`', "U+0060", VKEY_OEM_3 },
  { "LeftCurlyBracket", '{', "U+007B", VKEY_OEM_4 },
  { "Pipe", '|', "U+007C", VKEY_OEM_5 },
  { "RightCurlyBracket", '}', "U+007D", VKEY_OEM_6 },
  { "Del", '\0', "U+007F", VKEY_DELETE },
  { "InvertedExclamation", '\0', "U+00A1", VKEY_UNKNOWN },
  { "DeadGrave", '\0', "U+0300", VKEY_UNKNOWN },
  { "DeadEacute", '\0', "U+0301", VKEY_UNKNOWN },
  { "DeadCircumflex", '\0', "U+0302", VKEY_UNKNOWN },
  { "DeadTilde", '\0', "U+0303", VKEY_UNKNOWN },
  { "DeadMacron", '\0', "U+0304", VKEY_UNKNOWN },
  { "DeadBreve", '\0', "U+0306", VKEY_UNKNOWN },
  { "DeadAboveDot", '\0', "U+0307", VKEY_UNKNOWN },
  { "DeadUmlaut", '\0', "U+0308", VKEY_UNKNOWN },
  { "DeadAboveRing", '\0', "U+030A", VKEY_UNKNOWN },
  { "DeadDoubleacute", '\0', "U+030B", VKEY_UNKNOWN },
  { "DeadCaron", '\0', "U+030C", VKEY_UNKNOWN },
  { "DeadCedilla", '\0', "U+0327", VKEY_UNKNOWN },
  { "DeadOgonek", '\0', "U+0328", VKEY_UNKNOWN },
  { "DeadIota", '\0', "U+0345", VKEY_UNKNOWN },
  { "Euro", '\0', "U+20AC", VKEY_UNKNOWN },
  { "DeadVoicedSound", '\0', "U+3099", VKEY_UNKNOWN },
  { "DeadSemivoicedSound", '\0', "U+309A", VKEY_UNKNOWN }
};

static const int kNumKeyIdentifiers =
    sizeof(kKeyIdentifiers)/sizeof(KeyIdentifier);

static const KeyIdentifier* GetKeyIdentifierData(const std::string& id) {
  for (int i = 0; i < kNumKeyIdentifiers; ++i) {
    const KeyIdentifier& key = kKeyIdentifiers[i];
    if ((key.key_name[0] != '\0' && strcmp(id.c_str(), key.key_name) == 0) ||
        (id.length() == 1 && id[0] == key.character) ||
        (key.unicode_codepoint[0] != '\0' &&
           strcmp(id.c_str(), key.unicode_codepoint) == 0)) {
      return &key;
    }
  }

  return NULL;
}

}  // namespace


KeyboardCode KeyCodeFromKeyIdentifier(const std::string& key_identifier) {
  const KeyIdentifier* id = GetKeyIdentifierData(key_identifier);
  if (!id) {
    return VKEY_UNKNOWN;
  }
  return id->key_code;
}

}  // namespace ui
