/* Copyright (c) 2002-2010 Dovecot Sieve authors, see the included COPYING file
 */

#include "common.h"
#include "str.h"
#include "ioloop.h"
#include "network.h"
#include "istream.h"
#include "ostream.h"
#include "var-expand.h"
#include "commands.h"

#include "managesieve-quote.h"
#include "sieve-storage.h"

#include "client.h"

#include <stdlib.h>
#include <unistd.h>

static struct client *my_client; /* we don't need more than one currently */

static void client_idle_timeout(struct client *client)
{
	client_send_bye(client, "Disconnected for inactivity.");
	client_destroy(client, "Disconnected for inactivity");
}

struct client *client_create
(int fd_in, int fd_out, struct sieve_instance *svinst, 
	struct sieve_storage *storage)
{
	struct client *client;

	/* always use nonblocking I/O */
	net_set_nonblock(fd_in, TRUE);
	net_set_nonblock(fd_out, TRUE);

	client = i_new(struct client, 1);
	client->fd_in = fd_in;
	client->fd_out = fd_out;
	client->input = i_stream_create_fd(fd_in, managesieve_max_line_length, FALSE);
	client->output = o_stream_create_fd(fd_out, (size_t)-1, FALSE);

	o_stream_set_flush_callback(client->output, client_output, client);

	client->io = io_add(fd_in, IO_READ, client_input, client);
	client->parser = managesieve_parser_create(client->input, client->output,
					    managesieve_max_line_length);
	client->to_idle = timeout_add(CLIENT_IDLE_TIMEOUT_MSECS,
				      client_idle_timeout, client);

	client->cmd.pool = pool_alloconly_create("command pool", 8192);
	client->cmd.client = client;

	client->svinst = svinst;
	client->storage = storage;

	i_assert(my_client == NULL);
	my_client = client;

	if (hook_client_created != NULL)
		hook_client_created(&client);
	return client;
}

static const char *client_stats(struct client *client)
{	
	static struct var_expand_table static_tab[] = {
		{ 'i', NULL, "input" },
		{ 'o', NULL, "output" },
		{ '\0', NULL, NULL }
	};
	struct var_expand_table *tab;
	string_t *str;

	tab = t_malloc(sizeof(static_tab));
	memcpy(tab, static_tab, sizeof(static_tab));

	tab[0].value = dec2str(client->input->v_offset);
	tab[1].value = dec2str(client->output->offset);

	str = t_str_new(128);
	var_expand(str, logout_format, tab);
	return str_c(str);
}

static const char *client_get_disconnect_reason(struct client *client)
{
	errno = client->input->stream_errno != 0 ?
		client->input->stream_errno :
		client->output->stream_errno;
	return errno == 0 || errno == EPIPE ? "Connection closed" :
		t_strdup_printf("Connection closed: %m");
}

void client_destroy(struct client *client, const char *reason)
{
	int ret;

 	i_assert(!client->handling_input);
	i_assert(!client->destroyed);
	client->destroyed = TRUE;

	if (!client->disconnected) {
		client->disconnected = TRUE;
		if (reason == NULL)
			reason = client_get_disconnect_reason(client);
		i_info("%s %s", reason, client_stats(client));	
	}

	if (client->command_pending) {
		/* try to deinitialize the command */
		i_assert(client->cmd.func != NULL);
		i_stream_close(client->input);
		o_stream_close(client->output);
		client->input_pending = FALSE;

		ret = client->cmd.func(&client->cmd);
		i_assert(ret);
	}

	managesieve_parser_destroy(&client->parser);
	if (client->io != NULL)
		io_remove(&client->io);
	if (client->to_idle_output != NULL)
		timeout_remove(&client->to_idle_output);
	timeout_remove(&client->to_idle);

	i_stream_destroy(&client->input);
	o_stream_destroy(&client->output);

	if (close(client->fd_in) < 0)
		i_error("close(client in) failed: %m");
	if (client->fd_in != client->fd_out) {
		if (close(client->fd_out) < 0)
			i_error("close(client out) failed: %m");
	}

	pool_unref(&client->cmd.pool);
	i_free(client);

	/* quit the program */
	my_client = NULL;
	io_loop_stop(ioloop);
}

void client_disconnect(struct client *client, const char *reason)
{
	i_assert(reason != NULL);

	if (client->disconnected)
		return;

	i_info("Disconnected: %s %s", reason, client_stats(client));
	client->disconnected = TRUE;
	(void)o_stream_flush(client->output);

	i_stream_close(client->input);
	o_stream_close(client->output);
}

void client_disconnect_with_error(struct client *client, const char *msg)
{
	client_send_bye(client, msg);
	client_disconnect(client, msg);
}

int client_send_line(struct client *client, const char *data) 
{
	struct const_iovec iov[2];

	if (client->output->closed)
		return -1;

	iov[0].iov_base = data;
	iov[0].iov_len = strlen(data);
	iov[1].iov_base = "\r\n";
	iov[1].iov_len = 2;

	if (o_stream_sendv(client->output, iov, 2) < 0)
		return -1;
	client->last_output = ioloop_time;

	if (o_stream_get_buffer_used_size(client->output) >=
	    CLIENT_OUTPUT_OPTIMAL_SIZE) {
		/* buffer full, try flushing */
		return o_stream_flush(client->output);
	}
	return 1;
}

void client_send_response
(struct client *client, const char *oknobye, const char *resp_code, const char *msg)
{
	string_t *str;
	
	str = t_str_new(128);
	str_append(str, oknobye);

	if ( resp_code != NULL ) {
		str_append(str, " (");
		str_append(str, resp_code);
		str_append_c(str, ')');
	}

	if ( msg != NULL ) {
		str_append_c(str, ' ');
		managesieve_quote_append_string(str, msg, TRUE);
	}

	client_send_line(client, str_c(str));
}

void client_send_command_error
(struct client_command_context *cmd, const char *msg)
{
	struct client *client = cmd->client;
	const char *error, *cmd_name;
	bool fatal;

	if (msg == NULL) {
		msg = managesieve_parser_get_error(client->parser, &fatal);
		if (fatal) {
			client_disconnect_with_error(client, msg);
			return;
		}
	}

	if (cmd->name == NULL)
		error = t_strconcat
			("Error in MANAGESIEVE command: ", msg, NULL);
	else {
		cmd_name = t_str_ucase(cmd->name);
		error = t_strconcat
			("Error in MANAGESIEVE command ", cmd_name, ": ", msg, NULL);
	}

	client_send_no(client, error);

	if (++client->bad_counter >= CLIENT_MAX_BAD_COMMANDS) {
		client_disconnect_with_error(client,
			"Too many invalid MANAGESIEVE commands.");
	}

	/* client_read_args() failures rely on this being set, so that the
	   command processing is stopped even while command function returns
	   FALSE. */
	cmd->param_error = TRUE;
}

void client_send_storage_error(struct client *client,
             struct sieve_storage *storage)
{
	enum sieve_storage_error error_code;
	const char *error;

	error = sieve_storage_get_last_error(storage, &error_code);

	switch ( error_code ) {
	case SIEVE_STORAGE_ERROR_TEMP:
		client_send_noresp(client, "TRYLATER", error);
		break;

	case SIEVE_STORAGE_ERROR_QUOTA:
	case SIEVE_STORAGE_ERROR_NOSPACE: /* Not sure if this is appropriate */
		client_send_noresp(client, "QUOTA", error);
		break;

	case SIEVE_STORAGE_ERROR_NOTFOUND:
		client_send_noresp(client, "NONEXISTENT", error);
		break;

	case SIEVE_STORAGE_ERROR_ACTIVE:
		client_send_noresp(client, "ACTIVE", error);
		break;

	case SIEVE_STORAGE_ERROR_EXISTS:
		client_send_noresp(client, "ALREADYEXISTS", error);
		break;

	case SIEVE_STORAGE_ERROR_IMPOSSIBLE:
	default:
		client_send_no(client, error);
		break;
	}
}

bool client_read_args(struct client_command_context *cmd, unsigned int count,
		      unsigned int flags, struct managesieve_arg **args)
{
	int ret;

	i_assert(count <= INT_MAX);

	ret = managesieve_parser_read_args(cmd->client->parser, count, flags, args);
	if (ret >= (int)count) {
		/* all parameters read successfully */
		return TRUE;
	} else if (ret == -2) {
		/* need more data */
		if (cmd->client->input->closed) {
            /* disconnected */
            cmd->param_error = TRUE;
        }
		return FALSE;
	} else {
		/* error, or missing arguments */
		client_send_command_error(cmd, ret < 0 ? NULL :
					  "Missing arguments");
		return FALSE;
	}
}

bool client_read_string_args(struct client_command_context *cmd,
			     unsigned int count, ...)
{
	struct managesieve_arg *managesieve_args;
	va_list va;
	const char *str;
	unsigned int i;
	bool result = TRUE;

	if (!client_read_args(cmd, count, 0, &managesieve_args))
		return FALSE;

	va_start(va, count);
	for (i = 0; i < count; i++) {
		const char **ret = va_arg(va, const char **);

		if (managesieve_args[i].type == MANAGESIEVE_ARG_EOL) {
			client_send_command_error(cmd, "Missing arguments.");
			result = FALSE;
			break;
		}

		str = managesieve_arg_string(&managesieve_args[i]);
		if (str == NULL) {
			client_send_command_error(cmd, "Invalid arguments.");
			result = FALSE;
			break;
		}

		if (ret != NULL)
			*ret = str;
	}
	va_end(va);

	if (result && managesieve_args[i].type != MANAGESIEVE_ARG_EOL) {
		client_send_command_error(cmd, "Too many arguments.");
		result = FALSE;
	}

	return result;
}

void _client_reset_command(struct client *client)
{
	pool_t pool;
	size_t size;

	/* reset input idle time because command output might have taken a
	   long time and we don't want to disconnect client immediately then */
	client->last_input = ioloop_time;
	timeout_reset(client->to_idle);

	client->command_pending = FALSE;
    if (client->io == NULL && !client->disconnected) {
        i_assert(i_stream_get_fd(client->input) >= 0);
        client->io = io_add(i_stream_get_fd(client->input),
                    IO_READ, client_input, client);
    }
    o_stream_set_flush_callback(client->output, client_output, client);

	pool = client->cmd.pool;
	memset(&client->cmd, 0, sizeof(client->cmd));

	p_clear(pool);
	client->cmd.pool = pool;
	client->cmd.client = client;

	managesieve_parser_reset(client->parser);

	/* if there's unread data in buffer, remember that there's input
	   pending and we should get around to calling client_input() soon.
	   This is mostly for APPEND/IDLE. */
	(void)i_stream_get_data(client->input, &size);
	if (size > 0 && !client->destroyed)
		client->input_pending = TRUE;
}

/* Skip incoming data until newline is found,
   returns TRUE if newline was found. */
static bool client_skip_line(struct client *client)
{
	const unsigned char *data;
	size_t i, data_size;

	data = i_stream_get_data(client->input, &data_size);

	for (i = 0; i < data_size; i++) {
		if (data[i] == '\n') {
			client->input_skip_line = FALSE;
			i++;
			break;
		}
	}

	i_stream_skip(client->input, i);
	return !client->input_skip_line;
}

static bool client_handle_input(struct client_command_context *cmd)
{
	struct client *client = cmd->client;

	if (cmd->func != NULL) {
		/* command is being executed - continue it */
		if (cmd->func(cmd) || cmd->param_error) {
			/* command execution was finished */
			if (!cmd->param_error)
				client->bad_counter = 0;
			_client_reset_command(client);
			return TRUE;
		}

		/* unfinished */
        if (client->command_pending)
            o_stream_set_flush_pending(client->output, TRUE);
		return FALSE;
	}

	if (client->input_skip_line) {
		/* we're just waiting for new line.. */
		if (!client_skip_line(client))
			return FALSE;

		/* got the newline */
		_client_reset_command(client);

		/* pass through to parse next command */
	}

	if (cmd->name == NULL) {
		cmd->name = managesieve_parser_read_word(client->parser);
		if (cmd->name == NULL)
			return FALSE; /* need more data */
		cmd->name = p_strdup(cmd->pool, cmd->name);
	}

	if (cmd->name == '\0') {
        /* command not given - cmd_func is already NULL. */
    } else {
        /* find the command function */
        struct command *command = command_find(cmd->name);

        if (command != NULL) {
            cmd->func = command->func;
        }
    }

	client->input_skip_line = TRUE;
	if (cmd->func == NULL) {
		/* unknown command */
		client_send_command_error(cmd, "Unknown command.");
		_client_reset_command(client);
	} else {
		i_assert(!client->disconnected);

		client_handle_input(cmd);
	}

	return TRUE;
}

void client_input(void *context)
{
	struct client *client = context;
	struct client_command_context *cmd = &client->cmd;
	int ret;

	if (client->command_pending) {
		/* already processing one command. wait. */
		io_remove(&client->io);
		return;
	}

	client->input_pending = FALSE;
	client->last_input = ioloop_time;
	timeout_reset(client->to_idle);

	switch (i_stream_read(client->input)) {
	case -1:
		/* disconnected */
		client_destroy(client, NULL);
		return;
	case -2:
		/* parameter word is longer than max. input buffer size.
		   this is most likely an error, so skip the new data
		   until newline is found. */
		client->input_skip_line = TRUE;

		client_send_command_error(cmd, "Too long argument.");
		_client_reset_command(client);
		break;
	}

	client->handling_input = TRUE;
	o_stream_cork(client->output);
	do {
		T_BEGIN {
			ret = client_handle_input(cmd);
		} T_END;
	} while (ret && !client->disconnected);
    o_stream_uncork(client->output);
    client->handling_input = FALSE;

	if (client->command_pending)
		client->input_pending = TRUE;

	if (client->output->closed)
		client_destroy(client, NULL);
}

int client_output(void *context)
{
	struct client *client = context;
	struct client_command_context *cmd = &client->cmd;
	int ret;
	bool finished;

	client->last_output = ioloop_time;
    timeout_reset(client->to_idle);
    if (client->to_idle_output != NULL)
        timeout_reset(client->to_idle_output);

	if ((ret = o_stream_flush(client->output)) < 0) {
		client_destroy(client, NULL);
		return 1;
	}

	if (!client->command_pending)
		return 1;

	/* continue processing command */
	o_stream_cork(client->output);
	client->output_pending = TRUE;
	finished = cmd->func(cmd) || cmd->param_error;

	/* a bit kludgy check. normally we would want to get back to this
	   output handler, but IDLE is a special case which has command
	   pending but without necessarily anything to write. */
	if (!finished && client->output_pending)
		o_stream_set_flush_pending(client->output, TRUE);

	o_stream_uncork(client->output);

	if (finished) {
		/* command execution was finished */
		client->bad_counter = 0;
		_client_reset_command(client);

		if (client->input_pending)
			client_input(client);
	}
	return ret;
}

void clients_init(void)
{
	my_client = NULL;
}

void clients_deinit(void)
{
	if (my_client != NULL) {
		client_send_bye(my_client, "Server shutting down.");
		client_destroy(my_client, "Server shutting down");
	}
}

