/*
 * Please read Documentation/kprobes.txt for information what can be done with
 * kprobes and how to do it.
 */
#include <linux/kprobes.h>
#include <linux/module.h>
#include <linux/kallsyms.h>

#include <linux/buffer_head.h>
#include <linux/fs.h>

static struct jprobe my_jprobe;

/* My jprobe handler */
static void j_end_buffer_write_sync(struct buffer_head *bh, int uptodate)
{
	if (!uptodate || buffer_ordered(bh)) {
		char b[BDEVNAME_SIZE];

		printk(KERN_ERR "DEBUG: Returned buffer dev = %s, block = %lu, state = %lx\n", bdevname(bh->b_bdev, b), (unsigned long)bh->b_blocknr, (unsigned long)bh->b_state);
	}

	jprobe_return();
}

/* Create and insert jprobe to given function */
static int add_jprobe(struct jprobe *probe, char *name, kprobe_opcode_t *handler)
{
	int ret;

	probe->entry = handler;
	probe->kp.addr = (kprobe_opcode_t *) kallsyms_lookup_name(name);
	if (!my_jprobe.kp.addr) {
		printk(KERN_ERR "Couldn't find symbol %s to add jprobe.\n", name);
		return -1;
	}
	ret = register_jprobe(&my_jprobe);
	if (ret < 0) {
		printk(KERN_ERR "Couldn't register jprobe. Error %d.\n", ret);
		return -1;
	}
	return 0;
}

static int __init init_kprobe_module(void)
{
	int ret;

	ret = add_jprobe(&my_jprobe, "end_buffer_write_sync",
		(kprobe_opcode_t *)j_end_buffer_write_sync);
	if (!ret) {
		printk(KERN_INFO "Debugging module registered.\n");
		return 0;
	}
	return -1;
}

static void __exit cleanup_kprobe_module(void)
{
	unregister_jprobe(&my_jprobe);
	printk(KERN_INFO "Debugging module unregistered.\n");
}

MODULE_DESCRIPTION("Module with kprobe debugging");
MODULE_LICENSE("GPL");
module_init(init_kprobe_module);
module_exit(cleanup_kprobe_module);
