/*
 * Photo Image Print System
 * Copyright (C) 2002-2005 EPSON AVASYS Corporation.
 * Copyright (C) SEIKO EPSON CORPORATION 2002-2005.
 *
 *  This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, EPSON AVASYS Corporation gives permission to
 * link the code of this program with libraries which are covered by
 * the EPSON AVASYS Public License and distribute their linked
 * combinations.  You must obey the GNU General Public License in all
 * respects for all of the code used other than the libraries which
 * are covered by EPSON AVASYS Public License.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "files_parser.h"

extern	OPTION_DATA*	getOptData(int id, int value);

/* MediaTypeList class */

PVOID* mediatype_getnext(PVOID _item)
{
	MediaTypeNode* item = (MediaTypeNode*)_item;
	if (item == NULL)
	{
		return NULL;
	}
	return (PVOID*)&item->next;
}

int mediatype_compare(const PVOID* _item1, const PVOID* _item2)
{
	MediaTypeNode** item1 = (MediaTypeNode**)_item1;
	MediaTypeNode** item2 = (MediaTypeNode**)_item2;
	ASSERT(item1 && item2 && *item1 && *item2)
	if ((*item1)->media_type_id < (*item2)->media_type_id)
	{
		return -1;
	}
	else if ((*item1)->media_type_id == (*item2)->media_type_id)
	{
		return 0;
	}
	else
	{
		return 1;
	}
}

int mediatype_print(PVOID* _item, PVOID user_data)
{
#ifdef PPD_DEBUG
	MediaTypeNode** item = (MediaTypeNode**)_item;
	int i = 0, j = 0;
	ASSERT(item && item)
		printf("	Media Type Id:	%d\n", (*item)->media_type_id);
	for (i = 0; i < FUCHI_NUM; i++)
		for (j = 0; j < QLT_NUM; j++)
			printf("		PrintMode[%d][%d]: %d\n", i, j, (*item)->print_mode[i][j]);
#endif
	return 1;
}

PVOID mediatype_init(byte _media_type_id)
{
	MediaTypeNodeHandle handle = (MediaTypeNodeHandle)calloc(1, sizeof(MediaTypeNode));
	handle->media_type_id = _media_type_id;
	return (PVOID)handle;
}

void mediatype_done(PVOID* _handle_adr)
{
	MediaTypeNodeHandle* handle_adr = (MediaTypeNodeHandle*)_handle_adr;
	ASSERT(handle_adr && *handle_adr)
	free(*handle_adr);
	*handle_adr = NULL;
}


/* PrtOptList class */

PVOID* prtopt_getnext(PVOID _item)
{
	PrtOptNode* item = (PrtOptNode*)_item;
	if (item == NULL)
	{
		return NULL;
	}
	return (PVOID*)&item->next;
}

int prtopt_compare(const PVOID* _item1, const PVOID* _item2)
{
	PrtOptNode** item1 = (PrtOptNode**)_item1;
	PrtOptNode** item2 = (PrtOptNode**)_item2;
	ASSERT(item1 && item2 && *item1 && *item2)
	if ((*item1)->media_size_id < (*item2)->media_size_id)
	{
		return -1;
	}
	else if ((*item1)->media_size_id == (*item2)->media_size_id)
	{
		return 0;
	}
	else
	{
		return 1;
	}
}

int prtopt_print(PVOID* _item, PVOID user_data)
{
#ifdef PPD_DEBUG
	PrtOptNode** item = (PrtOptNode**)_item;
	ASSERT(item && *item)
	printf("Media Size Id:	%d\n", (*item)->media_size_id);
	list_for_each((*item)->media_type_list, mediatype_print, NULL);
#endif
	return 1;
}

PrtOptNodeHandle prtopt_init(byte _media_size_id, PaperNodeHandle _pnode)
{
	PrtOptNodeHandle handle = (PrtOptNodeHandle)calloc(1, sizeof(PrtOptNode));
	handle->media_size_id = _media_size_id;
	handle->pnode = _pnode;
	handle->media_type_list = list_init(NULL, NULL, 0, 
		mediatype_done, 
		mediatype_getnext, 
		mediatype_compare);
	handle->borderless_supported = false;
	return handle;
}

void prtopt_done(PVOID* _handle_adr)
{
	PrtOptNodeHandle* handle_adr = (PrtOptNodeHandle*)_handle_adr;
	ASSERT(handle_adr && *handle_adr)
	list_done(&(*handle_adr)->media_type_list);
	free(*handle_adr);
	*handle_adr = NULL;
}


/* Quality class */

PVOID* qlty_getnext(PVOID _item)
{
	QltyNode* item = (QltyNode*)_item;
	if (item == NULL)
	{
		return NULL;
	}
	return (PVOID*)&item->next;
}

int qlty_compare(const PVOID* _item1, const PVOID* _item2)
{
	QltyNode** item1 = (QltyNode**)_item1;
	QltyNode** item2 = (QltyNode**)_item2;
	ASSERT(item1 && item2 && *item1 && *item2)
	if ((*item1)->opdat_handle->value < (*item2)->opdat_handle->value)
	{
		return -1;
	}
	else if ((*item1)->opdat_handle->value == (*item2)->opdat_handle->value)
	{
		if ((*item1)->quality_id < (*item2)->quality_id)
			return -1;
		else if ((*item1)->quality_id == (*item2)->quality_id)
			return 0;
		else
			return 1;
	}
	else
	{
		return 1;
	}
}

int qlty_print(PVOID* _item, PVOID user_data)
{
#ifdef PPD_DEBUG
	QltyNode** item = (QltyNode**)_item;
	ASSERT(item && *item && (*item)->opdat_handle);
	printf("Quality Item [%d]:	%s	%d\n", (*item)->opdat_handle->value, (*item)->opdat_handle->rsc_name, (*item)->quality_id);
#endif
	return 1;
}

QltyNodeHandle qlty_init(OPTION_DATA *_opdat_handle, byte _quality_id)
{
	QltyNodeHandle handle = (QltyNodeHandle)calloc(1, sizeof(QltyNode));
	handle->opdat_handle = _opdat_handle;
	handle->quality_id = _quality_id;
	return handle;
}

void qlty_done(PVOID* _handle_adr)
{
	QltyNodeHandle* handle_adr = (QltyNodeHandle*)_handle_adr;
	ASSERT(handle_adr && *handle_adr)
	free(*handle_adr);
	*handle_adr = NULL;
}


/* PaperList class */

PVOID* paper_getnext(PVOID _item)
{
	PaperNode* item = (PaperNode*)_item;
	if (item == NULL)
	{
		return NULL;
	}
	return (PVOID*)&item->next;
}

int paper_compare(const PVOID* _item1, const PVOID* _item2)
{
	PaperNode** item1 = (PaperNode**)_item1;
	PaperNode** item2 = (PaperNode**)_item2;
	ASSERT(item1 && item2 && *item1 && *item2)
	if ((*item1)->id < (*item2)->id)
	{
		return -1;
	}
	else if ((*item1)->id == (*item2)->id)
	{
		return 0;
	}
	else
	{
		return 1;
	}
}

int paper_print(PVOID* _item, PVOID user_data)
{
#ifdef PPD_DEBUG
	PaperNode** item = (PaperNode**)_item;
	ASSERT(item && *item)
	printf("Paper: %s	%d	%s	%.0f	%.0f	%.0f	%.0f	%.0f	%.0f\n", 
		(*item)->code, (*item)->id, (*item)->name, (*item)->size.width, (*item)->size.height,
		(*item)->margin.left, (*item)->margin.top, (*item)->margin.right, (*item)->margin.bottom);
#endif
	return 1;
}

PaperNodeHandle paper_init(char* _code, byte _id, char* _name, 
	double _size_w, double _size_h, UNIT _size_u,
	double _margin_l, double _margin_t, double _margin_r, double _margin_b, UNIT _margin_u)
{
	PaperNodeHandle handle = (PaperNodeHandle)calloc(1, sizeof(PaperNode));
	strcpy(handle->code, _code);
	handle->id = _id;
	strcpy(handle->name, _name);
	handle->size.width = _size_w;
	handle->size.height = _size_h;
	strcpy(handle->size.unit, _size_u);
	handle->margin.left = _margin_l;
	handle->margin.top = _margin_t;
	handle->margin.right = _margin_r;
	handle->margin.bottom = _margin_b;
	strcpy(handle->margin.unit, _margin_u);
	return handle;
}

void paper_done(PVOID* _handle_adr)
{
	PaperNode** handle_adr = (PaperNode**)_handle_adr;
	ASSERT(handle_adr && *handle_adr)
		free(*handle_adr);
		*handle_adr = NULL;
}


/* File parsing functions */

typedef enum {
	LPR_NULL,
	LPR_COMMENT,
	LPR_PRINTER_ID,
	LPR_PAPERINFO_ENTRY,
	LPR_PRTOPT_ENTRY,
	LPR_OTHER
} LINE_PARSING_RSLT;

typedef enum {
	PAPER_CODE,
	PAPER_ID,
	PAPER_SIZE_WIDTH,
	PAPER_SIZE_HEIGHT,
	PAPER_MARGIN_LEFT,
	PAPER_MARGIN_TOP,	
	PAPER_MARGIN_RIGHT,
	PAPER_MARGIN_BOTTOM,
	PAPER_FIELD_NUM
} PAPER_FIELDS;

typedef enum {
	PRTOPT_MEDIA_SIZE,
	PRTOPT_MEDIA_TYPE,
	PRTOPT_QUALITY,
	PRTOPT_FUCHINASI,
	PRTOPT_FIELD_NUM
} PRTOPT_FIELDS;

bool	is_envelope(PrtOptNodeHandle po_node)
{
	bool result = false;
	if (po_node->pnode)
	{
		if (strstr(po_node->pnode->code, "ENV") != NULL)
			result = true;
	}
	return result;
}

LINE_PARSING_RSLT parse_line(const char* _line, int* argc, Str100* argv)
{
	int		line_parsing_rslt = LPR_NULL;
	char	*s_char_adr = NULL;
	char	*e_char_adr = NULL;
	char	*token;
	char	line[MAX_LINE];
	char	seps[]   = ",\n";

	strcpy(line, _line);
	s_char_adr = line;
	ltrim(rtrim(s_char_adr));

	if (s_char_adr[0] == '#')
	{
		line_parsing_rslt = LPR_COMMENT;
	}

	if (line_parsing_rslt == LPR_NULL)
	{
		e_char_adr = strchr(line, ']');
		if (s_char_adr[0] == '[' && e_char_adr != NULL)
		{
			*e_char_adr = '\0';
			strcpy(argv[0], &s_char_adr[1]);	
			*argc = 1;
			line_parsing_rslt = LPR_PRINTER_ID;
		}
	}

	if (line_parsing_rslt == LPR_NULL)
	{ 
		token = strtok(s_char_adr, seps);
		*argc = 0;
		while (token != NULL)
		{
			strcpy(argv[*argc], token);
			ltrim(rtrim(argv[*argc]));
			(*argc)++;
			token = strtok(NULL, seps);
		}
		if (*argc == PRTOPT_FIELD_NUM 
			&& isdigit(argv[PRTOPT_MEDIA_SIZE][0]) 
			&& isdigit(argv[PRTOPT_MEDIA_TYPE][0])
			&& isdigit(argv[PRTOPT_QUALITY][0]) 
			&& isdigit(argv[PRTOPT_FUCHINASI][0])
			&& atoi(argv[PRTOPT_QUALITY]) < QLT_NUM && atoi(argv[PRTOPT_FUCHINASI]) < FUCHI_NUM)
		{
			line_parsing_rslt = LPR_PRTOPT_ENTRY;
		}
		else if (*argc == PAPER_FIELD_NUM && isdigit(argv[PAPER_ID][0]) 
			&& isdigit(argv[PAPER_SIZE_WIDTH][0]) && isdigit(argv[PAPER_SIZE_HEIGHT][0]) 
			&& isdigit(argv[PAPER_MARGIN_LEFT][0]) && isdigit(argv[PAPER_MARGIN_TOP][0]) 
			&& isdigit(argv[PAPER_MARGIN_RIGHT][0]) && isdigit(argv[PAPER_MARGIN_BOTTOM][0]))
		{
			line_parsing_rslt = LPR_PAPERINFO_ENTRY;
		}
	}

	return line_parsing_rslt;
}

STATUS parse_paperlist_file(
	const char	*paperlist_path,
	ListHandle	paper_list
)
{
	STATUS	status = STS_SUCCESS;
	FILE	*stream;
	char	line[MAX_LINE];
	Str100	argv[10];
	int		argc = 0;
	int	paper_id = 0;
	OPTION_DATA*	paper_ref;

	if (paper_list->count != 0)
	{
		list_release(paper_list);
	}

	if ((stream = fopen(paperlist_path, "r")) == NULL)
	{
		status = STS_PAPERLIST_FILE_NOT_FOUND;
	}
	else
	{
		while (!feof(stream) && status == STS_SUCCESS)
		{
			if (fgets(line, MAX_LINE, stream) == NULL)
			{
				if (ferror(stream))
					status = STS_PAPERLIST_FILE_ERROR;
			}
			else
			{
				if (parse_line(line, &argc, argv) == LPR_PAPERINFO_ENTRY)
				{
					paper_id = atoi(argv[PAPER_ID]);
					paper_ref = getOptData(P_MEDIA_SIZE, paper_id);
					if (paper_ref != NULL)
					{
						list_add_item(paper_list, 
							paper_init(argv[PAPER_CODE], paper_id,
								paper_ref->x_name,
								atof(argv[PAPER_SIZE_WIDTH]), atof(argv[PAPER_SIZE_HEIGHT]), "dpi",
								atof(argv[PAPER_MARGIN_LEFT]), atof(argv[PAPER_MARGIN_TOP]), 
								atof(argv[PAPER_MARGIN_RIGHT]), atof(argv[PAPER_MARGIN_BOTTOM]), "dpi"));
					}
					else
					{
						status = STS_PAPERLIST_FILE_ERROR;
					}
				}
			}
		}

		fclose(stream);

		list_create_index(paper_list);
		list_for_each(paper_list, paper_print, NULL);
	}

	return status;
}

STATUS parse_prtopt_file(
	const char	*prtopt_path,
	const char	*model,
	const ListHandle paper_list,
	ListHandle	prtopt_list,
	ListHandle	qlty_list
)
{
	FILE	*stream;
	char	line[MAX_LINE];
	Str100	argv[10];
	int		argc = 0;
	LINE_PARSING_RSLT	lpr = LPR_NULL;
	STATUS	status = STS_SUCCESS;
	int		prev_msize_key = 0;			
	int		prev_mtype_key = 0;
	int		cur_msize_key = 0;
	int		cur_mtype_key = 0;
	MediaTypeNode*	mt_node = NULL;
	PrtOptNode*		po_node = NULL;
	int		qlty, fuchi;
	PaperNode	*key_pnode = NULL, **found_pnode_handle = NULL;
	OPTION_DATA	*found_opdat = NULL;
	QltyNode	*key_qnode = NULL;

	memset(argv, 0, sizeof(Str100)*sizeof(char));

	/* if prtopt is available, release it first */
	if (prtopt_list->count != 0)
	{
		list_release(prtopt_list);
	}

	if ((stream = fopen(prtopt_path, "r")) == NULL)
	{
		status = STS_PRTOPT_FILE_NOT_FOUND;
	}
	else
	{
		status = STS_PRTOPT_INFO_NOT_FOUND;
		while (!feof(stream) && status == STS_PRTOPT_INFO_NOT_FOUND)
		{ 
			if (fgets(line, MAX_LINE, stream) == NULL)
			{
				if (ferror(stream))
					status = STS_PRTOPT_FILE_ERROR;
			}
			else
			{
				if (parse_line(line, &argc, argv) == LPR_PRINTER_ID)
				{
					if (strcmp(model, argv[0]) == 0)
					{
						status = STS_SUCCESS;
					}
				}
			}
		}

		lpr = LPR_PRTOPT_ENTRY;
		while (!feof(stream) && status == STS_SUCCESS && (lpr == LPR_PRTOPT_ENTRY || lpr == LPR_NULL))
		{ 
			if (fgets(line, MAX_LINE, stream) == NULL)
			{
				if (ferror(stream))
					status = STS_PRTOPT_FILE_ERROR;
			}
			else
			{
				if ((lpr = parse_line(line, &argc, argv)) == LPR_PRTOPT_ENTRY)
				{
					cur_msize_key = atoi(argv[0]);
					cur_mtype_key = atoi(argv[1]);
					if (prev_msize_key != cur_msize_key || prtopt_list->count == 0)
					{ 
						key_pnode = paper_init("", cur_msize_key, "", 0, 0, "dpi", 0, 0, 0, 0, "dpi");
						if ((found_pnode_handle = (PaperNode**)list_find_item(paper_list, (PVOID*)&key_pnode)) != NULL)
						{
							/*  convert all unit to pts */
							convert_size_unit(&(*found_pnode_handle)->size, "pts");
							convert_margin_unit(&(*found_pnode_handle)->margin, "pts");
							po_node = prtopt_init(cur_msize_key, *found_pnode_handle);
							list_add_item(prtopt_list, po_node);
							prev_msize_key = cur_msize_key;
						}
						else
						{
							status = STS_PRTOPT_MEDIASIZE_NOT_RECOGNIZED;
						}
						paper_done((PVOID*)&key_pnode);
					}

					if (status == STS_SUCCESS && po_node && 
						(prev_msize_key != cur_msize_key || prtopt_list->count == 0 ||
							prev_mtype_key != cur_mtype_key || po_node->media_type_list->count == 0))
					{
						mt_node = mediatype_init(cur_mtype_key);
						list_add_item(
							((PrtOptNodeHandle)prtopt_list->last_item)->media_type_list,
							mt_node);
						prev_mtype_key = cur_mtype_key;
					}
					if (status == STS_SUCCESS && mt_node)
					{
						qlty = atoi(argv[2]);
							
						fuchi = atoi(argv[3]);
						mt_node->print_mode[FUCHI_ARI][qlty] = true;
						if (fuchi == 1)	
						{
							mt_node->print_mode[FUCHI_NASI][qlty] = true;
							if (is_envelope(po_node) == false)	
								po_node->borderless_supported = true;
						}
						/* fill quality list */
						if ((found_opdat = getOptData(P_MEDIA_TYPE, mt_node->media_type_id)) != NULL)
						{
							key_qnode = qlty_init(found_opdat, qlty);
							if (list_find_item(qlty_list, (PVOID*)&key_qnode) == NULL) 
								list_add_item(qlty_list, key_qnode);
							else
								qlty_done((PVOID*)&key_qnode);
						}
						else
						{
							status = STS_PRTOPT_MEDIATYPE_NOT_RECOGNIZED;
						}
					}
				}
			}
		}

		fclose(stream);
	}

	if (status == STS_SUCCESS)
	{
		list_for_each(prtopt_list, prtopt_print, NULL);
		list_for_each(qlty_list, qlty_print, NULL);
	}
	
	return status;
}
