/*
 * dvgrab.cc -- grabbing DV data and saving into an AVI file
 * Copyright (C) 2000 Arne Schirmacher <arne@schirmacher.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/** the dvgrab main program

    contains the main logic
*/

#include <assert.h> 
#include <fcntl.h> 
#include <unistd.h> 
#include <stdio.h> 
#include <stdlib.h> 
#include <signal.h> 
#include <pthread.h> 

#include "riff.h"
#include "avi.h"
#include "frame.h"

#define DVGRAB_VERSION "dvgrab 0.89\n"

#define RAW_BUF_SIZE    (10240)

enum { PAL_FORMAT, NTSC_FORMAT, AVI_DV1_FORMAT, AVI_DV2_FORMAT, QT_FORMAT, RAW_FORMAT, TEST_FORMAT, UNDEFINED };


bool    g_buffer_underrun;
bool    g_reader_active;
int     g_alldone;
int     g_filedone;
char    *g_dst_file_name;
int     g_frame_size;

int     g_autosplit;
int     g_timestamp;
int     g_card;
int     g_channel;
int     g_file_format;
int     g_frame_count;
int     g_frame_every;
int     g_frame_to_skip;
int     g_progress;
int     g_testmode;
AVIFile *g_avi = NULL;

deque < Frame* > g_buffer_queue;
deque < Frame* > g_output_queue;


pthread_mutex_t g_mutex;
pthread_t       g_thread;

void* read_frames(void* arg);

void usage()
{
    fprintf(stderr, "Usage: dvgrab [options] [file]\n");
    fprintf(stderr, "Try dvgrab --help for more information\n");
}


void getargs(int argc, char *argv[])
{
    int i;

    /* Set up all the defaults */

    g_autosplit = false;
    g_timestamp = false;
    g_card = 0;
    g_channel = 63;
    g_file_format = AVI_DV1_FORMAT;
    g_frame_count = 100000;
    g_frame_every = 1;
    g_dst_file_name = NULL;
    g_progress = false;
    g_testmode = false;

    for (i = 1; i < argc; ++i) {
        if (strcmp("--autosplit", argv[i]) == 0) {
            g_autosplit = true;
        } else if (strcmp("--timestamp", argv[i]) == 0) {
            g_timestamp = true;
        } else if ((strcmp("--format", argv[i]) == 0) && (i < argc)) {
            i++;
            if (strcmp("dv1", argv[i]) == 0) {
                g_file_format = AVI_DV1_FORMAT;
            } else if (strcmp("dv2", argv[i]) == 0) {
                g_file_format = AVI_DV2_FORMAT;
            } else if (strcmp("raw", argv[i]) == 0) {
                g_file_format = RAW_FORMAT;
            } else if (strcmp("qt", argv[i]) == 0) {
                g_file_format = QT_FORMAT;
            } else if (strcmp("test", argv[i]) == 0) {
                g_file_format = TEST_FORMAT;
            } else {
                usage();
                exit(1);
            }
        } else if ((strcmp("--frames", argv[i]) == 0) && (i < argc)) {
            i++;
            if (sscanf(argv[i], "%d", &g_frame_count) != 1) {
                usage();
                exit(1);
            }
        } else if ((strcmp("--every", argv[i]) == 0) && (i < argc)) {
            i++;
            if (sscanf(argv[i], "%d", &g_frame_every) != 1) {
                usage();
                exit(1);
            }
        } else if ((strcmp("--card", argv[i]) == 0) && (i < argc)) {
            i++;
            if (sscanf(argv[i], "%d", &g_card) != 1) {
                usage();
                exit(1);
            }
        } else if ((strcmp("--channel", argv[i]) == 0) && (i < argc)) {
            i++;
            if (sscanf(argv[i], "%d", &g_channel) != 1) {
                usage();
                exit(1);
            }
        } else if (strcmp("--progress", argv[i]) == 0) {
            g_progress = true;
        } else if (strcmp("--testmode", argv[i]) == 0) {
            g_testmode = true;
        } else if (strcmp("--help", argv[i]) == 0) {
            printf("Usage: %s [options] file\n", argv[0]);
            printf("\nOptions:\n\n");
            printf("  --autosplit       start a new file when the frame count is exceeded, a new\n");
            printf("                      recording is detected, or file becomes too big.\n");
            printf("                      Note: will loop forever -- you need to press ctrl-c\n");
            printf("                      at the end.\n");
            printf("                      Default is autosplit when file is slightly less than\n");
            printf("                      1 GByte.\n");
            printf("  --timestamp       put the date and time of recording into the file name.\n");
            printf("  --format dv1      save as a 'Type 1' DV AVI file. %s\n", g_file_format == AVI_DV1_FORMAT ? "This is the default." : " ");
            printf("  --format dv2      save as a 'Type 2' DV AVI file. %s\n", g_file_format == AVI_DV2_FORMAT ? "This is the default." : " ");
            printf("  --format raw      save only the raw DV information.\n");
            //                        printf("  --format qt       save as a QuickTime movie.\n");
            printf("  --format test     save in a format suitable as test input for --testmode.\n");
            printf("                    See the documentation for a discussion of the file types.\n");
            printf("  --frames number   number of frames to capture (default %d).\n", g_frame_count);
            printf("  --every number    write every n'th frame only (default %d, all frames).\n", g_frame_every);
            printf("  --card number     card number (default %d).\n", g_card);
            printf("  --channel number  iso channel number for listening (default %d).\n", g_channel);
            //                        printf("  --progress        print a '.' for each received frame.\n");
            printf("  --testmode        read video data from the file 'testdata.raw'\n");
            printf("                      instead from camcorder.\n");
            printf("  --help            display this help and exit.\n");
            printf("  --version         display version information and exit.\n");
            exit(0);
        } else if (strcmp("--version", argv[i]) == 0) {
            printf(DVGRAB_VERSION);
            exit(0);
        } else if (g_dst_file_name == NULL) {
            g_dst_file_name = argv[i];
        } else {
            usage();
            exit(1);
        }
    }
    if (g_dst_file_name == NULL) {
        usage();
        exit(1);
    }
}


Frame* GetNextFrame()
{
    Frame       *frame = NULL;

    while (frame == NULL) {

        pthread_mutex_lock(&g_mutex);

        if (g_output_queue.size() > 0) {
            frame = g_output_queue[0];
            g_output_queue.pop_front();
            // printf("writer < out: buffer %d, output %d\n",g_buffer_queue.size(), g_output_queue.size());
            // fflush(stdout);
        }

        pthread_mutex_unlock(&g_mutex);

        if (frame == NULL)
            usleep (100000);
    }

    return frame;
}


void DoneWithFrame(Frame *frame)
{
    pthread_mutex_lock(&g_mutex);
    g_buffer_queue.push_back(frame);
    // printf("writer > buf: buffer %d, output %d\n",g_buffer_queue.size(), g_output_queue.size());
    // fflush(stdout);
    pthread_mutex_unlock(&g_mutex);
}


int capture_avi()
{
    int frames_written;
    int file_count = 0;
    int incomplete_frames;
    Frame *frame = NULL;

    g_alldone = false;
    g_buffer_underrun = false;

    for (int i = 0; i < 50; ++i) {
        fail_null(frame = new Frame);
        g_buffer_queue.push_back(frame);
    }
    pthread_mutex_init(&g_mutex, NULL);
    g_reader_active = true;
    pthread_create(&g_thread, NULL, read_frames, NULL);

    while (g_reader_active == true) {
        frame = GetNextFrame();
        if (frame->IsComplete()) {
            g_frame_size = frame->GetFrameSize();
            break;
        } else {
            DoneWithFrame(frame);
        }
    }

    if (g_reader_active == false)
        g_alldone = true;

    while (g_alldone == false) {

        char filename[256];

        file_count++;
        if (g_timestamp == true) {
            struct tm recDate;
            frame->GetRecordingDate(recDate);
            sprintf(filename, "%s%4.4d.%2.2d.%2.2d_%2.2d-%2.2d-%2.2d.avi", g_dst_file_name,
                    recDate.tm_year + 1900, recDate.tm_mon + 1, recDate.tm_mday,
                    recDate.tm_hour, recDate.tm_min, recDate.tm_sec);
        } else
            sprintf(filename, "%s%3.3d.avi", g_dst_file_name, file_count);

        switch (g_file_format) {

        case AVI_DV1_FORMAT:
            g_avi = new AVI1File(filename);
            break;

        case AVI_DV2_FORMAT:
            g_avi = new AVI2File(filename);
            break;
        }

        AudioInfo info;
        frame->GetAudioInfo(info);
        if (frame->IsPAL() == true)
            g_avi->Init(AVI_PAL, info.frequency);
        else
            g_avi->Init(AVI_NTSC, info.frequency);

        frames_written = 0;
        incomplete_frames = 0;
        g_filedone = false;
        g_frame_to_skip = 0;

        while (g_alldone == false && g_filedone == false) {

            /* report a buffer underrun.  */

            if (g_buffer_underrun == true) {

                TimeCode timeCode;
                struct tm recDate;

                frame->GetTimeCode(timeCode);
                frame->GetRecordingDate(recDate);
                printf("%s: buffer underrun near: timecode %2.2d:%2.2d:%2.2d.%2.2d date %4.4d.%2.2d.%2.2d %2.2d:%2.2d:%2.2d\n", 
                       filename, timeCode.hour, timeCode.min, timeCode.sec, timeCode.frame,
                       recDate.tm_year + 1900, recDate.tm_mon + 1, recDate.tm_mday,
                       recDate.tm_hour, recDate.tm_min, recDate.tm_sec);
                g_buffer_underrun = false;
            }

            /* report a broken frame. */

            if (frame->IsComplete() == false) {

                TimeCode timeCode;
                struct tm recDate;

                frame->GetTimeCode(timeCode);
                frame->GetRecordingDate(recDate);
                printf("%s: frame dropped: timecode %2.2d:%2.2d:%2.2d.%2.2d date %4.4d.%2.2d.%2.2d %2.2d:%2.2d:%2.2d\n", 
                       filename, timeCode.hour, timeCode.min, timeCode.sec, timeCode.frame,
                       recDate.tm_year + 1900, recDate.tm_mon + 1, recDate.tm_mday,
                       recDate.tm_hour, recDate.tm_min, recDate.tm_sec);
                incomplete_frames++;
            } else {
                if (g_frame_to_skip == 0) {
                    g_avi->WriteFrame(*frame);
                    g_frame_to_skip = g_frame_every;
                    frames_written++;
                }
                g_frame_to_skip--;
            }

            DoneWithFrame(frame);

            /* If we have any frames in our output queue, continue processing.
               If the queue is empty but the receiver thread is still running we expect the queue to fill */

            if ((g_output_queue.size() > 0) || (g_reader_active == true)) {

                frame = GetNextFrame();

                /* Check certain criterias whether we should close the current file and start a new one.
                   If the file size gets close to 1 GByte, start a new file. */

                if (g_avi->GetFileSize() > 0x3f000000) {
                    g_filedone = true;
                }

                /* If the user wants autosplit, start a new file if a new recording is detected. */

                if (g_autosplit == true && frame->IsNewRecording()) {
                    g_filedone = true;
                }

                /* If the user wants autosplit, start a new file if the frame count is exceeded. */

                if (g_autosplit == true && frames_written >= g_frame_count) {
                    g_filedone = true;
                }

                /* If the user does not want autosplit, but has set a maximum frame count, turn everything off
                   and quit grabbing when this count is reached. */

                if (g_autosplit == false && frames_written >= g_frame_count) {
                    g_filedone = true;
                    g_alldone = true;
                    g_reader_active = false;
                }
            }
        }

        g_avi->WriteRIFF();

        printf("%s: %d frames\n", filename, frames_written);
        if (incomplete_frames > 0)
            printf("%d incomplete frames detected in %s\n", incomplete_frames, filename);
        fflush(stdout);

        delete g_avi;
        g_avi = NULL;
    }

    return 0;
}


extern int capture_test();

extern int capture_raw();


void signal_handler(int sig)
{
    /* replace this signal handler with the default (which aborts) */

    signal(SIGINT, SIG_IGN);

    /* setting these variables will let us fall out of the main loop */

    g_reader_active = false;
    g_alldone = true;
    if (g_avi != NULL) {

        g_avi->WriteRIFF();
        delete g_avi;
        g_avi = NULL;
    }
    exit(0);
}


int main(int argc, char *argv[])
{
    try {
        getargs(argc, argv);

        signal(SIGINT, signal_handler);

        switch (g_file_format) {

        case TEST_FORMAT:
            capture_test();
            break;

        case RAW_FORMAT:
            capture_raw();
            break;

        case AVI_DV1_FORMAT:
        case AVI_DV2_FORMAT:
            capture_avi();
            break;

        default:
            assert(0);
            break;
        }
    } catch (char *s) {
        printf("Exception caught.\n%s\n", s);
    }

    catch (...) {
        printf("Unexpected exception caught.\n");
    }
    return 0;
}





