/*
 * ieee1394io.cc -- asynchronously grabbing DV data
 * Copyright (C) 2000 Arne Schirmacher <arne@schirmacher.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stl.h>
#include <assert.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <pthread.h>

#include <libraw1394/raw1394.h>
#include <libraw1394/csr.h>
#include "frame.h"

#define RAW_BUF_SIZE	(10240)

/* global variables shared with the main thread */

extern bool	g_reader_active;
extern bool	g_buffer_underrun;
extern int	g_card;
extern int	g_channel;
extern int	g_testmode;
extern pthread_mutex_t	g_mutex;
extern deque < Frame* > g_buffer_queue;
extern deque < Frame* > g_output_queue;

/* private global variables */

int	g_testdata;
Frame	*g_current_frame;
unsigned char *g_frame;

int avi_iso_handler(raw1394handle_t handle, int channel, size_t length,
                    quadlet_t *data)
{
        /* The PAL/NTSC DV data has the following format:

           - packets of 496 bytes length
           - packets of  16 bytes length.

           The long packets contain the actual video and audio
           contents that goes into the AVI file. The contents of the
           other packets and the meaning of the header is unknown. If
           you know something about it, please let me know.
           
           The actual DV data starts at quadlet 4 of the long packet,
           so we have 480 bytes of DV data per packet.  For PAL, each
           rame is made out of 300 packets and there are 25 frames
           per second.  That is 144000 bytes per frame and 3600000
           bytes per second.  For NTSC we have 250 packages per frame
           and 30 frames per second.  That is 120000 bytes per frame
           and 3600000 bytes per second too.

           We also attempt to detect incomplete frames. The idea is:
           If we hit the begin of frame indicator and this is not the
           very first packet for this frame, then at least one packed
           has been dropped by the driver. This does not guarantee
           that no incomplete frames are saved, because if we miss the
           frame header of the next frame, we cant tell whether the
           last one is incomplete.  */


        if (length > 16) {

                unsigned char *p = (unsigned char*) & data[3];
                int section_type = p[0] >> 5;           /* section type is in bits 5 - 7 */
                int dif_sequence = p[1] >> 4;           /* dif sequence number is in bits 4 - 7 */
                int dif_block = p[2];

                /* if we are at the beginning of a frame, we put the previous frame in our output_queue.
                   Then we try to get an unused frame_buffer from the buffer_queue for the current frame.
                   We must lock the queues because they are shared between this thread and the main thread. */

                if (section_type == 0 && dif_sequence == 0) {
                        pthread_mutex_lock(&g_mutex);
                        if (g_current_frame != NULL) {
                                g_output_queue.push_back(g_current_frame);
                                g_current_frame = NULL;
                                // printf("reader > out: buffer %d, output %d\n", g_buffer_queue.size(), g_output_queue.size());
                                // fflush(stdout);
                        }


                        if (g_buffer_queue.size() > 0) {
                                g_current_frame = g_buffer_queue.front();
                                g_current_frame->bytesInFrame = 0;
                                g_buffer_queue.pop_front();
                                // printf("reader < buf: buffer %d, output %d\n",g_buffer_queue.size(), g_output_queue.size());
                                // fflush(stdout);
                        }
                        else
                                g_buffer_underrun = true;
                        pthread_mutex_unlock(&g_mutex);
                }

                if (g_current_frame != NULL) {

                        switch (section_type) {
                        case 0:           // 1 Header block
                                memcpy(g_current_frame->data + dif_sequence * 150 * 80, p, length - 16);
                                break;

                        case 1:           // 2 Subcode blocks
                                memcpy(g_current_frame->data + dif_sequence * 150 * 80 + (1 + dif_block) * 80, p, length - 16);
                                break;

                        case 2:           // 3 VAUX blocks
                                memcpy(g_current_frame->data + dif_sequence * 150 * 80 + (3 + dif_block) * 80, p, length - 16);
                                break;

                        case 3:           // 9 Audio blocks interleaved with video
                                memcpy(g_current_frame->data + dif_sequence * 150 * 80 + (6 + dif_block * 16) * 80, p, length - 16);
                                break;

                        case 4:           // 135 Video blocks interleaved with audio
                                memcpy(g_current_frame->data + dif_sequence * 150 * 80 + (7 + (dif_block / 15) + dif_block) * 80, p, length - 16);
                                break;

                        default:           // we cant handle any other data
                                break;
                        }
                        g_current_frame->bytesInFrame += length - 16;
                }
        }

        return 0;
}


int raw_iso_handler(raw1394handle_t handle, int channel, size_t length,
                    quadlet_t *data)
{
        if (length < RAW_BUF_SIZE) {
                *(int*)g_frame = length;
                memcpy(g_frame + 4, data, length);
        }
        return 0;
}


int my_reset_handler(raw1394handle_t handle)
{
        static int i = 0;

        printf("reset %d\n", i++);
        if (i == 100)
                g_reader_active = false;
        return 0;
}


int my_tag_handler(raw1394handle_t handle, unsigned long tag, int errcode)
{
        static int i = 0;

        printf("tag %d, tag %lx, errcode %d\n", i++, tag, errcode);
        if (i == 100)
                g_reader_active = false;
        return 0;
}


raw1394handle_t open_1394_driver(int channel, iso_handler_t handler)
{
        int numcards;
        struct raw1394_portinfo g_pinf[16];
        iso_handler_t	g_oldhandler;
        raw1394handle_t handle;

        if (!(handle = raw1394_get_handle())) {
                perror("raw1394 - couldn't get handle");
                exit( -1);
        }

        if ((numcards = raw1394_get_port_info(handle, g_pinf, 16)) < 0) {
                perror("raw1394 - couldn't get card info");
                exit( -1);
        }

        if (raw1394_set_port(handle, g_card) < 0) {
                perror("raw1394 - couldn't set port");
                exit( -1);
        }

        g_oldhandler = raw1394_set_iso_handler(handle, g_channel, handler);
        /*      raw1394_set_tag_handler(handle, my_tag_handler);              */
        raw1394_set_bus_reset_handler(handle, my_reset_handler);

        /* Starting iso receive */

        if (raw1394_start_iso_rcv(handle, channel) < 0) {
                perror("raw1394 - couldn't start iso receive");
                exit( -1);
        }
        return handle;
}


void close_1394_driver(int channel, raw1394handle_t handle)
{
	raw1394_stop_iso_rcv(handle, channel);
        raw1394_destroy_handle(handle);
}


void open_testdata()
{
        g_testdata = open("testdata.raw", O_RDONLY);
        assert(g_testdata != -1);
}


void read_testdata()
{
        quadlet_t	data[RAW_BUF_SIZE];
        size_t	length;
        size_t	count;

        pthread_mutex_lock(&g_mutex);
        int size = g_buffer_queue.size();
        pthread_mutex_unlock(&g_mutex);

	if (size > 0) {
            count = read(g_testdata, &length, sizeof(length));
	    if (count == sizeof(length)) {
                count = read(g_testdata, data, length);
                if (count == length) {
                        avi_iso_handler(NULL, 0, length, data);
                } else
                        g_reader_active = false;
        } else
                g_reader_active = false;
	}
}


void close_testdata()
{
        close(g_testdata);
}


void* read_frames(void* arg)
{
        raw1394handle_t handle;

        g_reader_active = true;
        g_current_frame = NULL;
        if (g_testmode == false)
                handle = open_1394_driver(g_channel, avi_iso_handler);
        else
                open_testdata();

        while (g_reader_active == true) {

                /* Poll the 1394 interface */

                if (g_testmode == false)
                        raw1394_loop_iterate(handle);
                else
                        read_testdata();
        }
        if (g_testmode == false)
                close_1394_driver(g_channel, handle);
        else
                close_testdata();

        /* if we are still using a frame, put it back */

        if (g_current_frame != NULL) {
                pthread_mutex_lock(&g_mutex);
                g_buffer_queue.push_back(g_current_frame);
                pthread_mutex_unlock(&g_mutex);
        }
        return NULL;
}

extern char *g_dst_file_name;
extern int g_frame_count;
extern int	g_alldone;
extern int usage();
#define fail_null(p) real_fail_null(p, __FILE__, __LINE__)
extern "C"
{
        extern void real_fail_null(void *p, char *file, int line);
}


int capture_test()
{
        int	frames_read;
        int	length;
        int	dst_fd;
        raw1394handle_t handle;

        if ((dst_fd = open(g_dst_file_name, O_WRONLY | O_CREAT, 00644)) < 0) {
                perror("problem with output file");
                usage();
                exit(1);
        }

        fail_null(g_frame = (unsigned char*)malloc(RAW_BUF_SIZE));

        handle = open_1394_driver(g_channel, raw_iso_handler);

        frames_read = 0;
        while ((!g_alldone) && (frames_read < g_frame_count)) {
                raw1394_loop_iterate(handle);
                length = *(int*)g_frame;
                write(dst_fd, g_frame, length + 4);
                frames_read++;
        }
        close_1394_driver(g_channel, handle);
        free(g_frame);
        close(dst_fd);

        return 0;
}


/* capture in a format suitable for playdv (part of libdv) */
/* courtesy Alastair Mayer [alastair@ajwm.net] */

int capture_raw()
{
        int frames_read;
        int length;
        int dst_fd;
        raw1394handle_t handle;
        bool found_first_frame;
        int skipped = 0;
//        unsigned char *g_frame;

        char filename[256];
        sprintf(filename, "%s.dv", g_dst_file_name);
        if ((dst_fd = open(filename, O_WRONLY | O_CREAT, 00644)) < 0) {
                perror("problem with output file");
                usage();
                exit(1);
        }

        fail_null(g_frame = (unsigned char*)malloc(RAW_BUF_SIZE));

        handle = open_1394_driver(g_channel, raw_iso_handler);

        frames_read = 0;
        found_first_frame = false;

        while ((!g_alldone) && (frames_read < g_frame_count)) {
                raw1394_loop_iterate(handle);
                length = *(int*)g_frame;
                if (length == 496) {
                        if (!found_first_frame) {
                                if (g_frame[16] == 0x1f &&
                                                g_frame[17] == 0x07)
                                        found_first_frame = true;
                                else skipped++;
                        }
                        if (skipped > 500) {
                                printf("skipped too many without finding frame\n");
                                break;
                        }
                        if (found_first_frame) {
                                write(dst_fd, (g_frame + 16), length - 16);
                                if (g_frame[16] == 0x1f && g_frame[17] == 0x07)
                                        frames_read++;
                        }
                }
        }
        close_1394_driver(g_channel, handle);
        free(g_frame);
        close(dst_fd);

        return 0;
}

