/*
 * riff.cc library for RIFF file format i/o
 * Copyright (C) 2000 Arne Schirmacher <arne@schirmacher.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "riff.h"
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <exception>

void real_fail_neg(int i, char *file, int line)
{
        static char exc[512];

        if (i < 0) {
                sprintf(exc, "exception in %s, line %d: function returned %d", file, line, i);
                throw exc;
        }
}


void real_fail_null(void *p, char *file, int line)
{
        static char exc[512];

        if (p == NULL) {
                sprintf(exc, "exception in %s, line %d: function returned NULL", file, line);
                throw exc;
        }
}


FOURCC make_fourcc(char *s)
{
        return (s[3] << 24) | (s[2] << 16) | (s[1] << 8) | s[0];
}


RIFFDirEntry::RIFFDirEntry()
{}


RIFFDirEntry::RIFFDirEntry(FOURCC t, FOURCC n, int l, int o, int p): type(t), name(n), length(l), offset(o), parent(p), written(0)
{}


RIFFFile::RIFFFile()
{
        fd = -1;
}


RIFFFile::RIFFFile(char *s)
{
        fail_neg(fd = open(s, O_RDWR | O_NONBLOCK | O_CREAT, 00644));
}


RIFFFile::~RIFFFile()
{
        if (fd != -1)
                close(fd);
}


int RIFFFile::AddDirectoryEntry(FOURCC type, FOURCC name, int length, int list)
{
        /* Put all parameters in an RIFFDirEntry object. The offset is
           currently unknown. */

        RIFFDirEntry    entry(type, name, length, 0 /* offset */, list);

        /* If the new chunk is in a list, then get the offset and size
           of that list. The offset of this chunk is the end of the list
           (parent_offset + parent_length) plus the size of the chunk
           header. */

        if (list != RIFF_NO_PARENT) {
                RIFFDirEntry parent = GetDirectoryEntry(list);
                entry.offset = parent.offset + parent.length + RIFF_HEADERSIZE;
        }

        /* The list which this new chunk is a member of has now increased in
           size. Get that directory entry and bump up its length by the size
           of the chunk. Since that list may also be contained in another
           list, walk up to the top of the tree. */

        while (list != RIFF_NO_PARENT) {
                RIFFDirEntry parent = GetDirectoryEntry(list);
                parent.length += RIFF_HEADERSIZE + length;
                SetDirectoryEntry(list, parent);
                list = parent.parent;
        }

        directory.insert(directory.end(), entry);

        return directory.size() - 1;
}


void RIFFFile::SetDirectoryEntry(int i, FOURCC type, FOURCC name, int length, int offset, int list)
{
        RIFFDirEntry    entry(type, name, length, offset, list);

        assert(i >= 0 && i < (int)directory.size());

        directory[i] = entry;
}


void RIFFFile::SetDirectoryEntry(int i, RIFFDirEntry entry)
{
        assert(i >= 0 && i < (int)directory.size());

        entry.written = false;
        directory[i] = entry;
}


void RIFFFile::GetDirectoryEntry(int i, FOURCC &type, FOURCC &name, int &length, int &offset, int &list)
{
        RIFFDirEntry entry;

        assert(i >= 0 && i < (int)directory.size());

        entry = directory[i];
        type = entry.type;
        name = entry.name;
        length = entry.length;
        offset = entry.offset;
        list = entry.parent;
}


RIFFDirEntry RIFFFile::GetDirectoryEntry(int i)
{
        assert(i >= 0 && i < (int)directory.size());

        return directory[i];
}


int RIFFFile::GetFileSize(void)
{
        if (directory.size() > 0)
                return directory[0].length;
        else
                return 0;
}


void RIFFFile::PrintDirectoryEntry(int i)
{
        RIFFDirEntry    entry;
        RIFFDirEntry    parent;
        FOURCC  entry_name;
        FOURCC list_name;

        /* Get all attributes of the chunk object. If it is contained
           in a list, get the name of the list too (otherwise the name of
           the list is blank). If the chunk object doesnt have a name (only
           LISTs and RIFFs have a name), the name is blank. */

        entry = GetDirectoryEntry(i);
        if (entry.parent != RIFF_NO_PARENT) {
                parent = GetDirectoryEntry(entry.parent);
                list_name = parent.name;
        } else {
                list_name = make_fourcc("    ");
        }
        if (entry.name != 0) {
                entry_name = entry.name;
        } else {
                entry_name = make_fourcc("    ");
        }

        /* Print out the ascii representation of type and name, as well as
           length and file offset. */

        printf("type: %c%c%c%c name: %c%c%c%c length: 0x%8.8x offset: 0x%8.8x list: %c%c%c%c\n",
               (char)((entry.type >> 0) & 0x000000ff),
               (char)((entry.type >> 8) & 0x000000ff),
               (char)((entry.type >> 16) & 0x000000ff),
               (char)((entry.type >> 24) & 0x000000ff),
               (char)((entry_name >> 0) & 0x000000ff),
               (char)((entry_name >> 8) & 0x000000ff),
               (char)((entry_name >> 16) & 0x000000ff),
               (char)((entry_name >> 24) & 0x000000ff),
               entry.length, (int)entry.offset,
               (char)((list_name >> 0) & 0x000000ff),
               (char)((list_name >> 8) & 0x000000ff),
               (char)((list_name >> 16) & 0x000000ff),
               (char)((list_name >> 24) & 0x000000ff));

        PrintDirectoryEntryData(entry);
}


void RIFFFile::PrintDirectoryEntryData(RIFFDirEntry entry)
{}


void RIFFFile::PrintDirectory()
{
        int i;
        int count = directory.size();

        for (i = 0; i < count; ++i)
                PrintDirectoryEntry(i);
}


int RIFFFile::FindDirectoryEntry(FOURCC type)
{
        int i;
        int count = directory.size();

        for (i = 0; i < count; ++i)
                if (directory[i].type == type)
                        return i;

        return -1;
}


/* Read in one chunk and add it to the directory. If the chunk happens
   to be of type LIST, then call ParseList recursively for it. */

void RIFFFile::ParseChunk(int parent)
{
        FOURCC type;
        int length;
        int pos;

        /* Check whether it is a LIST. If so, let ParseList deal with it */

        read(fd, &type, sizeof(type));
        if (type == make_fourcc("LIST")) {
                fail_neg(lseek(fd, -sizeof(type), SEEK_CUR));
                ParseList(parent);
        }

        /* it is a normal chunk, create a new directory entry for it */

        else {
                fail_neg(read(fd, &length, sizeof(length)));
                if (length & 1) length++;
                fail_neg(pos = lseek(fd, 0, SEEK_CUR));
                AddDirectoryEntry(type, 0, length, parent);
                fail_neg(lseek(fd, length, SEEK_CUR));
        }
}


/* Read in all chunks that are contained in a list and add them to the
   directory. */

void RIFFFile::ParseList(int parent)
{
        FOURCC type;
        FOURCC name;
        int length;
        int pos;
        int list;
        int listEnd;

        /* Read in the chunk header (type and length). */

        fail_neg(read(fd, &type, sizeof(type)));
        fail_neg(read(fd, &length, sizeof(length)));
        if (length & 1) length++;

        /* The contents of the list starts here. Obtain its offset. The list
           name (4 bytes) is already part of the contents). */

        fail_neg(pos = lseek(fd, 0, SEEK_CUR));
        fail_neg(read(fd, &name, sizeof(name)));

        /* Add an entry for this list. */

        list = AddDirectoryEntry(type, name, sizeof(name), parent);

        /* Read in any chunks contained in this list. This list is the
           parent for all chunks it contains. */

        listEnd = pos + length;
        while (pos < listEnd) {
                ParseChunk(list);
                fail_neg(pos = lseek(fd, 0, SEEK_CUR));
        }
}


void RIFFFile::ParseRIFF(void)
{
        int container = AddDirectoryEntry(make_fourcc("FILE"), make_fourcc("FILE"), 0, RIFF_NO_PARENT);

        ParseList(container);
}


void RIFFFile::ReadChunk(int chunk_index, void *data)
{
        RIFFDirEntry    entry;

        entry = GetDirectoryEntry(chunk_index);
        fail_neg(lseek(fd, entry.offset, SEEK_SET));
        fail_neg(read(fd, data, entry.length));
}


void RIFFFile::WriteChunk(int chunk_index, const void *data)
{
        RIFFDirEntry    entry;

        entry = GetDirectoryEntry(chunk_index);
        fail_neg(lseek(fd, entry.offset - RIFF_HEADERSIZE, SEEK_SET));
        fail_neg(write(fd, &entry.type, sizeof(entry.type)));
        fail_neg(write(fd, &entry.length, sizeof(entry.length)));
        fail_neg(write(fd, data, entry.length));

        /* Remember that this entry already has been written. */

        directory[chunk_index].written = true;
}


void RIFFFile::WriteRIFF(void)
{
        int     i;
        RIFFDirEntry    entry;
        int     count = directory.size();

        /* Start at the second entry (RIFF), since the first entry (FILE) is
           needed only for internal purposes and is not written to the file. */

        for (i = 1; i < count; ++i) {

                /* Only deal with entries that havent been written */

                entry = GetDirectoryEntry(i);
                if (entry.written == false) {

                        /* A chunk entry consist of its type and length, a list entry
                           has an additional name. Look up the entry, seek to the start
                           of the header, which is at the offset of the data start minus
                           the header size and write out the items. */

                        fail_neg(lseek(fd, entry.offset - RIFF_HEADERSIZE, SEEK_SET));
                        fail_neg(write(fd, &entry.type, sizeof(entry.type)));
                        fail_neg(write(fd, &entry.length, sizeof(entry.length)));

                        /* If it has a name, it is a list. Write out the extra name
                           field. */

                        if (entry.name != 0) {
                                fail_neg(write(fd, &entry.name, sizeof(entry.name)));
                        }

                        /* Remember that this entry already has been written. */

                        directory[i].written = true;
                }

        }
}

