/*
 * avi.cc library for AVI file format i/o
 * Copyright (C) 2000 Arne Schirmacher <arne@schirmacher.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "riff.h"
#include "avi.h"
#include <fcntl.h>
#include <unistd.h>

#include <string>
#include <iostream>
#include <strstream>
#include <iomanip>

#define PADDING_SIZE (512)
#define IX00_INDEX_SIZE (4028)

static char g_zeroes[PADDING_SIZE];

AVIFile::AVIFile() : RIFFFile(), index_type( -1), current_ix00( -1), file_list( -1), riff_list( -1), hdrl_list( -1), avih_chunk( -1), movi_list( -1), indx_chunk( -1), ix00_chunk( -1), idx1_chunk( -1), junk_chunk( -1)
{}



AVIFile::AVIFile(const char *s) : RIFFFile(s), index_type( -1), current_ix00( -1), file_list( -1), riff_list( -1), hdrl_list( -1), avih_chunk( -1), movi_list( -1), indx_chunk( -1), ix00_chunk( -1), idx1_chunk( -1), junk_chunk( -1)
{}



AVIFile::~AVIFile()
{}

/** Initialize the AVI structure to its initial state, either for PAL or NTSC format
 
    Initialize the AVIFile attributes: mainHdr, indx, ix00, idx1
 
    \todo consolidate AVIFile::Init, AVI1File::Init, AVI2File::Init. They are somewhat redundant.
    \param format pass AVI_PAL or AVI_NTSC
    \param sampleFrequency the sample frequency of the audio content
 
*/

void AVIFile::Init(int format, int sampleFrequency)
{
    int i;

    assert((format == AVI_PAL) || (format == AVI_NTSC));

    switch (format) {
    case AVI_PAL:
        mainHdr.dwMicroSecPerFrame = 40000;
        mainHdr.dwSuggestedBufferSize = 144008;
        break;

    case AVI_NTSC:
        mainHdr.dwMicroSecPerFrame = 33366;
        mainHdr.dwSuggestedBufferSize = 120008;
        break;

    default:                 /* no default allowed */
        assert(0);
        break;
    }

    /* Initialize the avih chunk */

    mainHdr.dwMaxBytesPerSec = 3600000;
    mainHdr.dwPaddingGranularity = PADDING_SIZE;
    mainHdr.dwFlags = 2048;             // 2064;
    mainHdr.dwTotalFrames = 0;
    mainHdr.dwInitialFrames = 0;
    mainHdr.dwStreams = 1;
    mainHdr.dwWidth = 0;
    mainHdr.dwHeight = 0;
    mainHdr.dwReserved[0] = 0;
    mainHdr.dwReserved[1] = 0;
    mainHdr.dwReserved[2] = 0;
    mainHdr.dwReserved[3] = 0;

    /* Initialize the indx chunk */

    indx.wLongsPerEntry = 4;
    indx.bIndexSubType = 0;
    indx.bIndexType = AVI_INDEX_OF_INDEXES;
    indx.nEntriesInUse = 0;
    // indx.dwChunkId = make_fourcc("00__");
    indx.dwReserved[0] = 0;
    indx.dwReserved[1] = 0;
    indx.dwReserved[2] = 0;
    for (i = 0; i < 2014; ++i) {
        indx.aIndex[i].qwOffset = 0;
        indx.aIndex[i].dwSize = 0;
        indx.aIndex[i].dwDuration = 0;
    }

    /* The ix00 chunk will be added dynamically in avi_write_frame
              as needed */
}


/** Find position and size of a given frame in the file
 
    Depending on which index is available, search one of them to
    find position and frame size
 
    \todo the size parameter is redundant. All frames have the same size, which is also in the mainHdr.
    \todo all index related operations should be isolated 
    \todo offset should be 64 bit
    \bug this won't work with very long AVI files, because these have several indexes
    \param offset the file offset to the start of the frame
    \param size the size of the frame
    \param the number of the frame we wish to find
    \return 0 if the frame could be found, -1 otherwise
*/

int AVIFile::GetDVFrameInfo(int &offset, int &size, int frameNum)
{
    switch (index_type) {
    case 1:

	/* find relevant index in indx */

	int i;	

	for (i = 0; frameNum >= indx.aIndex[i].dwDuration; frameNum -= indx.aIndex[i].dwDuration)
		;

	if (i != current_ix00) {
	    lseek(fd, indx.aIndex[i].qwOffset + 8, SEEK_SET);
    	    read(fd, &ix00, indx.aIndex[i].dwSize - 8);
	    current_ix00 = i;
	}

        if (frameNum < ix00.nEntriesInUse) {
            offset = ix00.qwBaseOffset + ix00.aIndex[frameNum].dwOffset;
            size = ix00.aIndex[frameNum].dwSize;
            return 0;
        } else
            return -1;
        break;

    case 2:
        int index = -1;
        int frameNumIndex = 0;
        FOURCC chunkId = make_fourcc("00dc");
        for (int i = 0; i < idx1.nEntriesInUse; ++i) {
            if (idx1.aIndex[i].dwChunkId == chunkId) {
                if (frameNumIndex == frameNum) {
                    index = i;
                    break;
                }
                ++frameNumIndex;
            }
        }
        if (index != -1) {
            offset = idx1.aIndex[index].dwOffset + 8;
            size = idx1.aIndex[index].dwSize;
            return 0;
        } else
            return -1;
        break;
    }
    return -1;
}


/** Read in a frame
 
    \todo use 64 bit seek
    \todo we actually don't need the frame here, we could use just a void pointer
    \param frame a reference to the frame object that will receive the frame data
    \param frameNum the frame number to read
    \return 0 if the frame could be read, -1 otherwise
*/

int AVIFile::GetDVFrame(Frame &frame, int frameNum)
{
    int	offset;
    int	size;

    if (GetDVFrameInfo(offset, size, frameNum) != 0)
        return -1;
    lseek(fd, offset, SEEK_SET);
    read(fd, frame.data, size);

    return 0;
}


int AVIFile::GetTotalFrames() const
{
    return mainHdr.dwTotalFrames;
}



/** prints out a directory entry in text form

    Every subclass of RIFFFile is supposed to override this function
    and to implement it for the entry types it knows about. For all
    other entry types it should call its parent::PrintDirectoryData.

    \todo user 64 bit routines
    \param entry the entry to print
*/

void AVIFile::PrintDirectoryEntryData(const RIFFDirEntry &entry) const
{
    if (entry.type == make_fourcc("avih")) {

        int             i;
        MainAVIHeader   main_avi_header;

        lseek(fd, entry.offset, SEEK_SET);
        read(fd, &main_avi_header, sizeof(MainAVIHeader));

        cout << "    dwMicroSecPerFrame:    " << (int)main_avi_header.dwMicroSecPerFrame << endl
        << "    dwMaxBytesPerSec:      " << (int)main_avi_header.dwMaxBytesPerSec << endl
        << "    dwPaddingGranularity:  " << (int)main_avi_header.dwPaddingGranularity << endl
        << "    dwFlags:               " << (int)main_avi_header.dwFlags << endl
        << "    dwTotalFrames:         " << (int)main_avi_header.dwTotalFrames << endl
        << "    dwInitialFrames:       " << (int)main_avi_header.dwInitialFrames << endl
        << "    dwStreams:             " << (int)main_avi_header.dwStreams << endl
        << "    dwSuggestedBufferSize: " << (int)main_avi_header.dwSuggestedBufferSize << endl
        << "    dwWidth:               " << (int)main_avi_header.dwWidth << endl
        << "    dwHeight:              " << (int)main_avi_header.dwHeight << endl;
        for (i = 0; i < 4; ++i)
            cout << "    dwReserved[" << i << "]:        " << (int)main_avi_header.dwReserved[i] << endl;

    } else if (entry.type == make_fourcc("strh")) {

        AVIStreamHeader	avi_stream_header;

        lseek(fd, entry.offset, SEEK_SET);
        read(fd, &avi_stream_header, sizeof(AVIStreamHeader));

        cout << "    fccType:         '"
        << (char)((avi_stream_header.fccType >> 0) & 0x000000ff)
        << (char)((avi_stream_header.fccType >> 8) & 0x000000ff)
        << (char)((avi_stream_header.fccType >> 16) & 0x000000ff)
        << (char)((avi_stream_header.fccType >> 24) & 0x000000ff)
        << '\'' << endl
        <<      "    fccHandler:      '"
        << (char)((avi_stream_header.fccHandler >> 0) & 0x000000ff)
        << (char)((avi_stream_header.fccHandler >> 8) & 0x000000ff)
        << (char)((avi_stream_header.fccHandler >> 16) & 0x000000ff)
        << (char)((avi_stream_header.fccHandler >> 24) & 0x000000ff)
        << '\'' << endl
        << "    dwFlags:         " << (int)avi_stream_header.dwFlags << endl
        << "    wPriority:       " << (int)avi_stream_header.wPriority << endl
        << "    wLanguage:       " << (int)avi_stream_header.wLanguage << endl
        << "    dwInitialFrames: " << (int)avi_stream_header.dwInitialFrames << endl
        << "    dwScale:         " << (int)avi_stream_header.dwScale << endl
        << "    dwRate:          " << (int)avi_stream_header.dwRate << endl
        << "    dwLength:        " << (int)avi_stream_header.dwLength << endl
        << "    dwQuality:       " << (int)avi_stream_header.dwQuality << endl
        << "    dwSampleSize:    " << (int)avi_stream_header.dwSampleSize << endl;

    } else if (entry.type == make_fourcc("indx")) {

        int             i;
        AVISuperIndex   avi_super_index;

        lseek(fd, entry.offset, SEEK_SET);
        read(fd, &avi_super_index, sizeof(AVISuperIndex));

        cout << "    wLongsPerEntry: " <<  (int)avi_super_index.wLongsPerEntry << endl
        << "    bIndexSubType:  " <<  (int)avi_super_index.bIndexSubType << endl
        << "    bIndexType:     " <<  (int)avi_super_index.bIndexType << endl
        << "    nEntriesInUse:  " <<  (int)avi_super_index.nEntriesInUse << endl
        << "    dwChunkId:      '"
        << (char)((avi_super_index.dwChunkId >> 0) & 0x000000ff)
        << (char)((avi_super_index.dwChunkId >> 8) & 0x000000ff)
        << (char)((avi_super_index.dwChunkId >> 16) & 0x000000ff)
        << (char)((avi_super_index.dwChunkId >> 24) & 0x000000ff)
        << '\'' << endl
        << "    dwReserved[0]:  " <<  (int)avi_super_index.dwReserved[0] << endl
        << "    dwReserved[1]:  " <<  (int)avi_super_index.dwReserved[1] << endl
        << "    dwReserved[2]:  " <<  (int)avi_super_index.dwReserved[2] << endl;
        for (i = 0; i < avi_super_index.nEntriesInUse; ++i) {
            cout << ' ' << setw(4) << setfill(' ') << i 
            << ": qwOffset    : 0x" << setw(12) << setfill('0') << hex << avi_super_index.aIndex[i].qwOffset << endl
            << "       dwSize      : 0x" << setw(8) << avi_super_index.aIndex[i].dwSize << endl
            << "       dwDuration  : " << dec << avi_super_index.aIndex[i].dwDuration << endl;
        }
    }

    /* This is the Standard Index. It is an array of offsets and
       sizes relative to some start offset. */

    else if (entry.type == make_fourcc("ix00")) {

        int             i;
        AVIStdIndex     avi_std_index;

        lseek(fd, entry.offset, SEEK_SET);
        read(fd, &avi_std_index, sizeof(AVIStdIndex));

        cout << "    wLongsPerEntry: " << (int)avi_std_index.wLongsPerEntry << endl
        << "    bIndexSubType:  " << (int)avi_std_index.bIndexSubType << endl
        << "    bIndexType:     " << (int)avi_std_index.bIndexType << endl
        << "    nEntriesInUse:  " << (int)avi_std_index.nEntriesInUse << endl
        << "    dwChunkId:      '"
        << (char)((avi_std_index.dwChunkId >> 0) & 0x000000ff)
        << (char)((avi_std_index.dwChunkId >> 8) & 0x000000ff)
        << (char)((avi_std_index.dwChunkId >> 16) & 0x000000ff)
        << (char)((avi_std_index.dwChunkId >> 24) & 0x000000ff)
        << '\'' << endl
        << "    qwBaseOffset:   0x" << setw(12) << hex << avi_std_index.qwBaseOffset << endl
        << "    dwReserved:     " << dec << (int)avi_std_index.dwReserved << endl;
        for (i = 0; i < avi_std_index.nEntriesInUse; ++i) {
            cout << ' ' << setw(4) << setfill(' ') << i 
            << ": dwOffset    : 0x" << setw(8) << setfill('0') << hex << avi_std_index.aIndex[i].dwOffset
            << " (0x" << avi_std_index.qwBaseOffset + avi_std_index.aIndex[i].dwOffset << ')' << endl
            << "       dwSize      : 0x" << setw(8) << avi_std_index.aIndex[i].dwSize << dec << endl;
        }

    } else if (entry.type == make_fourcc("idx1")) {

        int i;
        int numEntries = entry.length / sizeof(int) / 4;
        int *idx1 = new int[numEntries * 4];

        lseek(fd, entry.offset, SEEK_SET);
        read(fd, idx1, entry.length);

        for (i = 0; i < numEntries; ++i) {

            cout << "    dwChunkId   : '"
            << (char)((idx1[i * 4 + 0] >> 0) & 0x000000ff)
            << (char)((idx1[i * 4 + 0] >> 8) & 0x000000ff)
            << (char)((idx1[i * 4 + 0] >> 16) & 0x000000ff)
            << (char)((idx1[i * 4 + 0] >> 24) & 0x000000ff)
            << '\'' << endl
            << "    dwType      : 0x" << setw(8) << hex << idx1[i * 4 + 1] << endl
            << "    dwOffset    : 0x" << setw(8) << idx1[i * 4 + 2] << endl
            << "    dwSize      : 0x" << setw(8) << idx1[i * 4 + 3] << dec << endl;
        }

        delete[] idx1;
    }
}


void AVIFile::ParseList(int parent)
{
    FOURCC type;
    FOURCC name;
    int length;
    int pos;
    int list;
    int listEnd;

    /* Read in the chunk header (type and length). */

    fail_neg(read(fd, &type, sizeof(type)));
    fail_neg(read(fd, &length, sizeof(length)));
    if (length & 1) length++;

    /* The contents of the list starts here. Obtain its offset. The list
       name (4 bytes) is already part of the contents). */

    fail_neg(pos = lseek(fd, 0, SEEK_CUR));
    fail_neg(read(fd, &name, sizeof(name)));

    if (name != make_fourcc("movi")) {

    /* Add an entry for this list. */

    list = AddDirectoryEntry(type, name, sizeof(name), parent);

    /* Read in any chunks contained in this list. This list is the
       parent for all chunks it contains. */

    listEnd = pos + length;
    while (pos < listEnd) {
        ParseChunk(list);
        fail_neg(pos = lseek(fd, 0, SEEK_CUR));
    }
    } else {
    /* Add an entry for this list. */

    list = AddDirectoryEntry(type, name, length, parent);

        fail_neg(pos = lseek(fd, length - 4, SEEK_CUR));
    }  
}

void AVIFile::ParseRIFF()
{
    RIFFFile::ParseRIFF();

    avih_chunk = FindDirectoryEntry(make_fourcc("avih"));
    if (avih_chunk != -1)
        ReadChunk(avih_chunk, (void*)&mainHdr);
}


void AVIFile::ReadIndex()
{
    indx_chunk = FindDirectoryEntry(make_fourcc("indx"));
    if (indx_chunk != -1) {
        ReadChunk(indx_chunk, (void*)&indx);
        index_type = 1;
    }
    idx1_chunk = FindDirectoryEntry(make_fourcc("idx1"));
    if (idx1_chunk != -1) {
        ReadChunk(idx1_chunk, (void*)&idx1);
        idx1.nEntriesInUse = GetDirectoryEntry(idx1_chunk).length / 16;
        index_type = 2;
    }
}


void AVIFile::FlushIX00Index()
{
    FOURCC type;
    FOURCC name;
    int length;
    int offset;
    int parent;
    int i;

    /* Write out the old index. When this function is
       entered for the first time, there is no index to
       write.  Note: this may be an expensive operation
       because of a time consuming seek to the former file
       position. */

    if (ix00_chunk != -1)
        WriteChunk(ix00_chunk, &ix00);

    /* make a new ix00 chunk. */

    ix00_chunk = AddDirectoryEntry(make_fourcc("ix00"), 0, sizeof(AVIStdIndex), movi_list);
    GetDirectoryEntry(ix00_chunk, type, name, length, offset, parent);

    /* fill out all required fields. The offsets in the
       array are relative to qwBaseOffset, so fill in the
       offset to the next free location in the file
       there. */

    ix00.wLongsPerEntry = 2;
    ix00.bIndexSubType = 0;
    ix00.bIndexType = AVI_INDEX_OF_CHUNKS;
    ix00.nEntriesInUse = 0;
    ix00.dwChunkId = indx.dwChunkId;
    ix00.qwBaseOffset = offset + length;
    ix00.dwReserved = 0;

    for (i = 0; i < IX00_INDEX_SIZE; ++i) {
        ix00.aIndex[i].dwOffset = 0;
        ix00.aIndex[i].dwSize = 0;
    }

    /* add a reference to this new index in our super
                     index. */

    i = indx.nEntriesInUse++;
    indx.aIndex[i].qwOffset = offset - 8;
    indx.aIndex[i].dwSize = length + 8;
    indx.aIndex[i].dwDuration = 0;
}


void AVIFile::UpdateIndex(int frame_chunk)
{
    FOURCC type;
    FOURCC name;
    int length;
    int offset;
    int parent;
    int i;

    /* update the appropiate entry in the super index. It reflects
       the number of frames in the referenced index. */

    i = indx.nEntriesInUse - 1;
    ++indx.aIndex[i].dwDuration;

    /* update the standard index. Calculate the file position of
       the new frame. */

    GetDirectoryEntry(frame_chunk, type, name, length, offset, parent);
    i = ix00.nEntriesInUse++;
    ix00.aIndex[i].dwOffset = offset - ix00.qwBaseOffset;
    ix00.aIndex[i].dwSize = length;
}


void AVIFile::WriteRIFF()
{}



void AVIFile::WriteFrame(const Frame &frame)
{}



AVI1File::AVI1File() : AVIFile(), strl1_list( -1), strh1_chunk( -1), strf1_chunk( -1), odml_list( -1), dmlh_chunk( -1)
{}



AVI1File::AVI1File(const char *s) : AVIFile(s), strl1_list( -1), strh1_chunk( -1), strf1_chunk( -1), odml_list( -1), dmlh_chunk( -1)
{}



AVI1File::~AVI1File()
{}

/* Initialize the AVI structure to its initial state, either for PAL
   or NTSC format */

void AVI1File::Init(int format, int sampleFrequency)
{
    int i;

    assert((format == AVI_PAL) || (format == AVI_NTSC));

    AVIFile::Init(format, sampleFrequency);

    switch (format) {
    case AVI_PAL:
        stream1Hdr.dwScale = 400000;
        stream1Hdr.dwSuggestedBufferSize = 144008;

        /* I do not know what these values mean, I copied them
           from an existing PAL AVI file */

        /* initialize the strf chunk */

        dvinfo.dwDVAAuxSrc = 0xd1e030d0;
        dvinfo.dwDVAAuxCtl = 0xffa0cf3f;
        dvinfo.dwDVAAuxSrc1 = 0xd1e03fd0;
        dvinfo.dwDVAAuxCtl1 = 0xffa0cf3f;
        dvinfo.dwDVVAuxSrc = 0xff20ffff;
        dvinfo.dwDVVAuxCtl = 0xfffdc83f;
        dvinfo.dwDVReserved[0] = 0;
        dvinfo.dwDVReserved[1] = 0;
        break;

    case AVI_NTSC:
        stream1Hdr.dwScale = 333666;
        stream1Hdr.dwSuggestedBufferSize = 120008;

        /* I do not know what these values mean, I copied them from an existing NTSC AVI file */

        /* initialize the strf chunk */

        dvinfo.dwDVAAuxSrc = 0xc0c000c0;
        dvinfo.dwDVAAuxCtl = 0xffa0cf3f;
        dvinfo.dwDVAAuxSrc1 = 0xc0c001c0;
        dvinfo.dwDVAAuxCtl1 = 0xffa0cf3f;
        dvinfo.dwDVVAuxSrc = 0xff80ffff;
        dvinfo.dwDVVAuxCtl = 0xfffcc83f;
        dvinfo.dwDVReserved[0] = 0;
        dvinfo.dwDVReserved[1] = 0;
        break;

    default:                 /* no default allowed */
        assert(0);
        break;
    }

    indx.dwChunkId = make_fourcc("00__");

    /* Initialize the strh chunk */

    stream1Hdr.fccType = make_fourcc("iavs");
    stream1Hdr.fccHandler = make_fourcc("dvsd");
    stream1Hdr.dwFlags = 0;
    stream1Hdr.wPriority = 0;
    stream1Hdr.wLanguage = 0;
    stream1Hdr.dwInitialFrames = 0;
    stream1Hdr.dwRate = 10000000;
    stream1Hdr.dwStart = 0;
    stream1Hdr.dwLength = 0;
    stream1Hdr.dwQuality = 0;
    stream1Hdr.dwSampleSize = 0;
    stream1Hdr.rcFrame.top = 0;
    stream1Hdr.rcFrame.bottom = 0;
    stream1Hdr.rcFrame.left = 0;
    stream1Hdr.rcFrame.right = 0;

    /* Initialize the dmlh chunk. I have no clue what this means
       though */

    for (i = 0; i < 62; ++i)
        (dmlh)[i] = 0;
    (dmlh)[0] = -1;            /* frame count + 1? */

    /* This is a simple directory structure setup. For details see the
       "OpenDML AVI File Format Extensions" document.
       
       An AVI file contains basically two types of objects, a
       "chunk" and a "list" object. The list object contains any
       number of chunks. Since a list is also a chunk, it is
       possible to create a hierarchical "list of lists"
       structure.

       Every AVI file starts with a "RIFF" object, which is a list
       of several other required objects. The actual DV data is
       contained in a "movi" list, each frame is in its own chunk.

       Old AVI files (pre OpenDML V. 1.02) contain only one RIFF
       chunk of less than 1 GByte size per file. The current
       format which allow for almost arbitrary sizes can contain
       several RIFF chunks of less than 1 GByte size. Old software
       however would only deal with the first RIFF chunk.

       Note that the first entry (g_file_list) isnt actually part
       of the AVI file. I use this (pseudo-) directory entry to
       keep track of the RIFF chunks and their positions in the
       AVI file. */

    /* Create the container directory entry */

    file_list = AddDirectoryEntry(make_fourcc("FILE"), make_fourcc("FILE"), 0, RIFF_NO_PARENT);

    /* Create a basic directory structure. Only chunks defined from here on will be written to the AVI file. */

    riff_list = AddDirectoryEntry(make_fourcc("RIFF"), make_fourcc("AVI "), RIFF_LISTSIZE, file_list);
    hdrl_list = AddDirectoryEntry(make_fourcc("LIST"), make_fourcc("hdrl"), RIFF_LISTSIZE, riff_list);
    avih_chunk = AddDirectoryEntry(make_fourcc("avih"), 0, sizeof(MainAVIHeader), hdrl_list);
    strl1_list = AddDirectoryEntry(make_fourcc("LIST"), make_fourcc("strl"), RIFF_LISTSIZE, hdrl_list);
    strh1_chunk = AddDirectoryEntry(make_fourcc("strh"), 0, sizeof(AVIStreamHeader), strl1_list);
    strf1_chunk = AddDirectoryEntry(make_fourcc("strf"), 0, sizeof(dvinfo), strl1_list);
    indx_chunk = AddDirectoryEntry(make_fourcc("indx"), 0, sizeof(AVISuperIndex), strl1_list);
    odml_list = AddDirectoryEntry(make_fourcc("LIST"), make_fourcc("odml"), RIFF_LISTSIZE, hdrl_list);
    dmlh_chunk = AddDirectoryEntry(make_fourcc("dmlh"), 0, 0x00f8, odml_list);
    junk_chunk = AddDirectoryEntry(make_fourcc("JUNK"), 0, 0x0014, riff_list);
    movi_list = AddDirectoryEntry(make_fourcc("LIST"), make_fourcc("movi"), RIFF_LISTSIZE, riff_list);

    /* The ix00 chunk will be added dynamically to the movi_list in avi_write_frame
              as needed */

    ix00_chunk = -1;
}

/* Write a DV video frame. This is somewhat complex... */

void AVI1File::WriteFrame(const Frame &frame)
{
    int num_frames;
    int frame_chunk;
    int junk_chunk;
    int num_blocks;

    /* Check if we need a new ix00 Standard Index. It has a
       capacity of IX00_INDEX_SIZE frames. Whenever we exceed that
       number, we need a new index. The new ix00 chunk is also
       part of the movi list. */

    if (((mainHdr.dwTotalFrames - 0) % IX00_INDEX_SIZE) == 0) {
        FlushIX00Index();
    }

    /* Write the DV frame data.

       Make a new 00__ chunk for the new frame, write out the
       frame, then add a JUNK chunk which is sized such that we
       end up on a 512 bytes boundary. */

    frame_chunk = AddDirectoryEntry(make_fourcc("00__"), 0, frame.GetFrameSize(), movi_list);
    WriteChunk(frame_chunk, frame.data);
    num_blocks = frame.GetFrameSize() / PADDING_SIZE + 1;
    junk_chunk = AddDirectoryEntry(make_fourcc("JUNK"), 0, num_blocks * PADDING_SIZE - frame.GetFrameSize() - 2 * RIFF_HEADERSIZE, movi_list);
    WriteChunk(junk_chunk, g_zeroes);

    /* update some variables with the new frame count. */

    num_frames = ++mainHdr.dwTotalFrames;
    stream1Hdr.dwLength = num_frames;
    (dmlh)[0] = num_frames;

    UpdateIndex(frame_chunk);

    /* Find out if the current riff list is close to 1 GByte in
       size. If so, start a new (extended) RIFF. The only allowed
       item in the new RIFF chunk is a movi list (with video
       frames and indexes as usual). */

    //        GetDirectoryEntry(riff_list, type, name, length, offset, parent);
    //        if (length > 0x3f000000) {
    //                riff_list = AddDirectoryEntry(make_fourcc("RIFF"), make_fourcc("AVIX"), RIFF_LISTSIZE, file_list);
    //                movi_list = AddDirectoryEntry(make_fourcc("LIST"), make_fourcc("movi"), RIFF_LISTSIZE, riff_list);
    //        }
}



void AVI1File::WriteRIFF()
{
    WriteChunk(avih_chunk, (void*)&mainHdr);
    WriteChunk(strh1_chunk, (void*)&stream1Hdr);
    WriteChunk(strf1_chunk, (void*)&dvinfo);
    WriteChunk(indx_chunk, (void*)&indx);
    WriteChunk(dmlh_chunk, (void*)&dmlh);
    WriteChunk(ix00_chunk, (void*)&ix00);

    RIFFFile::WriteRIFF();
}


AVI2File::AVI2File() : AVIFile(), strl1_list( -1), strh1_chunk( -1), strf1_chunk( -1), strl2_list( -1), strh2_chunk( -1), strf2_chunk( -1)
{}



AVI2File::AVI2File(const char *s) : AVIFile(s), strl1_list( -1), strh1_chunk( -1), strf1_chunk( -1), strl2_list( -1), strh2_chunk( -1), strf2_chunk( -1)
{}



AVI2File::~AVI2File()
{}

/* Initialize the AVI structure to its initial state, either for PAL
   or NTSC format */

void AVI2File::Init(int format, int sampleFrequency)
{
    assert((format == AVI_PAL) || (format == AVI_NTSC));

    switch (format) {

    case AVI_PAL:
        mainHdr.dwMicroSecPerFrame = 40000;
        mainHdr.dwMaxBytesPerSec = 4147200;
        mainHdr.dwPaddingGranularity = 0;
        mainHdr.dwFlags = 16;            //2048;  // 2064;
        mainHdr.dwTotalFrames = 0;
        mainHdr.dwInitialFrames = 0;
        mainHdr.dwStreams = 2;
        mainHdr.dwSuggestedBufferSize = 0;
        mainHdr.dwWidth = 720;
        mainHdr.dwHeight = 576;
        mainHdr.dwReserved[0] = 40000;
        mainHdr.dwReserved[1] = 1000000;
        mainHdr.dwReserved[2] = 0;
        mainHdr.dwReserved[3] = 0;

        /* Initialize the strh chunk */

        stream1Hdr.fccType = make_fourcc("vids");
        stream1Hdr.fccHandler = make_fourcc("dvsd");
        stream1Hdr.dwFlags = 0;
        stream1Hdr.wPriority = 0;
        stream1Hdr.wLanguage = 0;
        stream1Hdr.dwInitialFrames = 0;
        stream1Hdr.dwScale = 40000;
        stream1Hdr.dwRate = 1000000;
        stream1Hdr.dwStart = 0;
        stream1Hdr.dwLength = 0;
        stream1Hdr.dwSuggestedBufferSize = 0;
        stream1Hdr.dwQuality = -1;
        stream1Hdr.dwSampleSize = 0;
        stream1Hdr.rcFrame.top = 0;
        stream1Hdr.rcFrame.bottom = 0;
        stream1Hdr.rcFrame.left = 0;
        stream1Hdr.rcFrame.right = 0;

        bitmapinfo.dwSize = sizeof(bitmapinfo);
        bitmapinfo.biWidth = 720;
        bitmapinfo.biHeight = 576;
        bitmapinfo.biPlanes = 1;
        bitmapinfo.biBitCount = 24;
        bitmapinfo.biCompression = make_fourcc("dvsd");
        bitmapinfo.biSizeImage = bitmapinfo.biWidth * bitmapinfo.biHeight * 3;
        bitmapinfo.biXPelsPerMeter = 0;
        bitmapinfo.biYPelsPerMeter = 0;
        bitmapinfo.biClrUsed = 0;
        bitmapinfo.biClrImportant = 0;

        stream2Hdr.fccType = make_fourcc("auds");
        stream2Hdr.fccHandler = 0;
        stream2Hdr.dwFlags = 0;
        stream2Hdr.wPriority = 0;
        stream2Hdr.wLanguage = 0;
        stream2Hdr.dwInitialFrames = 0;
        stream2Hdr.dwScale = 1;
        stream2Hdr.dwRate = sampleFrequency;
        stream2Hdr.dwStart = 0;
        stream2Hdr.dwLength = 0;
        stream2Hdr.dwSuggestedBufferSize = 0;
        stream2Hdr.dwQuality = -1;
        stream2Hdr.dwSampleSize = 2;
        stream2Hdr.rcFrame.top = 0;
        stream2Hdr.rcFrame.bottom = 0;
        stream2Hdr.rcFrame.left = 0;
        stream2Hdr.rcFrame.right = 0;

        break;

    case AVI_NTSC:
        mainHdr.dwMicroSecPerFrame = 33366;
        mainHdr.dwMaxBytesPerSec = 4147200;
        mainHdr.dwPaddingGranularity = 0;
        mainHdr.dwFlags = 16;            //2048;  // 2064;
        mainHdr.dwTotalFrames = 0;
        mainHdr.dwInitialFrames = 0;
        mainHdr.dwStreams = 2;
        mainHdr.dwSuggestedBufferSize = 0;
        mainHdr.dwWidth = 720;
        mainHdr.dwHeight = 480;
        mainHdr.dwReserved[0] = 33366;
        mainHdr.dwReserved[1] = 1000000;
        mainHdr.dwReserved[2] = 0;
        mainHdr.dwReserved[3] = 0;

        /* Initialize the strh chunk */

        stream1Hdr.fccType = make_fourcc("vids");
        stream1Hdr.fccHandler = make_fourcc("dvsd");
        stream1Hdr.dwFlags = 0;
        stream1Hdr.wPriority = 0;
        stream1Hdr.wLanguage = 0;
        stream1Hdr.dwInitialFrames = 0;
        stream1Hdr.dwScale = 33366;
        stream1Hdr.dwRate = 1000000;
        stream1Hdr.dwStart = 0;
        stream1Hdr.dwLength = 0;
        stream2Hdr.dwSuggestedBufferSize = 0;
        stream1Hdr.dwQuality = -1;
        stream1Hdr.dwSampleSize = 0;
        stream1Hdr.rcFrame.top = 0;
        stream1Hdr.rcFrame.bottom = 0;
        stream1Hdr.rcFrame.left = 0;
        stream1Hdr.rcFrame.right = 0;

        bitmapinfo.dwSize = sizeof(bitmapinfo);
        bitmapinfo.biWidth = 720;
        bitmapinfo.biHeight = 480;
        bitmapinfo.biPlanes = 1;
        bitmapinfo.biBitCount = 24;
        bitmapinfo.biCompression = make_fourcc("dvsd");
        bitmapinfo.biSizeImage = bitmapinfo.biWidth * bitmapinfo.biHeight * 3;
        bitmapinfo.biXPelsPerMeter = 0;
        bitmapinfo.biYPelsPerMeter = 0;
        bitmapinfo.biClrUsed = 0;
        bitmapinfo.biClrImportant = 0;

        stream2Hdr.fccType = make_fourcc("auds");
        stream2Hdr.fccHandler = 0;
        stream2Hdr.dwFlags = 0;
        stream1Hdr.wPriority = 0;
        stream1Hdr.wLanguage = 0;
        stream2Hdr.dwInitialFrames = 0;
        stream2Hdr.dwScale = 1;
        stream2Hdr.dwRate = sampleFrequency;
        stream2Hdr.dwStart = 0;
        stream2Hdr.dwLength = 0;
        stream2Hdr.dwSuggestedBufferSize = 0;
        stream2Hdr.dwQuality = -1;
        stream2Hdr.dwSampleSize = 2;
        stream2Hdr.rcFrame.top = 0;
        stream2Hdr.rcFrame.bottom = 0;
        stream2Hdr.rcFrame.left = 0;
        stream2Hdr.rcFrame.right = 0;

        break;
    }
    waveformatex.wFormatTag = 1;
    waveformatex.nChannels = 2;
    waveformatex.nSamplesPerSec = sampleFrequency;
    waveformatex.nAvgBytesPerSec = sampleFrequency * 4;
    waveformatex.nBlockAlign = 4;
    waveformatex.wBitsPerSample = 16;

    for (int i = 0; i < 4000; ++i) {
        idx1.aIndex[i].dwChunkId = 0;
        idx1.aIndex[i].dwFlags = 0;
        idx1.aIndex[i].dwOffset = 0;
        idx1.aIndex[i].dwSize = 0;
    }
    idx1.nEntriesInUse = 0;

    //        for (int i = 0; i < 64000; ++i)
    //                idx1[i] = 0;
    //        idx1_entries = 0;

    file_list = AddDirectoryEntry(make_fourcc("FILE"), make_fourcc("FILE"), 0, RIFF_NO_PARENT);

    /* Create a basic directory structure. Only chunks defined from here on will be written to the AVI file. */

    riff_list = AddDirectoryEntry(make_fourcc("RIFF"), make_fourcc("AVI "), RIFF_LISTSIZE, file_list);
    hdrl_list = AddDirectoryEntry(make_fourcc("LIST"), make_fourcc("hdrl"), RIFF_LISTSIZE, riff_list);
    avih_chunk = AddDirectoryEntry(make_fourcc("avih"), 0, sizeof(MainAVIHeader), hdrl_list);
    strl1_list = AddDirectoryEntry(make_fourcc("LIST"), make_fourcc("strl"), RIFF_LISTSIZE, hdrl_list);
    strh1_chunk = AddDirectoryEntry(make_fourcc("strh"), 0, sizeof(AVIStreamHeader), strl1_list);
    strf1_chunk = AddDirectoryEntry(make_fourcc("strf"), 0, sizeof(BITMAPINFOHEADER), strl1_list);
    strl2_list = AddDirectoryEntry(make_fourcc("LIST"), make_fourcc("strl"), RIFF_LISTSIZE, hdrl_list);
    strh2_chunk = AddDirectoryEntry(make_fourcc("strh"), 0, sizeof(AVIStreamHeader), strl2_list);
    strf2_chunk = AddDirectoryEntry(make_fourcc("strf"), 0, sizeof(WAVEFORMATEX), strl2_list);
    movi_list = AddDirectoryEntry(make_fourcc("LIST"), make_fourcc("movi"), RIFF_LISTSIZE, riff_list);
}


void AVI2File::WriteRIFF()
{
    WriteChunk(avih_chunk, (void*)&mainHdr);
    WriteChunk(strh1_chunk, (void*)&stream1Hdr);
    WriteChunk(strf1_chunk, (void*)&bitmapinfo);
    WriteChunk(strh2_chunk, (void*)&stream2Hdr);
    WriteChunk(strf2_chunk, (void*)&waveformatex);
    int idx1_chunk = AddDirectoryEntry(make_fourcc("idx1"), 0, idx1.nEntriesInUse * 16, riff_list);
    WriteChunk(idx1_chunk, (void*)&idx1);

    RIFFFile::WriteRIFF();
}

/** Write a DV video frame
 
    \param frame the frame to write
*/

void AVI2File::WriteFrame(const Frame &frame)
{
    int         num_frames;
    int         audio_chunk;
    int         frame_chunk;
    char        soundbuf[20000];
    int		audio_size;

    audio_size = frame.ExtractAudio(soundbuf);

    /* Write audio data if we have it */

    if (audio_size > 0) {
        audio_chunk = AddDirectoryEntry(make_fourcc("01wb"), 0, audio_size, movi_list);
        stream2Hdr.dwLength += audio_size;
        WriteChunk(audio_chunk, soundbuf);
        idx1.aIndex[idx1.nEntriesInUse].dwChunkId = make_fourcc("01wb");
        idx1.aIndex[idx1.nEntriesInUse].dwFlags = 0;
        idx1.aIndex[idx1.nEntriesInUse].dwOffset = GetDirectoryEntry(audio_chunk).offset - RIFF_HEADERSIZE;
        idx1.aIndex[idx1.nEntriesInUse].dwSize = audio_size;
        idx1.nEntriesInUse++;
    }

    /* Write video data */

    frame_chunk = AddDirectoryEntry(make_fourcc("00dc"), 0, frame.GetFrameSize(), movi_list);
    WriteChunk(frame_chunk, frame.data);
    idx1.aIndex[idx1.nEntriesInUse].dwChunkId = make_fourcc("00dc");
    idx1.aIndex[idx1.nEntriesInUse].dwFlags = 0x10;
    idx1.aIndex[idx1.nEntriesInUse].dwOffset = GetDirectoryEntry(frame_chunk).offset - RIFF_HEADERSIZE;
    idx1.aIndex[idx1.nEntriesInUse].dwSize = frame.GetFrameSize();
    idx1.nEntriesInUse++;

    /* update some variables with the new frame count. */

    num_frames = ++mainHdr.dwTotalFrames;
    stream1Hdr.dwLength = num_frames;
    mainHdr.dwReserved[3] = num_frames;
}
