/* Doors for Linux.
   Copyright (C) 1998 Jason Lango

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/*
**  door.h - Public interface to the Doors API for Linux.
**  Jason Lango <jal@rampant.org>
*/

#ifndef __DOOR_H__
#define __DOOR_H__

/* Non-portable definitions begin with np_.  These definitions are
 * used internally by the doors library and in this header file to
 * simplify the function prototypes, however users of the library
 * should avoid using them.
 */

#include <sys/cdefs.h>

#ifndef __KERNEL__
#include <sys/types.h>
#endif

/*
 * Pointer types must be at least 64 bits long, to support multiple
 * ABI kernels, e.g. sparc64.
 */

typedef u_int32_t              door_attr_t;
typedef u_int64_t              door_id_t;
typedef u_int64_t              door_ptr_t;
typedef u_int32_t              np_door_uid_t;
typedef u_int32_t              np_door_gid_t;
typedef u_int32_t              np_door_pid_t;
typedef u_int32_t              np_door_size_t;


typedef struct {
	door_attr_t d_attributes;	/* describes the parameter */
	union {
		struct {
			int d_descriptor;	/* door descriptor */
			door_id_t d_id;		/* unique door id */
		} d_desc;
	} d_data;
} door_desc_t;

#define DOOR_DESCRIPTOR		0x1000

typedef struct {
	char		*data_ptr;	/* buf ptr		(in/out) */
	np_door_size_t	data_size;	/* buf size		(in/out) */
	door_desc_t	*desc_ptr;	/* descriptors		(in/out) */
	np_door_size_t	desc_num;	/* num desc		(in/out) */
	char		*rbuf;		/* result buffer	(in/out) */
	np_door_size_t	rsize;		/* result buffer size	(in/out) */
} door_arg_t;

typedef struct door_info {
	np_door_pid_t	di_target;	/* server pid */
	door_ptr_t	di_proc;	/* server procedure */
	door_ptr_t	di_data;	/* data cookie */
	door_attr_t	di_attributes;	/* door attributes */
	door_id_t	di_uniquifier;	/* unique door identifier */
} door_info_t;

#define DOOR_UNREF		0x0001
#define DOOR_PRIVATE		0x0002
#define DOOR_LOCAL		0x0004
#define DOOR_REVOKED		0x0008

#define NP_DOOR_ATTRS		(DOOR_UNREF|DOOR_PRIVATE|DOOR_LOCAL|\
				DOOR_REVOKED)

#ifdef __KERNEL__
#define DOOR_UATTR_MASK		0x00FF
#define DOOR_UNREF_PENDING	0x0100
#define DOOR_UNREF_DELIVERED	0x0200
#endif /* __KERNEL__ */

/* When the number of descriptors referring to a door with the
 * DOOR_UNREF attribute drops to 1, the door's server_proc is called
 * with argp set to DOOR_UNREF_DATA (and arg_size, dp, and n_desc all
 * zero).
 */
#define DOOR_UNREF_DATA		((void*)-1)

typedef struct door_cred {
	np_door_uid_t	dc_euid;	/* client effective uid */
	np_door_gid_t	dc_egid;	/* client effective gid */
	np_door_uid_t	dc_ruid;	/* client real uid */
	np_door_gid_t	dc_rgid;	/* client real gid */
	np_door_pid_t	dc_pid;		/* client pid */
} door_cred_t;

#ifndef __KERNEL__

__BEGIN_DECLS

typedef void (*np_door_server_proc_t)(void *cookie, char *argp,
	size_t arg_size, door_desc_t *dp, size_t n_desc);
typedef void (*np_door_create_proc_t)(door_info_t *info);

int __door_call __P((int d, door_arg_t *params));
int door_call __P((int d, door_arg_t *params));

int __door_create __P((np_door_server_proc_t server_procedure, void *cookie,
	u_int attributes));
int door_create __P((np_door_server_proc_t server_procedure, void *cookie,
	u_int attributes));

int __door_return __P((void *data_ptr, size_t data_size, door_desc_t *desc_ptr,
	size_t num_desc));
int door_return __P((void *data_ptr, size_t data_size, door_desc_t *desc_ptr,
	size_t num_desc));

int __door_cred __P((door_cred_t *info));
int door_cred __P((door_cred_t *info));

int __np_door_fattach __P((int d, const char *name));
int np_door_fattach __P((int d, const char *name));

int __np_door_fdetach __P((const char *name));
int np_door_fdetach __P((const char *name));

int __door_info __P((int d, struct door_info *info));
int door_info __P((int d, struct door_info *info));

np_door_create_proc_t __door_server_create __P((np_door_create_proc_t));
np_door_create_proc_t door_server_create __P((np_door_create_proc_t));

#define __fattach __np_door_fattach
#define fattach np_door_fattach

#define __fdetach __np_door_fdetach
#define fdetach np_door_fdetach

__END_DECLS

#endif /* !__KERNEL__ */

#endif /* __DOOR_H__ */
